/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_SEQUENCE_WALKER_TASK_H_
#define _GB2_SEQUENCE_WALKER_TASK_H_

#include <core_api/Task.h>
#include <core_api/LRegion.h>

namespace GB2 {

class DNATranslation;
class SequenceWalkerSubtask;

enum StrandOption {
    StrandOption_DirectOnly,
    StrandOption_ComplementOnly,
    StrandOption_Both
};
class GB2_COREAPI_EXPORT SequenceWalkerConfig {
public:
    
    //TODO: allow select custom strand only!

    SequenceWalkerConfig();

    const char*     seq;            //  sequence to split
    int             seqSize;        //  size of the sequence to split
    LRegion         range;          //  if not empty -> only this region is processed
    DNATranslation* complTrans; 
    DNATranslation* aminoTrans;
    
    int				chunkSize;          // optimal chunk size, used by default for all regions except last one
    int				lastChunkExtraLen;  // extra length allowed to be added to the last chunk
    int				overlapSize;        // overlap for 2 neighbor regions
	int				nThreads;
    StrandOption    strandToWalk;
};

class GB2_COREAPI_EXPORT SequenceWalkerCallback {
public:
    virtual ~SequenceWalkerCallback(){}

    virtual void onRegion(SequenceWalkerSubtask* t, TaskStateInfo& ti) = 0;
    
    /* implement this to give SequenceWalkerSubtask required resources
     * here are resources for ONE(!) SequenceWalkerSubtask execution e.g. for one execution of onRegion function
     */
    virtual QList< TaskResourceUsage > getResources( SequenceWalkerSubtask * t ) {Q_UNUSED(t); return QList< TaskResourceUsage >(); }
};

class GB2_COREAPI_EXPORT SequenceWalkerTask : public Task {
    Q_OBJECT
public:
    SequenceWalkerTask(const SequenceWalkerConfig& config, SequenceWalkerCallback* callback, 
        const QString& name, TaskFlags tf = TaskFlags_NR_FOSCOE);
    
    SequenceWalkerCallback*     getCallback() const {return callback;}
    const SequenceWalkerConfig& getConfig() const {return config;}

    // reverseMode - start splitting from the end of the range
    static QList<LRegion> splitRange(const LRegion& range, int chunkSize, int overlapSize, int lastChunkExtraLen, bool reverseMode);

    void setError(const QString& err) {stateInfo.setError(err);}

private:
    QList<SequenceWalkerSubtask*> prepareSubtasks();
    QList<SequenceWalkerSubtask*> createSubs(const QList<LRegion>& chunks, bool doCompl, bool doAmino);

    SequenceWalkerConfig    config;
    SequenceWalkerCallback* callback;
};

class GB2_COREAPI_EXPORT SequenceWalkerSubtask : public Task {
    Q_OBJECT
public:
    SequenceWalkerSubtask(SequenceWalkerTask* t, const LRegion& globalReg, bool lo, bool ro, 
                        const char* localSeq, int localLen, bool doCompl, bool doAmino);

    void run();
    
    const char* getRegionSequence();
    
    int  getRegionSequenceLen();
    
    bool isDNAComplemented() const {return doCompl;}
    
    bool isAminoTranslated() const {return doAmino;}
    
    LRegion getGlobalRegion() const {return globalRegion;}
    
    const SequenceWalkerConfig& getGlobalConfig() const {return t->getConfig();}

    bool intersectsWithOverlaps(const LRegion& globalReg) const;
    bool hasLeftOverlap() const {return leftOverlap;}
    bool hasRightOverlap() const {return rightOverlap;}
    
private:
    bool needLocalRegionProcessing() const {return (doAmino || doCompl) && processedSeqImage.isEmpty();}
    void prepareLocalRegion();

    SequenceWalkerTask*     t;
    LRegion                 globalRegion;
    const char*             localSeq;
    const char*             originalLocalSeq;
    int                     localLen;
    int                     originalLocalLen;
    bool                    doCompl;
    bool                    doAmino;
    bool                    leftOverlap;
    bool                    rightOverlap;
    
    QByteArray              processedSeqImage;

};


}//namespace

#endif
