/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_WORKFLOW_ATTR_H_
#define _GB2_WORKFLOW_ATTR_H_

#include <cassert>

#include <QtCore/QVariant>
#include <QtScript/QScriptValue>
#include <QtScript/QScriptEngine>

#include <core_api/Task.h>
#include <script/ScriptTask.h>

#include <workflow/Datatype.h>
#include <workflow/Descriptor.h>

namespace GB2 {

typedef QString ActorId;
inline ActorId str2aid(const QString& s) {return s;}

// attribute value can be obtained from script
class WCOREAPI_EXPORT AttributeScript {
public:
    bool isEmpty() const;
    
    void setScriptText(const QString & t);
    const QString & getScriptText() const;
    
    void setScriptVar(const Descriptor & desc, const QVariant & val);
    const QMap<Descriptor, QVariant> & getScriptVars() const;
    void clearScriptVars();
    
    bool hasVarWithId(const QString & varName)const;
    void setVarValueWithId(const QString & varName, const QVariant & value);
    
private:
    QString                         text;
    QMap<Descriptor, QVariant>      vars; // (desc, val)
    
}; // AttributeScript


class WCOREAPI_EXPORT Attribute : public Descriptor {
public:
    Attribute(const Descriptor& d, const DataTypePtr type, bool required = false, const QVariant & defValue = QVariant());
    
    const DataTypePtr getAttributeType()const;
    
    const bool isRequiredAttribute() const;
    
    void setAttributeValue(const QVariant & newVal);
    const QVariant & getAttributePureValue() const;
    
    // base realization without scripting. to support scripting for other types: see template realizations
    template<typename T> T getAttributeValue() const {
        return getAttributeValueWithoutScript<T>();
    }
    
    const AttributeScript & getAttributeScript() const;
    AttributeScript & getAttributeScript();
    
    QVariant toVariant() const; // stores value and script data in variant
    bool fromVariant(const QVariant& variant); // reads value and script from variant
    
private:
    template<typename T> T getAttributeValueWithoutScript() const {
        return value.value<T>();
    }
    
private:
    const DataTypePtr   type;
    const bool          required;
    QVariant            value;
    AttributeScript     scriptData;
    
}; // Attribute

// getAttributeValue function realizations with scripting support
template<>
inline QString Attribute::getAttributeValue() const {
    if( scriptData.isEmpty() ) {
        return getAttributeValueWithoutScript<QString>();
    }
    // run script
    QScriptEngine engine;
    QMap<QString, QScriptValue> scriptVars;
    foreach( const Descriptor & key, scriptData.getScriptVars().uniqueKeys() ) {
        assert(!key.getId().isEmpty());
        scriptVars[key.getId()] = engine.newVariant(scriptData.getScriptVars().value(key));
    }
    
    TaskStateInfo tsi;
    QScriptValue scriptResult = ScriptTask::runScript(&engine, scriptVars, scriptData.getScriptText(), tsi);
    
    // FIXME: report errors!
    // FIXME: write to log
    if( tsi.cancelFlag ) {
        if( !tsi.hasErrors() ) {
            tsi.setError("Script task canceled");
        }
    }
    if(tsi.hasErrors()) {
        return QString();
    }
    if( scriptResult.isString() ) {
        return scriptResult.toString();
    }
    
    return QString();
}

} // GB2 namespace

Q_DECLARE_METATYPE(GB2::AttributeScript)

#endif
