/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QDebug>

#include <workflow_support/WorkflowUtils.h>
#include <workflow_support/CoreDataTypes.h>

#include "IntegralBusModel.h"
#include "IntegralBusType.h"

/* TRANSLATOR GB2::Workflow::BusPort */
namespace GB2 {
namespace Workflow {

/*******************************
 * BusPort
 *******************************/
static void filterAmbiguousSlots(QList<Descriptor>& keys, const QMap<Descriptor, DataTypePtr>& map, QStrStrMap& result) {
    foreach(DataTypePtr val, map) {
        const QList<Descriptor> lst = map.keys(val);
        if (lst.size() != 1) {
            foreach(Descriptor d, lst) {
                result.insert(d.getId(), "");
                keys.removeOne(d);
            }
        }
    }
}

static Actor* getLinkedActor(ActorId id, Port* output) {
    if (output->owner()->getId() == id) {
        return output->owner();
    }
    foreach(Port* transit, output->owner()->getInputPorts()) {
        foreach(Port* p, transit->getLinks().uniqueKeys()) {
            Actor* a = getLinkedActor(id,p);
            if (a) return a;
        }
    }
    return NULL;
}

static QMap<QString, QStringList> getListMappings(const QStrStrMap& bm, const Port* p) {
    assert(p->isInput());    
    DataTypePtr dt = p->getType();
    QMap<QString, QStringList> res;
    if (dt->isList()) {
        QString val = bm.value(p->getId());
        if (!val.isEmpty()) {
            res.insert(p->getId(), val.split(";"));
        }
    } else if (dt->isMap()) {
        foreach(Descriptor d, dt->getAllDescriptors()) {
            QString val = bm.value(d.getId());
            if (dt->getDatatypeByDescriptor(d)->isList() && !val.isEmpty()) {
                res.insert(d.getId(), val.split(";"));
            }
        }
    }
    return res;
}

static void dumpBusMap(const QStrStrMap & busMap) {
    foreach( const QString & key, busMap.uniqueKeys() ) {
        qDebug() << "!" << key << "!" << busMap.value(key) << "!";
    }
}

const QString BusPort::BUS_MAP = "bus-map";

BusPort::BusPort(const PortDescriptor& d, Actor* p) : Port(d,p), recursing(false) {
    addParameter(BUS_MAP, new Attribute(Descriptor(BUS_MAP), DataTypePtr()));
}

DataTypePtr BusPort::getType() const {
    return isInput() ? type : getBusType();
}

DataTypePtr BusPort::getBusType() const {
    if (recursing) {
        return DataTypePtr(new IntegralBusType(Descriptor(), QMap<Descriptor, DataTypePtr>()));
    }
    recursing = true;
    IntegralBusType* t = new IntegralBusType(Descriptor(*this), QMap<Descriptor, DataTypePtr>());
    foreach (Port* p, owner()->getInputPorts()) {
        if ((p->getFlags()&BLIND_INPUT) == 0){
            t->addInputs(p);
        }
    }
    t->addOutput(type, this);
    recursing = false;
    return DataTypePtr(t);
}

Actor* BusPort::getProducer(const QString& slot) {
    QList<Actor*> l = getProducers(slot);
    if (l.size() == 1) {
        return l.first();
    } else {
        return NULL;
    }
}

QList<Actor*> BusPort::getProducers(const QString& slot) {
    QList<Actor*> res;
    Attribute* at = getParameter(BUS_MAP);
    if(at == NULL) {
        return res;
    }
    QStrStrMap busMap = at->getAttributeValue<QStrStrMap>();
    QString slotValue = busMap.value(slot);
    QStringList vals = slotValue.split(";");
    foreach(QString val, vals) {
        ActorId id = IntegralBusType::parseSlotDesc(val);
        Actor * a = getLinkedActorById(id);
        if(a != NULL) {
            res << a;
        }
    }
    return res;
}

Actor* BusPort::getLinkedActorById(ActorId id) const {
    QList<Actor*> res;
    foreach(Port* peer, getLinks().uniqueKeys()) {
        Actor* ac = getLinkedActor(id,peer);
        if(ac != NULL) {
            res << ac;
        }
    }

    Actor * ret = NULL;
    if( res.size() == 0 ) {
        ret = NULL;
    } else if( res.size() > 1 ) {
        ret = res.first();
        assert(false);
    } else {
        ret = res.first();
    }
    return ret;
}

void BusPort::remap(const QMap<ActorId, ActorId>& m) {
    Attribute* a = getParameter(BUS_MAP);
    if (a) {
        QStrStrMap busMap = a->getAttributeValue<QStrStrMap>();
        IntegralBusType::remap(busMap, m);
        setParameter(BUS_MAP, qVariantFromValue<QStrStrMap>(busMap));
    }
}

void BusPort::setupBusMap() {
    if( !isInput() || getWidth() != 1 ) {
        return;
    }
    
    DataTypePtr to = getType();
    if (!to->isMap()) {
        QMap<Descriptor, DataTypePtr> map;
        map.insert(*this, to);
        to = new MapDataType(Descriptor(), map);
    }

    DataTypePtr from = bindings.uniqueKeys().first()->getType();
    QList<Descriptor> keys = to->getAllDescriptors();
    QStrStrMap busMap = getParameter(BusPort::BUS_MAP)->getAttributeValue<QStrStrMap>();
    filterAmbiguousSlots(keys, to->getDatatypesMap(), busMap);
    foreach(Descriptor key, keys) {
        DataTypePtr elementDatatype = to->getDatatypeByDescriptor(key);
        QStringList candidates = DesignerUtils::findMatchingTypesAsStringList(from, elementDatatype);
        if (elementDatatype->isList()) {
            candidates += DesignerUtils::findMatchingTypesAsStringList(from, elementDatatype->getDatatypeByDescriptor());
            QString res = candidates.join(";");
            busMap.insert(key.getId(), res);
        } else if (candidates.size() != 1) {
            //no unambiguous match, reset
            busMap.insert(key.getId(), "");
        } else {
            // for string type initial value - empty. Because string type is always path to file and not needed for binding
            if( elementDatatype == CoreDataTypes::STRING_TYPE() ) {
                busMap.insert(key.getId(), "");
            } else {
                busMap.insert(key.getId(), candidates.first());
                
                /*QString old = busMap.value(key.getId());
                candidates.append(Descriptor(""));
                int idx = busMap.contains(key.getId()) ? candidates.indexOf(old) : 0;
                Descriptor actual = (idx >= 0) ? candidates.at(idx) : candidates.first();
                busMap.insert(key.getId(), actual.getId());*/
            }
        }
    }
    
    setParameter(BUS_MAP, qVariantFromValue<QStrStrMap>(busMap));
}

bool BusPort::validate(QStringList& l) const {
    bool good = Configuration::validate(l);
    if (isInput() && !validator) {
        good &= ScreenedSlotValidator::validate(QStringList(), this, l);
    }
    return good;
}

/*******************************
* ScreenedSlotValidator
*******************************/
bool ScreenedSlotValidator::validate( const QStringList& screenedSlots, const BusPort* vport, QStringList& l) 
{
    bool good = true;
    {
        if (vport->getWidth() == 0) {
            l.append(BusPort::tr("No input data supplied"));
            return false;
        }
        QStrStrMap bm = vport->getParameter(BusPort::BUS_MAP)->getAttributeValue<QStrStrMap>();
        int busWidth = bm.size();
        QMap<QString, QStringList> listMap = getListMappings(bm, vport);
        // iterate over all producers and exclude valid mappings from bus bindings
        foreach(Port* p, vport->getLinks().uniqueKeys()) {
            assert(qobject_cast<BusPort*>(p));//TBD?
            DataTypePtr t = p->getType();
            assert(t->isMap());
            {
                foreach(Descriptor d, t->getAllDescriptors()) {
                    foreach(QString key, bm.keys(d.getId())) {
                        //log.debug("reducing bus from key="+ikey+" to="+rkey);
                        assert(!key.isEmpty());
                        bm.remove(key);
                    }
                    foreach(QString key, listMap.uniqueKeys()) {
                        QStringList& l = listMap[key];
                        l.removeAll(d.getId());
                        if (l.isEmpty()) {
                            listMap.remove(key);
                            bm.remove(key);
                        }
                    }
                }
            }
        }
        if (busWidth == bm.size()) {
            l.append(BusPort::tr("No input data supplied"));
            good = false;
        }
        {
            QMapIterator<QString,QString> it(bm);
            while (it.hasNext())
            {
                it.next();
                const QString& slot = it.key();
                QString slotName = vport->getType()->getDatatypeDescriptor(slot).getDisplayName();
                //assert(!slotName.isEmpty());
                if (it.value().isEmpty()) {
                    if (!screenedSlots.contains(slot)) {
                        l.append(BusPort::tr("Warning, empty input slot: %1").arg(slotName));
                    }
                } else {
                    l.append(BusPort::tr("Bad slot binding: %1 to %2").arg(slotName).arg(it.value()));
                    good = false;
                }
            }
        }
        {
            QMapIterator<QString,QStringList> it(listMap);
            while (it.hasNext())
            {
                it.next();
                const QString& slot = it.key();
                QString slotName = vport->getType()->getDatatypeDescriptor(slot).getDisplayName();
                assert(!slotName.isEmpty());
                assert(!it.value().isEmpty());
                l.append(BusPort::tr("Bad slot binding: %1 to %2").arg(slotName).arg(it.value().join(",")));
                good = false;
            }
        }
    }
    return good;
}

bool ScreenedSlotValidator::validate( const Configuration* cfg, QStringList& output ) const {
    return validate(screenedSlots, static_cast<const BusPort*>(cfg), output);
}


/*******************************
* ScreenedParamValidator
*******************************/
ScreenedParamValidator::ScreenedParamValidator(const QString& id, const QString& port, const QString& slot) 
: id(id), port(port), slot(slot) {}


bool ScreenedParamValidator::validate(const Configuration* cfg, QStringList& output) const {
    Attribute* param = cfg->getParameter(id);
    QVariant val = param->getAttributePureValue();
    const Workflow::Actor* a = dynamic_cast<const Workflow::Actor*>(cfg);
    assert(a);
    Workflow::Port* p = a->getPort(port);
    assert(p->isInput());
    QVariant busMap = p->getParameter(Workflow::BusPort::BUS_MAP)->getAttributePureValue();
    QString slotVal = busMap.value<QStrStrMap>().value(slot);
    const bool noParam = ( val.isNull() || val.toString().isEmpty() ) && param->getAttributeScript().isEmpty();
    const bool noSlot = slotVal.isNull() || slotVal.isEmpty();
    if (noParam && noSlot) {
        QString slotName = p->getType()->getDatatypeDescriptor(slot).getDisplayName(); 
        assert(!slotName.isEmpty());
        output.append(GB2::DesignerUtils::tr("Either parameter '%1' or input slot '%2' must be set")
            .arg(param->getDisplayName()).arg(slotName));//FIXME translator class
        return false;
    }
    if (noParam == noSlot) {
        QString slotName = p->getType()->getDatatypeDescriptor(slot).getDisplayName();
        assert(!slotName.isEmpty());
        output.append(GB2::DesignerUtils::tr("Warning, parameter '%1' overrides bus data slot '%2'")
            .arg(param->getDisplayName()).arg(slotName));//FIXME translator class
    }
    return true;
}

}//Workflow namespace
}//GB2namespace
