/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowModel.h"
#include "ConfigurationEditor.h"

namespace GB2 {
namespace Workflow {

/**************************
 * VisualDescriptor
 **************************/
VisualDescriptor::VisualDescriptor(const Descriptor& d, const QString & _iconPath) : Descriptor(d), iconPath(_iconPath) {
}

void VisualDescriptor::setIconPath( const QString & ip ) {
    iconPath = ip;
}

QIcon VisualDescriptor::getIcon() {
    if( icon.isNull() && !iconPath.isEmpty() ) {
        icon = QIcon(iconPath);
    }
    return icon;
}

void VisualDescriptor::setIcon( QIcon i ) {
    assert( iconPath.isEmpty() );
    icon = i;
}

/**************************
* PortDescriptor
**************************/
PortDescriptor::PortDescriptor(const Descriptor& desc, DataTypePtr type, bool input, bool multi, uint f )
: Descriptor(desc), type(type), input(input), multi(multi), flags(f) {
}

bool PortDescriptor::isInput() const {
    return input;
}

bool PortDescriptor::isOutput() const {
    return !input;
}

bool PortDescriptor::isMulti() const {
    return multi;
}

uint PortDescriptor::getFlags() const {
    return flags;
}

DataTypePtr PortDescriptor::getType() const {
    return type;
}

/**************************
* Port
**************************/
Port::Port(const PortDescriptor& d, Actor* p) : PortDescriptor(d), proc(p) {
}

Actor * Port::owner() const {
    return proc;
}

QMap<Port*, Link*> Port::getLinks() const {
    return bindings;
}

int Port::getWidth() const {
    return bindings.size();
}

void Port::setParameter(const QString& name, const QVariant& val) {
    Configuration::setParameter(name, val);
    emit bindingChanged();
}

void Port::remap(const QMap<ActorId, ActorId>&) {
}

bool Port::canBind(const Port* other) const {
    if (this == other || proc == other->proc || isInput() == other->isInput()) {
        return false;
    }
    if ((!isMulti() && getWidth() != 0) || (!other->isMulti() && other->getWidth() != 0)) {
        return false;
    }
    return !bindings.contains(const_cast<Port*>(other));
}

void Port::addLink(Link* b) {
    Port* peer = isInput()? b->source() : b->destination();
    assert(this == (isInput()? b->destination() : b->source()));
    //assert(canBind(peer));
    assert(!bindings.contains(peer));
    bindings[peer] = b;
    emit bindingChanged();
}

void Port::removeLink(Link* b) {
    Port* peer = isInput()? b->source() : b->destination();
    assert(this == (isInput()? b->destination() : b->source()));
    assert(bindings.contains(peer));
    bindings.remove(peer);
    emit bindingChanged();
}

/**************************
 * Actor
 **************************/
Actor::Actor(ActorPrototype* proto) : proto(proto), doc(NULL), numberInSchema(-1) {
}

Actor::Actor(const Actor&) : QObject(), Configuration(), Peer(), numberInSchema(-1) {
    assert(false);
}

Actor::~Actor() {
    qDeleteAll(ports.values()); 
    delete doc;
}

ActorPrototype * Actor::getProto() const {
    return proto;
}

ActorId Actor::getId() const {
    return QString("%1").arg( (int) (size_t) this);
}

QString Actor::getLabel() const {
    if( label.isEmpty() ) {
        return QString("%1 %2").arg(getProto()->getDisplayName()).arg(getId());
    } else {
        return label;
    }
}

void Actor::setLabel(const QString& l) {
    label = l; 
    emit si_labelChanged();
}

Port * Actor::getPort(const QString& id) const {
    return ports.value(id);
}

QList<Port*> Actor::getPorts() const {
    return ports.values();
}

void Actor::setParameter(const QString& name, const QVariant& val) {
    Configuration::setParameter(name, val);
    emit si_modified();
}

ActorDocument * Actor::getDescription() const {
    return doc;
}

void Actor::setDescription(ActorDocument* d) {
    assert(d != NULL);
    doc = d;
}

const QMap<QString, QString> & Actor::getParamAliases() const {
    return paramAliases;
}

QMap<QString, QString> & Actor::getParamAliases() {
    return paramAliases;
}

void Actor::remap(const QMap<ActorId, ActorId>& m) {
    foreach(Port* p, ports) {
        p->remap(m);
    }
}

QList<Port*> Actor::getInputPorts() const {
    QList<Port*> l; foreach (Port* p, ports.values()) if (p->isInput()) l<<p; 
    return l;
}

QList<Port*> Actor::getOutputPorts() const {
    QList<Port*> l; foreach (Port* p, ports.values()) if (p->isOutput()) l<<p; 
    return l;
}

void Actor::setNumberInSchema(int n) {
    assert(n >= 0);
    numberInSchema = n;
}

int Actor::getNumberInSchema() const {
    return numberInSchema;
}

QString Actor::getPortAttributeShortName(const QString & attrSlotId) const {
    return QString("%1_%2").arg(attrSlotId).arg(numberInSchema);
}

/**************************
 * ActorPrototype
 **************************/
QList<PortDescriptor*> ActorPrototype::getPortDesciptors() const {
    return ports;
}

QList<Attribute*> ActorPrototype::getAttributes() const {
    return attrs;
}

void ActorPrototype::addAttribute( Attribute * a ) {
    assert(a != NULL);
    attrs << a;
}

int ActorPrototype::removeAttribute( Attribute * attr ) {
    assert(attr != NULL);
    return attrs.removeAll( attr );
}

void ActorPrototype::setEditor(ConfigurationEditor* e) {
    assert(e != NULL);
    ed = e;
}

ConfigurationEditor * ActorPrototype::getEditor()const {
    return ed;
}

void ActorPrototype::setValidator(ConfigurationValidator* v) {
    assert(v != NULL);
    val = v;
}

void ActorPrototype::setPrompter(Prompter* p) {
    assert(p != NULL);
    prompter = p;
}

void ActorPrototype::setPortValidator(const QString& id, ConfigurationValidator* v) {
    assert(v != NULL);
    portValidators[id] = v;
}

bool ActorPrototype::isAcceptableDrop(const QMimeData*, QVariantMap* ) const {
    return false;
}

Port* ActorPrototype::createPort(const PortDescriptor& d, Actor* p) {
    return new Port(d, p);
}

Actor* ActorPrototype::createInstance(const QMap<QString, QVariant>& params) {
    Actor* proc = new Actor(this);

    foreach(PortDescriptor* pd, getPortDesciptors()) {
        Port* p = createPort(*pd, proc);
        QString pid = pd->getId();
        if (portValidators.contains(pid)) {
            p->setValidator(portValidators.value(pid));
        }
        proc->ports[pid] = p;
    }
    foreach(Attribute* a, getAttributes()) {
        proc->addParameter(a->getId(), new Attribute(*a));
    }
    if (ed) {
        proc->setEditor(ed);
    }
    if (val) {
        proc->setValidator(val);
    }
    if (prompter) {
        proc->setDescription(prompter->createDescription(proc));
    }

    QMapIterator<QString, QVariant> i(params);
    while (i.hasNext()) {
        i.next();
        proc->setParameter(i.key(), i.value());
    }
    return proc;
}


Attribute * ActorPrototype::getAttribute( const QString & id ) const {
    Attribute * res = NULL;
    foreach( Attribute * a, attrs ) {
        if( a->getId() == id ) {
            res = a;
            break;
        }
    }
    return res;
}

ActorPrototype::ActorPrototype(const Descriptor& d, 
                               const QList<PortDescriptor*>& ports, 
                               const QList<Attribute*>& attrs)
                  : VisualDescriptor(d), attrs(attrs), ports(ports), ed(NULL), val(NULL), prompter(NULL) {
}

ActorPrototype::~ActorPrototype()
{
    qDeleteAll(attrs);
    qDeleteAll(ports);
    delete ed;
    delete val;
    delete prompter;
    qDeleteAll(portValidators);
}

/**************************
 * Peer
 **************************/
Peer::Peer() : peer(NULL) {
}

void * Peer::getPeer() const {
    return peer;
}

void Peer::setPeer(void* p) {
    peer = p;
}



/**************************
 * Link
 **************************/
Link::Link()  : src(NULL), dest(NULL) {
}

Link::Link(Port* p1, Port* p2) {
    connect(p1, p2);
}

Port * Link::source() const {
    return src;
}

Port * Link::destination() const {
    return dest;
}

void Link::connect(Port* p1, Port* p2) {
    assert(p1->canBind(p2));
    if (p1->isInput()) {
        dest = p1;
        src = p2;
    } else {
        dest = p2;
        src = p1;
    }
    p1->addLink(this);
    p2->addLink(this);
}

/**************************
 * ActorDocument
 **************************/
ActorDocument::ActorDocument(Actor* a) : QTextDocument(a), target(a) {
}

void ActorDocument::update(const QVariantMap& ) {
}

}//Workflow namespace
}//GB2namespace
