/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_WORKFLOW_MODEL_H_
#define _GB2_WORKFLOW_MODEL_H_

#include <workflow/Datatype.h>
#include <workflow/Attribute.h>
#include <workflow/Configuration.h>
#include <workflow/ConfigurationEditor.h>
#include <workflow/ConfigurationValidator.h>

#include <QtCore/QMimeData>
#include <QtGui/QIcon>
#include <QtGui/QTextDocument>

#include <assert.h>

namespace GB2 {

namespace Workflow {

class Actor;
class ActorDocument;
class CompositeActor;
class PortDescriptor;
class Port;
class Link;
class Schema;

class WCOREAPI_EXPORT VisualDescriptor : public Descriptor {
public:
    VisualDescriptor(const Descriptor& d, const QString & _iconPath = QString());
    void setIconPath( const QString & iconPath_ );

    //getIcon will create icon from its path (if path exists) on the first call. 
    //This is compatibility issue for congene.
    QIcon getIcon();
    void setIcon(QIcon icon_);
    
private:
    QString iconPath;
    QIcon icon;
    
}; // VisualDescriptor

class WCOREAPI_EXPORT Prompter {
public:
    virtual ActorDocument* createDescription(Actor*) = 0;
    virtual ~Prompter() {};
}; // Prompter

class WCOREAPI_EXPORT ActorPrototype : public VisualDescriptor {
public:
    ActorPrototype(const Descriptor& desc, 
        const QList<PortDescriptor*>& ports = QList<PortDescriptor*>(), 
        const QList<Attribute*>& attrs = QList<Attribute*>());
    
    virtual ~ActorPrototype();
    QList<PortDescriptor*> getPortDesciptors() const;
    
    QList<Attribute*> getAttributes() const;
    Attribute * getAttribute( const QString & id ) const;
    void addAttribute( Attribute * a );
    int removeAttribute( Attribute * attr );
    
    void setEditor(ConfigurationEditor* e);
    ConfigurationEditor * getEditor() const;
    void setValidator(ConfigurationValidator* v);
    void setPrompter(Prompter* p);
    void setPortValidator(const QString& id, ConfigurationValidator* v);

    virtual bool isAcceptableDrop(const QMimeData*, QVariantMap* = NULL) const;

    typedef QMap<QString,QVariant> ParamsMap;
    virtual Actor* createInstance(const QMap<QString, QVariant>& params = ParamsMap());

protected:
    virtual Port* createPort(const PortDescriptor& d, Actor* p);

protected:
    QList<Attribute*> attrs;
    QList<PortDescriptor*> ports; 
    ConfigurationEditor* ed;
    ConfigurationValidator* val;
    Prompter* prompter;
    QMap<QString, ConfigurationValidator*> portValidators;
    
}; // ActorPrototype

class WCOREAPI_EXPORT Peer {
public:
    Peer();
    virtual ~Peer() {}
    
    template<typename T> inline T* castPeer() const {
        return (T*)(peer);
    }
    
    void* getPeer() const;
    void setPeer(void* p);
    
protected:
    void* peer;
    
}; // Peer

class WCOREAPI_EXPORT Actor: public QObject, public Configuration, public Peer {
    Q_OBJECT
public:
    virtual ~Actor();
    ActorPrototype* getProto() const;
    ActorId getId() const;

    QString getLabel() const;
    void setLabel(const QString& l);

    Port* getPort(const QString& id) const;
    QList<Port*> getPorts() const;
    QList<Port*> getInputPorts() const;
    QList<Port*> getOutputPorts() const;
    
    virtual void setParameter(const QString& name, const QVariant& val);
    
    ActorDocument* getDescription() const;
    void setDescription(ActorDocument* d);
    
    const QMap<QString, QString> & getParamAliases() const;
    QMap<QString, QString> & getParamAliases();
    
    virtual void remap(const QMap<ActorId, ActorId>&);
    
    void setNumberInSchema(int n);
    int getNumberInSchema() const;
    // create short alias for attribute, that actor put to port
    // used in scripting for naming data came from bus
    QString getPortAttributeShortName(const QString & attrSlotId)const;
        
signals:
    void si_labelChanged();
    void si_modified();
    
protected:
    friend class ActorPrototype;
    Actor(ActorPrototype* proto);
    Actor(const Actor&);
    
protected:
    QString label;
    ActorPrototype* proto;
    QMap<QString,Port*> ports;
    ActorDocument* doc;
    
    // user can set aliases for schema parameters and use them in cmdline
    QMap<QString, QString> paramAliases; // ( paramName, alias ) pairs
    
    int numberInSchema; // schema numbers its actors with unique numbers (usually: 1, 2, 3 ...)
    
}; // Actor

class WCOREAPI_EXPORT PortDescriptor : public Descriptor {
public:
    PortDescriptor(const Descriptor& desc, DataTypePtr type, bool input, bool multi = false, uint f = 0);
    virtual ~PortDescriptor() {}
    
    bool isInput() const;
    bool isOutput() const;
    bool isMulti() const;
    uint getFlags() const;
    virtual DataTypePtr getType() const;
    
protected:
    DataTypePtr type;
    bool input;
    bool multi;
    uint flags;
    
}; // PortDescriptor

class WCOREAPI_EXPORT Port : public QObject, public PortDescriptor, public Configuration,  public Peer {
    Q_OBJECT
public:
    Port(const PortDescriptor& d, Actor* p);
    virtual ~Port() {}
    
    Actor* owner() const;
    
    QMap<Port*,Link*> getLinks() const;
    virtual void addLink(Link*);
    virtual void removeLink(Link*);
    int getWidth() const;
    
    virtual void setParameter(const QString& name, const QVariant& val);
    virtual bool canBind(const Port* other) const;
    virtual void remap(const QMap<ActorId, ActorId>&);
    
signals:
    void bindingChanged();
    
protected:
    Actor* proc;
    QMap<Port*,Link*> bindings;
    
}; // Port

class WCOREAPI_EXPORT Link : public QObject, public Peer {
    Q_OBJECT
public:
    Link();
    Link(Port* p1, Port* p2);
    virtual ~Link() {}
    
    void connect(Port* p1, Port* p2);
    Port* source() const;
    Port* destination() const;
    
private:
    Port *src;
    Port *dest;
    
}; // Link

class WCOREAPI_EXPORT ActorDocument : public QTextDocument {
public:
    ActorDocument(Actor* a);
    virtual ~ActorDocument() {}
    
    virtual void update(const QVariantMap& );
    
protected:
    Actor* target;
    
}; // ActorDocument

}//Workflow namespace
}//GB2 namespace


#endif
