/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QSpinBox>
#include <QtGui/QComboBox>
#include <QtGui/QLineEdit>
#include <QtGui/QToolButton>
#include <QtGui/QHBoxLayout>
#include <QtGui/QFileDialog>
#include <QtCore/QCoreApplication>
#include <QtGui/QKeyEvent>

#include <core_api/Log.h>
#include <workflow/IntegralBusModel.h>
#include <util_gui/DialogUtils.h>
#include <util_tasks/SaveDocumentTask.h>
#include <script/ScriptEditorDialog.h>

#include "DelegateEditors.h"
#include "WorkflowUtils.h"

namespace GB2 {

/********************************
 * SpinBoxDelegate
 ********************************/
QWidget *SpinBoxDelegate::createEditor(QWidget *parent,
    const QStyleOptionViewItem &/* option */,
    const QModelIndex &/* index */) const
{
    QSpinBox *editor = new QSpinBox(parent);
    DesignerUtils::setQObjectProperties(*editor, spinProperties);

    return editor;
}

void SpinBoxDelegate::setEditorData(QWidget *editor,
                                    const QModelIndex &index) const
{
    int value = index.model()->data(index, ConfigurationEditor::ItemValueRole).toInt();
    QSpinBox *spinBox = static_cast<QSpinBox*>(editor);
    spinBox->setValue(value);
}

void SpinBoxDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                                   const QModelIndex &index) const
{
    QSpinBox *spinBox = static_cast<QSpinBox*>(editor);
    spinBox->interpretText();
    int value = spinBox->value();
    model->setData(index, value, ConfigurationEditor::ItemValueRole);
}

QVariant SpinBoxDelegate::getDisplayValue( const QVariant& v) const {
    QSpinBox editor;
    DesignerUtils::setQObjectProperties(editor, spinProperties);
    editor.setValue(v.toInt());
    return editor.text();
}

/********************************
* DoubleSpinBoxDelegate
********************************/
QWidget *DoubleSpinBoxDelegate::createEditor(QWidget *parent,
                                       const QStyleOptionViewItem &/* option */,
                                       const QModelIndex &/* index */) const
{
    QDoubleSpinBox *editor = new QDoubleSpinBox(parent);
    DesignerUtils::setQObjectProperties(*editor, spinProperties);

    return editor;
}

void DoubleSpinBoxDelegate::setEditorData(QWidget *editor,
                                    const QModelIndex &index) const
{
    double value = index.model()->data(index, ConfigurationEditor::ItemValueRole).toDouble();
    QDoubleSpinBox *spinBox = static_cast<QDoubleSpinBox*>(editor);
    spinBox->setValue(value);
}

void DoubleSpinBoxDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                                   const QModelIndex &index) const
{
    QDoubleSpinBox *spinBox = static_cast<QDoubleSpinBox*>(editor);
    spinBox->interpretText();
    double value = spinBox->value();
    model->setData(index, value, ConfigurationEditor::ItemValueRole);
}

QVariant DoubleSpinBoxDelegate::getDisplayValue( const QVariant& v) const {
    QDoubleSpinBox editor;
    DesignerUtils::setQObjectProperties(editor, spinProperties);
    editor.setValue(v.toDouble());
    return editor.text();
}

/********************************
* ComboBoxDelegate
********************************/
QWidget *ComboBoxDelegate::createEditor(QWidget *parent,
                                       const QStyleOptionViewItem &/* option */,
                                       const QModelIndex &/* index */) const
{
    QComboBox *editor = new QComboBox(parent);
    //editor->setFrame(false);
    //editor->setSizePolicy(QSizePolicy(QSizePolicy::Preferred,QSizePolicy::Preferred));

    QMapIterator<QString, QVariant> it(items);
    while (it.hasNext())
    {
        it.next();
        editor->addItem(it.key(), it.value());
    }
    
    connect( editor, SIGNAL( activated( const QString & ) ), this, SIGNAL( si_valueChanged( const QString & ) ) );
    return editor;
}

void ComboBoxDelegate::setEditorData(QWidget *editor,
                                    const QModelIndex &index) const
{
    QVariant val = index.model()->data(index, ConfigurationEditor::ItemValueRole);
    QComboBox *box = static_cast<QComboBox*>(editor);
    int idx = box->findData(val);
    box->setCurrentIndex(idx);
}

void ComboBoxDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                                   const QModelIndex &index) const
{
    QComboBox *box = static_cast<QComboBox*>(editor);
    QVariant val = box->itemData(box->currentIndex());
    model->setData(index, val, ConfigurationEditor::ItemValueRole);
}

QVariant ComboBoxDelegate::getDisplayValue(const QVariant& val) const {
    QString display = items.key(val);
    emit si_valueChanged( display );
    return QVariant( display );
}

/********************************
* URLLineEdit
********************************/
void URLLineEdit::sl_onBrowse() {
    LastOpenDirHelper lod(type);

    QString name;
    if (multi) {
        QStringList lst = QFileDialog::getOpenFileNames(NULL, tr("Select file(s)"), lod.dir, FileFilter);
        name = lst.join(";");
        if (!lst.isEmpty()) {
            lod.url = lst.first();
        }
    } else {
        lod.url = name = QFileDialog::getSaveFileName(NULL, tr("Select a file"), lod.dir, FileFilter, 0, QFileDialog::DontConfirmOverwrite);
    }
    if (!name.isEmpty()) {
        setText(name);
        QKeyEvent accept(QEvent::KeyPress, Qt::Key_Enter, Qt::NoModifier);
        if (QCoreApplication::sendEvent(this, &accept)) {
            return;
        }
    }
    setFocus();
}

/********************************
* URLDelegate
********************************/
QWidget *URLDelegate::createEditor(QWidget *parent,
                                       const QStyleOptionViewItem &/* option */,
                                       const QModelIndex &/* index */) const
{
    QWidget * widget = new QWidget(parent);
    URLLineEdit* documentURLEdit = new URLLineEdit(FileFilter, type, multi, widget);
    documentURLEdit->setObjectName("URLLineEdit");
    documentURLEdit->setFrame(false);
    documentURLEdit->setSizePolicy(QSizePolicy(QSizePolicy::Preferred,QSizePolicy::Preferred));
    widget->setFocusProxy(documentURLEdit);
    QToolButton * toolButton = new QToolButton(widget);
    toolButton->setVisible( showButton );
    toolButton->setText("...");
    toolButton->setSizePolicy(QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Preferred));
    connect(toolButton, SIGNAL(clicked()), documentURLEdit, SLOT(sl_onBrowse()));
    
    QHBoxLayout* layout = new QHBoxLayout(widget);
    layout->setSpacing(0);
    layout->setMargin(0);
    layout->addWidget(documentURLEdit);
    layout->addWidget(toolButton);
    
    return widget;
}

void URLDelegate::setEditorData(QWidget *editor,
                                    const QModelIndex &index) const
{
    QString val = index.model()->data(index, ConfigurationEditor::ItemValueRole).toString();
    QLineEdit* ed = editor->findChild<QLineEdit*>("URLLineEdit");
    assert(ed);
    ed->setText(val);
}

void URLDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                                   const QModelIndex &index) const
{
    QLineEdit* ed = editor->findChild<QLineEdit*>("URLLineEdit");
    assert(ed);
    QString val = ed->text().replace('\\', '/').trimmed();
    model->setData(index, val, ConfigurationEditor::ItemValueRole);
    if (multi) {
        QVariantList vl;
        foreach(QString s, val.split(";")) {
            vl.append(s.trimmed());
        }
        model->setData(index, vl, ConfigurationEditor::ItemListValueRole);
    }
}

void URLDelegate::sl_showEditorButton( bool show ) {
    showButton = show;
}

/********************************
* FileModeDelegate
********************************/
FileModeDelegate::FileModeDelegate(bool appendSupported, QObject *parent) 
: ComboBoxDelegate(QVariantMap(), parent) {
    items.insert(GB2::DesignerUtils::tr("Overwrite"), SaveDoc_Overwrite);
    items.insert(GB2::DesignerUtils::tr("Rename"), SaveDoc_Roll);
    if (appendSupported) {
        items.insert(GB2::DesignerUtils::tr("Append"), SaveDoc_Append);
    }
}

/********************************
 * SchemaRunModeDelegate
 ********************************/
const QString SchemaRunModeDelegate::THIS_COMPUTER_STR      = SchemaRunModeDelegate::tr( "This computer" );
const QString SchemaRunModeDelegate::REMOTE_COMPUTER_STR    = SchemaRunModeDelegate::tr( "Remote computer" );

SchemaRunModeDelegate::SchemaRunModeDelegate( QObject * parent )
: ComboBoxDelegate( QVariantMap(), parent ) {
    items.insert( THIS_COMPUTER_STR, true );
    items.insert( REMOTE_COMPUTER_STR, false );
    
    connect( this, SIGNAL( si_valueChanged( const QString & ) ), this, 
        SLOT( sl_valueChanged( const QString & ) ) );
}

void SchemaRunModeDelegate::sl_valueChanged( const QString & val ) {
    emit si_showOpenFileButton( THIS_COMPUTER_STR == val );
}

/********************************
* AttributeScriptDelegate
********************************/
static QString createScriptHeader(const AttributeScript & attrScript) {
    QString header;
    foreach( const Descriptor & desc, attrScript.getScriptVars().keys() ) {
        header += QString("var %1; // %2\n").arg(desc.getId()).arg(desc.getDisplayName());
    }
    return header;
}

const int NO_SCRIPT_ITEM_ID = 0;
const int USER_SCRIPT_ITEM_ID = 1;
const QPair<QString, int> NO_SCRIPT_ITEM_STR("no script", NO_SCRIPT_ITEM_ID);
const QPair<QString, int> USER_SCRIPT_ITEM_STR("user script", USER_SCRIPT_ITEM_ID);

const QString SCRIPT_PROPERTY = "combo_script_property";

AttributeScriptDelegate::AttributeScriptDelegate(QObject *parent) : PropertyDelegate(parent) {
}

AttributeScriptDelegate::~AttributeScriptDelegate() {
}

void AttributeScriptDelegate::sl_comboActivated(int itemId ) {
    QComboBox * editor = qobject_cast<QComboBox*>(sender());
    assert(editor != NULL);
    
    switch(itemId) {
    case NO_SCRIPT_ITEM_ID:
        {
            editor->setItemData( USER_SCRIPT_ITEM_ID, "", ConfigurationEditor::ItemValueRole );
            return;
        }
    case USER_SCRIPT_ITEM_ID:
        {
            QComboBox * combo = qobject_cast<QComboBox*>(sender());
            assert(combo != NULL);
            AttributeScript attrScript = combo->property(SCRIPT_PROPERTY.toAscii().constData()).value<AttributeScript>();
            
            ScriptEditorDialog dlg(editor, createScriptHeader(attrScript));
            dlg.setScriptText(attrScript.getScriptText());
            
            int rc = dlg.exec();
            if(rc != QDialog::Accepted) {
                editor->setItemData( USER_SCRIPT_ITEM_ID, 
                    qVariantFromValue<AttributeScript>(attrScript), ConfigurationEditor::ItemValueRole );
                return;
            }
            attrScript.setScriptText(dlg.getScriptText());
            editor->setItemData( USER_SCRIPT_ITEM_ID, 
                qVariantFromValue<AttributeScript>(attrScript), ConfigurationEditor::ItemValueRole );
            return;
        }
    default:
        assert(false);
    }
}

QWidget * AttributeScriptDelegate::createEditor(QWidget *parent, const QStyleOptionViewItem &option, const QModelIndex &index) const {
    QComboBox *editor = new QComboBox(parent);
    editor->addItem(NO_SCRIPT_ITEM_STR.first);
    editor->addItem(USER_SCRIPT_ITEM_STR.first);
    connect(editor, SIGNAL(activated(int)), SLOT(sl_comboActivated(int)));
    return editor;
}

void AttributeScriptDelegate::setEditorData(QWidget *editor, const QModelIndex &index) const {
    QComboBox * combo = qobject_cast<QComboBox*>(editor);
    assert(combo != NULL);
    AttributeScript attrScript = index.model()->data(index, ConfigurationEditor::ItemValueRole).value<AttributeScript>();
    if( attrScript.isEmpty() ) {
        combo->setCurrentIndex(NO_SCRIPT_ITEM_STR.second);
    } else {
        combo->setCurrentIndex(USER_SCRIPT_ITEM_STR.second);
    }
    combo->setProperty(SCRIPT_PROPERTY.toAscii().constData(), qVariantFromValue<AttributeScript>(attrScript));
}

void AttributeScriptDelegate::setModelData(QWidget *editor, QAbstractItemModel *model, const QModelIndex &index) const {
    QComboBox * combo = qobject_cast<QComboBox*>(editor);
    assert(combo != NULL);
    model->setData(index, combo->itemData(USER_SCRIPT_ITEM_ID, ConfigurationEditor::ItemValueRole), ConfigurationEditor::ItemValueRole);
}

QVariant AttributeScriptDelegate::getDisplayValue(const QVariant& val) const{
    AttributeScript attrScript = val.value<AttributeScript>();
    QString ret = attrScript.isEmpty() ? NO_SCRIPT_ITEM_STR.first : USER_SCRIPT_ITEM_STR.first;
    return QVariant(ret);
}

/********************************
 * AttributeScriptDelegate
 ********************************/
//InputPortDataDelegate::InputPortDataDelegate(const QVariantMap& items, QObject *parent ) : ComboBoxDelegate(items, parent) {
//}
//
//InputPortDataDelegate::~InputPortDataDelegate() {
//}

}//namespace GB2
