/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowUtils.h"

#include <workflow/Descriptor.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/IntegralBusType.h>
#include <workflow_support/SchemaSerializer.h>
#include <core_api/GObject.h>
#include <core_api/DocumentModel.h>
#include <document_format/DocumentFormatUtils.h>
#include <util_gui/DialogUtils.h>

#include <QtCore/QDir>
#include <QtCore/QFileInfo>
#include <QtCore/QUrl>
#include <QtGui/QListWidgetItem>

namespace GB2 {

/*****************************
 * DesignerUtils
 *****************************/
QString DesignerUtils::getRichDoc(const Descriptor& d) {
    if(d.getDisplayName().isEmpty()) {
        if( d.getDocumentation().isEmpty() ) {
            return QString();
        } else {
            return QString("%1").arg(d.getDocumentation());
        }
    } else {
        if( d.getDocumentation().isEmpty() ) {
            return QString("<b>%1</b>").arg(d.getDisplayName());
        } else {
            return QString("<b>%1</b> : %2").arg(d.getDisplayName()).arg(d.getDocumentation());
        }
    }
}

const QString DesignerUtils::SCHEMA_FILE_EXT("uws");
QString DesignerUtils::getSchemaFileFilter(bool any) {
    return DialogUtils::prepareFileFilter(tr("UGENE workflow schemes"), QStringList(SCHEMA_FILE_EXT), any);
}

QString DesignerUtils::getDropUrl(QList<DocumentFormat*>& fs, const QMimeData* md) {
    QString url;
    const GObjectMimeData* gomd = qobject_cast<const GObjectMimeData*>(md);
    const DocumentMimeData* domd = qobject_cast<const DocumentMimeData*>(md);
    if (gomd) {
        GObject* obj = gomd->objPtr.data();
        if (obj) {
            fs << obj->getDocument()->getDocumentFormat();
            url = obj->getDocument()->getURLString();
        }
    } else if (domd) {
        Document* doc = domd->objPtr.data();
        if (doc) {
            fs << doc->getDocumentFormat();
            url = doc->getURLString();
        }
    } else if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            url = urls.at(0).toLocalFile();
            fs += DocumentFormatUtils::detectFormat(url);
        }
    }
    return url;
}

void DesignerUtils::setQObjectProperties(QObject &o , const QVariantMap & params) {
    QMapIterator<QString, QVariant> i(params);
    while (i.hasNext()) {
        i.next();
        //log.debug("set param " + i.key() + "="+i.value().toString());
        o.setProperty(i.key().toAscii(), i.value());
    }
}

QStringList DesignerUtils::expandToUrls(const QString& s) {
    QStringList urls = s.split(";");
    QStringList result;
    QRegExp wcard("[*?\\[\\]]");
    foreach(QString url, urls) 
    {
        int idx = url.indexOf(wcard);
        if (idx >= 0) {
            int dirIdx = url.lastIndexOf('/', idx);
            QDir dir;
            if (dirIdx >= 0) {
                dir = QDir(url.left(dirIdx));
                url = url.right(url.length() - dirIdx - 1);
            }

            foreach(QFileInfo fi, dir.entryInfoList((QStringList() << url), QDir::Files|QDir::NoSymLinks)) {
                result << fi.absoluteFilePath();
            }
        } else {
            //if (QFile::exists(url)) 
            {
                result << url;
            }
        }
    }
    return result;
}

bool DesignerUtils::validate(const Schema& schema, QList<QListWidgetItem*>* infoList) {
    bool good = true;
    foreach (Actor* a, schema.procs) {
        foreach(Port* p, a->getPorts()) {
            QStringList l;
            bool ag = p->validate(l);
            good &= ag;
            if (infoList && !l.isEmpty()) {
                foreach(QString s, l) {
                    QListWidgetItem* item = new QListWidgetItem(a->getProto()->getIcon(), 
                        QString("%1 : %2").arg(a->getLabel()).arg(s));
                    item->setData(PORT_REF, p->getId());
                    item->setData(ACTOR_REF, a->getId());
                    infoList->append(item);
                }
            }
        }
    }

    foreach (const Iteration& it, schema.iterations) {
        Schema sh;
        QMap<ActorId, ActorId> map = SchemaSerializer::deepCopy(schema, &sh);
        sh.applyConfiguration(it, map);

        foreach (Actor* a, sh.procs) {
            QStringList l;
            bool ag = a->validate(l);
            good &= ag;
            if (infoList && !l.isEmpty()) {
                foreach(QString s, l) {
                    QListWidgetItem* item = new QListWidgetItem(a->getProto()->getIcon(), 
                        tr("Iteration '%3', %1 : %2").arg(a->getLabel()).arg(s).arg(it.name));
                    item->setData(ACTOR_REF, map.key(a->getId()));
                    item->setData(ITERATION_REF, it.id);
                    infoList->append(item);
                }
            }
        }
    }
    return good;
}

bool DesignerUtils::validate(const Schema& schema, QList<QMap<int, QVariant> >* infoList) {
    bool good = true;
    foreach (Actor* a, schema.procs) {
        foreach(Port* p, a->getPorts()) {
            QStringList l;
            bool ag = p->validate(l);
            good &= ag;
            if (infoList && !l.isEmpty()) {
                foreach(QString s, l) {
                    QMap<int, QVariant> item;
                    item[TEXT_REF] = QString("%1 : %2").arg(a->getLabel()).arg(s);
                    item[PORT_REF] = p->getId();
                    item[ACTOR_REF] = a->getId();
                    infoList->append(item);
                }
            }
        }
    }

    foreach (const Iteration& it, schema.iterations) {
        Schema sh;
        QMap<ActorId, ActorId> map = SchemaSerializer::deepCopy(schema, &sh);
        sh.applyConfiguration(it, map);

        foreach (Actor* a, sh.procs) {
            QStringList l;
            bool ag = a->validate(l);
            good &= ag;
            if (infoList && !l.isEmpty()) {
                foreach(QString s, l) {
                    QMap<int, QVariant> item;
                    item[TEXT_REF] = tr("Iteration '%3', %1 : %2").arg(a->getLabel()).arg(s).arg(it.name);
                    item[ACTOR_REF] = map.key(a->getId());
                    item[ITERATION_REF] = it.id;
                    infoList->append(item);
                }
            }
        }
    }
    return good;
}

QList<Descriptor> DesignerUtils::findMatchingTypes(DataTypePtr set, DataTypePtr elementDataType) {
    QList<Descriptor> result;
    foreach(const Descriptor& d, set->getAllDescriptors()) {
        if (set->getDatatypeByDescriptor(d) == elementDataType) {
            result.append(d);
        }
    }
    return result;
}

QStringList DesignerUtils::findMatchingTypesAsStringList(DataTypePtr set, DataTypePtr elementDatatype) {
    QList<Descriptor> descList = findMatchingTypes(set, elementDatatype);
    QStringList res;
    foreach( const Descriptor & desc, descList ) {
        res << desc.getId();
    }
    return res;
}

const Descriptor EMPTY_VALUES_DESC("", DesignerUtils::tr("<empty>"), DesignerUtils::tr("Default value"));

QList<Descriptor> DesignerUtils::findMatchingCandidates(DataTypePtr from, DataTypePtr elementDatatype) {
    QList<Descriptor> candidates = findMatchingTypes(from, elementDatatype);
    if (elementDatatype->isList()) {
        candidates += findMatchingTypes(from, elementDatatype->getDatatypeByDescriptor());
    } else {
        candidates.append(EMPTY_VALUES_DESC);
    }
    return candidates;
}

QList<Descriptor> DesignerUtils::findMatchingCandidates(DataTypePtr from, DataTypePtr to, const Descriptor & key) {
    return findMatchingCandidates(from, to->getDatatypeByDescriptor(key));
}

Descriptor DesignerUtils::getCurrentMatchingDescriptor(const QList<Descriptor> & candidates, DataTypePtr to, 
                                                       const Descriptor & key, const QStrStrMap & bindings) {
    DataTypePtr elementDatatype = to->getDatatypeByDescriptor(key);
    if (elementDatatype->isList()) {
        QString currentVal = bindings.value(key.getId());
        if (!currentVal.isEmpty()) {
            return Descriptor(currentVal, tr("<List of values>"), tr("List of values"));
        } else {
            return EMPTY_VALUES_DESC;
        }
    } else {
        int idx = bindings.contains(key.getId()) ? candidates.indexOf(bindings.value(key.getId())) : 0;
        return idx >= 0 ? candidates.at(idx) : candidates.first();
    }
}

DataTypePtr DesignerUtils::getToDatatypeForBusport(BusPort * p) {
    assert(p != NULL);
    DataTypePtr to;
    DataTypePtr t = to = p->getType();
    if (!t->isMap()) {
        QMap<Descriptor, DataTypePtr> map;
        map.insert(*p, t);
        to = new MapDataType(Descriptor(), map);
        //IntegralBusType* bt = new IntegralBusType(Descriptor(), QMap<Descriptor, DataTypePtr>());
        //bt->addOutput(t, p);
    }
    return to;
}

DataTypePtr DesignerUtils::getFromDatatypeForBusport(BusPort * p, DataTypePtr to) {
    assert(p != NULL);
    
    DataTypePtr from;
    if (p->isOutput() || p->getWidth() == 0) {
        //nothing to edit, go info mode
        from = to;
    } else {
        //port is input and has links, go editing mode
        IntegralBusType* bt = new IntegralBusType(Descriptor(), QMap<Descriptor, DataTypePtr>());
        bt->addInputs(p);
        from = bt;
    }
    return from;
}

/*****************************
* PrompterBaseImpl
*****************************/
QVariant PrompterBaseImpl::getParameter(const QString& id) {
    if (map.contains(id)) {
        return map.value(id);
    } else {
        return target->getParameter(id)->getAttributePureValue();
    }
}

QString PrompterBaseImpl::getURL(const QString& id, bool * empty ) {
    QString url = getParameter(id).toString();
    if( empty != NULL ) { *empty = false; }
    if( !target->getParameter(id)->getAttributeScript().isEmpty() ) {
        url = "got from user script";
    } else if (url.isEmpty()) {
        url = "<font color='red'>"+tr("unset")+"</font>";
        if( empty != NULL ) { *empty = true; }
    } else if (url.indexOf(";") != -1) {
        url = tr("the list of files");
    } else {
        url = QFileInfo(url).fileName();
    }
    return url;
}

QString PrompterBaseImpl::getRequiredParam(const QString& id) {
    QString url = getParameter(id).toString();
    if (url.isEmpty()) {
        url = "<font color='red'>"+tr("unset")+"</font>";
    }
    return url;
}

QString PrompterBaseImpl::getScreenedURL(BusPort* input, const QString& id, const QString& slot) {
    bool empty = false;
    QString attrUrl = QString("<u>%1</u>").arg(getURL(id, &empty));
    if( !empty ) {
        return attrUrl;
    }
    
    Actor * origin = input->getProducer( slot );
    QString slotUrl;
    if( origin != NULL ) {
        slotUrl = tr("file(s) alongside of input sources of <u>%1</u>").arg(origin->getLabel());
        return slotUrl;
    }
    
    assert( !attrUrl.isEmpty() );
    return attrUrl;
}

QString PrompterBaseImpl::getProducers( const QString& port, const QString& slot )
{
    BusPort* input = qobject_cast<BusPort*>(target->getPort(port));
    QList<Actor*> producers = input->getProducers(slot);

    QStringList labels;
    foreach(Actor* a, producers) {
        labels << a->getLabel();
    }
    return labels.join(", ");
}
}//namespace
