/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CircularView.h"

#include <core_api/DNAAlphabet.h>
#include <core_api/SelectionModel.h>
#include <core_api/Log.h>
#include <core_api/Timer.h>
#include <core_api/AppContext.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/AnnotationSettings.h>

#include <selection/DNASequenceSelection.h>
#include <util_gui/GScrollBar.h>
#include <util_text/FormatUtils.h>
#include <util_gui/GraphUtils.h>

#include <QtGui/QTextEdit>
#include <QtGui/QGridLayout>
#include <QtGui/QPainter>
#include <QtGui/QFontMetrics>
#include <QtGui/QDialog>

#include "CircularItems.h"

namespace GB2 {

static LogCategory log(ULOG_CAT_ADV);
const int CircularViewRenderArea::OUTER_ELLIPSE_SIZE = 512;
const int CircularViewRenderArea::ELLIPSE_DELTA = 22;
const int CircularViewRenderArea::INNER_ELLIPSE_SIZE = OUTER_ELLIPSE_SIZE - CV_REGION_ITEM_WIDTH;
const int CircularViewRenderArea::RULER_ELLIPSE_SIZE = INNER_ELLIPSE_SIZE - CV_REGION_ITEM_WIDTH;
const int CircularViewRenderArea::MIDDLE_ELLIPSE_SIZE = (INNER_ELLIPSE_SIZE + OUTER_ELLIPSE_SIZE) / 2;
const int CircularViewRenderArea::ARROW_LENGTH = 32;
const int CircularViewRenderArea::ARROW_HEIGHT_DELTA = 4;
const int CircularViewRenderArea::MAX_DISPLAYING_LABELS = 20;




#define MIN_OUTER_SIZE 100
CircularView::CircularView(QWidget* p, ADVSequenceObjectContext* ctx)
: GSequenceLineViewAnnotated(p, ctx), isCircularDNA(false)
{
    const DNASequence& dna = ctx->getSequenceObject()->getDNASequence();
    if (dna.info.contains(DNAInfo::LOCUS)) {
        DNALocusInfo loi = dna.info.value(DNAInfo::LOCUS).value<DNALocusInfo>();
        if (loi.topology == "circular") {
            isCircularDNA = true;
        }
    }
    
    foreach(AnnotationTableObject* obj, ctx->getAnnotationObjects()) {
        registerAnnotations(obj->getAnnotations());
    }

    renderArea = new CircularViewRenderArea(this);
    ra = qobject_cast<CircularViewRenderArea*>(renderArea);
    setMouseTracking(true);

    sBar = new QScrollBar(Qt::Vertical, renderArea);

    connect(ctx->getSequenceGObject(), SIGNAL(si_sequenceChanged()), this, SLOT(sl_sequenceChanged()));
    pack();
}

CircularView::~CircularView() {}

CircularViewRenderArea* CircularView::getRenderArea() const {
    return ra;
}

const QMap<Annotation*,CircularAnnotationItem*>& CircularView:: getCircularItems() const {
    return ra->circItems;
}

const QList<CircularAnnotationLabel*>& CircularView::getLabelList() const {
    return ra->labelList;
}

CircularRuler* CircularView::getRuler() const {
    return ra->ruler;
}

TextItem* CircularView::getSeqNameLabel() const {
    return ra->seqNameItem;
}

TextItem* CircularView::getSeqLenLabel() const {
    return ra->seqLenItem;
}

CircularSelectionItem* CircularView::getSelItem() const {
    return ra->selItem;
}

QSize CircularView::sizeHint() const {
    return QSize(600, 600);
}

void CircularView::pack() {
    setMinimumHeight(10);
    layout = new QVBoxLayout();
    layout->setContentsMargins(0,0,0,0);
    layout->addWidget(renderArea);
    setLayout(layout);
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Maximum);
}

void CircularView::updateVerticalScrollBar(int areaHeight) {
    int offset = areaHeight - height();
    sBar->disconnect(this);
    sBar->setMinimum(int(-offset/2));
    sBar->setMaximum(int(offset/2));
    sBar->setSingleStep(1);
    sBar->setPageStep(int(offset/2));
    sBar->setSliderPosition(0);

    connect(sBar, SIGNAL(valueChanged(int)), SLOT(sl_onSBarMoved(int)));
}

void CircularView::sl_onSBarMoved(int pos) {
    Q_UNUSED(pos)
    addUpdateFlags(GSLV_UF_NeedCompleteRedraw);
    renderArea->update();
}

void CircularView::mousePressEvent(QMouseEvent * e) {
    GSequenceLineViewAnnotated::mousePressEvent(e);
    CircularViewRenderArea* ra = qobject_cast<CircularViewRenderArea*>(renderArea);
    QPoint p = toRenderAreaPoint(e->pos());
    QPoint point(p.x() - width()/2 , p.y() - height()/2);
    qreal arcsin = coordToAngle(point);
    lastPressPos = 180 * 16 * arcsin / PI;
    lastPressPos-=ra->rotationDegree*16;
    if(lastPressPos<0) {
        lastPressPos+=360*16;
    }
    lastMovePos = lastPressPos;
    lastMouseY = point.y();
    currectSelectionLen = 0;
    QWidget::mousePressEvent(e);
}

void CircularView::mouseMoveEvent( QMouseEvent * e )
{
    QPoint areaPoint = toRenderAreaPoint(e->pos());
    QPoint point(areaPoint.x() - width()/2 , areaPoint.y() - height()/2);
    qreal arcsin = coordToAngle(point);
    CircularViewRenderArea* ra = qobject_cast<CircularViewRenderArea*>(renderArea);
    ra->mouseAngle = arcsin;

    if (e->buttons() & Qt::LeftButton) {        
        float a = 180 * 16 * arcsin / PI;
        a-=ra->rotationDegree*16;
        if(a<0) {
            a+=360*16;
        }
        //currectSelectionLen = a - lastPressPos;

        // compute selection
        int seqLen = ctx->getSequenceLen();
        int selStart = lastPressPos / 5760.0 * seqLen + 0.5f;
        int selEnd = a / 5760.0 * seqLen + 0.5f;
        int selLen = selEnd-selStart;
        //int selLen = currectSelectionLen / 5760.0 * seqLen;

        if (selLen < 0) {
            selStart = selStart + selLen;
            selLen *= -1;
        }
        if (selLen > seqLen) {
            selLen = seqLen;
        }
        if (selStart + selLen > seqLen) {
            assert(seqLen-selStart>=0);
            setSelection(LRegion(selStart, seqLen - selStart));
        } else {
            setSelection(LRegion(selStart, selLen));
        }
        lastMovePos = a;
        lastMouseY = point.y();
    }
    QWidget::mouseMoveEvent(e);
    renderArea->update();
}

void CircularView::mouseReleaseEvent(QMouseEvent* e) {
	GSequenceLineViewAnnotated::mouseReleaseEvent(e);
}

void CircularView::wheelEvent(QWheelEvent *we) {
    //check if mouse hovers
    bool renderAreaWheel = QRect(renderArea->x(), renderArea->y(),
        renderArea->width(), renderArea->height()).contains(we->pos());
    if (!renderAreaWheel) {
        QWidget::wheelEvent(we);
        return;
    }
    setFocus();
    CircularViewRenderArea* ra = qobject_cast<CircularViewRenderArea*>(renderArea);
    if(we->delta() > 0) {
        ra->rotationDegree+=10;
    }
    else {
        ra->rotationDegree-=10;
    }
    if(ra->rotationDegree<0) {
        ra->rotationDegree+=360.0;
    }
    if(ra->rotationDegree>360) {
        ra->rotationDegree-=360.0;
    }
    addUpdateFlags(GSLV_UF_NeedCompleteRedraw);
    renderArea->update();
    QWidget::wheelEvent(we);
}

void CircularView::sl_onAnnotationSelectionChanged(AnnotationSelection* selection, const QList<Annotation*>& added, const QList<Annotation*>& removed) {
    GSequenceLineViewAnnotated::sl_onAnnotationSelectionChanged(selection, added, removed);
    renderArea->update();
}

void CircularView::sl_onDNASelectionChanged( LRegionsSelection* thiz, const QList<LRegion>& added, const QList<LRegion>& removed )
{
    GSequenceLineViewAnnotated::sl_onDNASelectionChanged(thiz, added, removed);
    renderArea->update();
}

QList<AnnotationSelectionData> CircularView::selectAnnotationByCoord( const QPoint& coord ) const
{
    QList<AnnotationSelectionData> res;
    CircularViewRenderArea* renderArea = qobject_cast<CircularViewRenderArea*>(this->renderArea);
    QPoint cp(coord - QPoint(width()/2, height()/2));
    foreach(CircularAnnotationItem* item, renderArea->circItems) {
        int region = item->containsRegion(cp);
        if(region != -1) {
            res.append(AnnotationSelectionData(item->getAnnotation(), region));
            return res;
        }
    }
    foreach(CircularAnnotationItem* item, renderArea->circItems) {
        foreach(CircurlarAnnotationRegionItem* r, item->getRegions()) {
            CircularAnnotationLabel* lbl = r->getLabel();
            if(lbl->isVisible() && lbl->contains(cp)) {
                res.append(AnnotationSelectionData(item->getAnnotation(), item->getRegions().indexOf(r)));
                return res;
            }
        }
    }
    return res;
}

void CircularView::resizeEvent( QResizeEvent* e )
{
    CircularViewRenderArea* renderArea = qobject_cast<CircularViewRenderArea*>(this->renderArea);
    qreal s = qMin(qreal(height()/500.0), qreal(width()/500.0));
    int yLvl = renderArea->regionY.count();
    int outerEllipseSize=CircularViewRenderArea::OUTER_ELLIPSE_SIZE*s - renderArea->ellipseDelta*yLvl;

    if(outerEllipseSize<MIN_OUTER_SIZE) {
        sBar->setHidden(false);
        sBar->setFixedHeight(height());
        updateVerticalScrollBar(MIN_OUTER_SIZE + renderArea->ellipseDelta*yLvl);
    }
    else {
        //sBar->setSliderPosition(0);
        sBar->setHidden(true);
        renderArea->outerEllipseSize=outerEllipseSize;
        renderArea->innerEllipseSize=renderArea->outerEllipseSize-CV_REGION_ITEM_WIDTH;
        renderArea->rulerEllipseSize=renderArea->outerEllipseSize-CV_REGION_ITEM_WIDTH;
        renderArea->middleEllipseSize = (renderArea->outerEllipseSize + renderArea->innerEllipseSize)/2;
        renderArea->arrowLength=CircularViewRenderArea::ARROW_LENGTH*s;
        //renderArea->arrowHeightDelta=int(CircularViewRenderArea::ARROW_HEIGHT_DELTA*s);

        QFont font;
        QFontMetrics fm(font);
        int lblHeight = fm.height();
        renderArea->maxDisplayingLabels = int(height()/lblHeight);
    }
    addUpdateFlags(GSLV_UF_ViewResized);
    renderArea->update();
	QWidget::resizeEvent(e);
}

qreal CircularView::coordToAngle(const QPoint point) {
    float norm = sqrt((double)point.x()*point.x() + point.y()*point.y());
    float arcsin = asin(abs(point.y())/norm);
    if(point.x() < 0) {
        arcsin = PI - arcsin;
    }
    if(point.y() < 0) {
        arcsin = 2*PI - arcsin;
    }
    return arcsin;
}

/************************************************************************/
/* CircularViewRenderArea                                               */
/************************************************************************/
#define NOTCH_SIZE 5
CircularViewRenderArea::CircularViewRenderArea(CircularView* d)
: GSequenceLineViewAnnotatedRenderArea(d, true), outerEllipseSize(OUTER_ELLIPSE_SIZE),
ellipseDelta(ELLIPSE_DELTA), innerEllipseSize(INNER_ELLIPSE_SIZE), rulerEllipseSize(RULER_ELLIPSE_SIZE),
middleEllipseSize(MIDDLE_ELLIPSE_SIZE), arrowLength(ARROW_LENGTH),
arrowHeightDelta(ARROW_HEIGHT_DELTA), maxDisplayingLabels(MAX_DISPLAYING_LABELS),
circularView(d), rotationDegree(0), mouseAngle(0) {
    setMouseTracking(true);

    ADVSequenceObjectContext* ctx = view->getSequenceContext();

    int len = ctx->getSequenceLen();
    int start = 1;
    ruler = new CircularRuler(QPoint(0,0), rulerEllipseSize, rotationDegree, start, len, rulerFont, NOTCH_SIZE);

    //build annotation items to get number of region levels for proper resize
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    foreach(AnnotationTableObject* ao, ctx->getAnnotationObjects()) {
        foreach(Annotation* a, ao->getAnnotations()) {
            AnnotationSettings* as = asr->getAnnotationSettings(a->getAnnotationName());
            buildAnnotationItem(DrawAnnotationPass_DrawFill, a, false, as);
        }
    }
    //////////////////////////////////////////////////////////////////////////

    seqNameItem=new TextItem();
    seqLenItem=new TextItem();

    selItem = new CircularSelectionItem();
}

void CircularViewRenderArea::drawAll(QPaintDevice* pd) {
    QPainter p(pd);
    p.setRenderHint(QPainter::Antialiasing);
    GSLV_UpdateFlags uf = view->getUpdateFlags();
    bool completeRedraw = uf.testFlag(GSLV_UF_NeedCompleteRedraw) || uf.testFlag(GSLV_UF_ViewResized) || 
        uf.testFlag(GSLV_UF_AnnotationsChanged);

    int scrolPos = circularView->getScrollPos();
    if (completeRedraw) {
        QPainter pCached(cachedView);
        pCached.setRenderHint(QPainter::Antialiasing);
        pCached.fillRect(0, 0, pd->width(), pd->height(), Qt::white);
        /*pCached.translate(pd->width()/2, pd->height()/2);*/
        pCached.translate(parentWidget()->width()/2, parentWidget()->height()/2 - scrolPos);
        pCached.setPen(Qt::black);
        drawRuler(pCached);
        drawAnnotations(pCached);
        pCached.end();
    }
    p.drawPixmap(0, 0, *cachedView);
    /*p.translate(width()/2, height()/2);*/
    p.translate(parentWidget()->width()/2, parentWidget()->height()/2 - scrolPos);

    drawSequenceName(p);

    drawAnnotationsSelection(p);

    drawSequenceSelection(p);

    drawMarker(p);
}

void CircularViewRenderArea::drawAnnotationsSelection(QPainter& p) {
    ADVSequenceObjectContext* ctx = view->getSequenceContext();

    if(ctx->getAnnotationsSelection()->getSelection().isEmpty()) {
        return;
    }

    foreach(CircularAnnotationItem* item, circItems.values()) {
        item->setSelected(false);
    }

    /*foreach(const AnnotationSelectionData& asd, ctx->getAnnotationsSelection()->getSelection()) {
        CircularAnnotationItem* item = circItems[asd.annotation];
        item->setSelected(true);
        item->paint(&p, NULL, this);
        foreach(const CircurlarAnnotationRegionItem* r, item->getRegions()) {
            CircularAnnotationLabel* lbl = r->getLabel();
            if(lbl->isVisible()) {
                lbl->paint(&p, NULL, this);
            }
        }
    }*/
    foreach(const AnnotationSelectionData& asd, ctx->getAnnotationsSelection()->getSelection()) {
        AnnotationTableObject* o = asd.annotation->getGObject();
        if (ctx->getAnnotationObjects().contains(o)) {
            if(circItems.contains(asd.annotation)) {
                CircularAnnotationItem* item = circItems[asd.annotation];
                item->setSelected(true);
                item->paint(&p, NULL, this);
                foreach(const CircurlarAnnotationRegionItem* r, item->getRegions()) {
                    CircularAnnotationLabel* lbl = r->getLabel();
                    if(lbl->isVisible()) {
                        lbl->paint(&p, NULL, this);
                    }
                }
            }
        }
    }
}

#define RULER_PAD 40
void CircularViewRenderArea::drawSequenceName(QPainter& p) {
    QPen boldPen(Qt::black);
    boldPen.setWidth(3);
    ADVSequenceObjectContext* ctx = view->getSequenceContext();

    assert( ctx->getSequenceGObject() != NULL ); 

    //QString docName = ctx->getSequenceGObject()->getDocument()->getName();
    QString docName = ctx->getSequenceGObject()->getGObjectName();
    QString seqLen = QString::number(ctx->getSequenceLen()) + " bp";
    int docNameFullLength = docName.length();

    QFont font = p.font();
    QFontMetrics fm(font);
    int cw = fm.width('O');
    int symbolsAlowed = (rulerEllipseSize - RULER_PAD)/cw;
    if(symbolsAlowed<docNameFullLength) {
        docName=docName.mid(0,symbolsAlowed - 2);
        docName+="..";
    }
    
    p.setPen(boldPen);

    seqNameItem->setText(docName);
    seqNameItem->setPos(QPoint(0,0));
    seqNameItem->paint(&p, NULL, this);

    seqLenItem->setText(seqLen);
    seqLenItem->setPos(QPoint(0, seqNameItem->boundingRect().height()));
    seqLenItem->paint(&p, NULL, this);
}

void CircularViewRenderArea::drawSequenceSelection( QPainter& p )
{
    selItem->clear();
    
    ADVSequenceObjectContext* ctx = view->getSequenceContext();
    int seqLen = ctx->getSequenceLen();
    const QList<LRegion>& selection = view->getSequenceContext()->getSequenceSelection()->getSelectedRegions();

    foreach(const LRegion& r, selection) {
        QPainterPath* pathPtr = new QPainterPath();
        QPainterPath& path = *pathPtr;
        int yLevel = regionY.count() - 1;
        QRect outerRect(-outerEllipseSize/2 - yLevel * ellipseDelta/2 - ARROW_HEIGHT_DELTA,
            -outerEllipseSize/2 - yLevel * ellipseDelta/2 - ARROW_HEIGHT_DELTA,
            outerEllipseSize + yLevel * ellipseDelta + ARROW_HEIGHT_DELTA*2,
            outerEllipseSize + yLevel * ellipseDelta + ARROW_HEIGHT_DELTA*2);
        QRectF innerRect(-rulerEllipseSize/2 + NOTCH_SIZE, -rulerEllipseSize/2 + NOTCH_SIZE,
            rulerEllipseSize - 2*NOTCH_SIZE, rulerEllipseSize-2*NOTCH_SIZE);
        float startAngle = r.startPos / (float)seqLen * 360 + rotationDegree;
        float spanAngle = r.len / (float)seqLen * 360;
        path.moveTo(outerRect.width()/2 * cos(-startAngle / 180.0 * PI),
            -outerRect.width()/2 * sin(-startAngle / 180.0 * PI));
        path.arcTo(outerRect, -startAngle, -spanAngle);
        path.arcTo(innerRect, -startAngle-spanAngle, spanAngle);
        path.closeSubpath();
        selItem->addPath(path);
    }

    selItem->paint(&p, NULL, this);
}

void CircularViewRenderArea::drawRuler( QPainter& p )
{   
    ruler->setRotationDegree(rotationDegree);
    ruler->setRulerSize(rulerEllipseSize);
    ruler->paint(&p, NULL, this);
}

void CircularViewRenderArea::drawAnnotations(QPainter& p) {
    ADVSequenceObjectContext* ctx = view->getSequenceContext();

    QPen pen1(Qt::SolidLine);
    pen1.setWidth(1);
    foreach(CircularAnnotationItem* item, circItems) {
        delete item;
    }
    circItems.clear();
    labelList.clear();
    annotationYLevel.clear();
    regionY.clear();
    circItems.clear();

    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    //for(QSet<AnnotationTableObject*>::const_iterator i = ctx->getAnnotationObjects().begin(); i != ctx->getAnnotationGObjects().constEnd(); i++) {
    //TODO: there need const order of annotation tables
    foreach(AnnotationTableObject* ao, ctx->getAnnotationObjects()) {
        foreach(Annotation* a, ao->getAnnotations()) {
            AnnotationSettings* as = asr->getAnnotationSettings(a->getAnnotationName());
            buildAnnotationItem(DrawAnnotationPass_DrawFill, a, false, as);
        }
    }
    pen1.setWidth(2);
    foreach(AnnotationTableObject* ao, ctx->getAnnotationObjects()) {
        foreach(Annotation* a, ao->getAnnotations()) {
            AnnotationSettings* as = asr->getAnnotationSettings(a->getAnnotationName());
            buildAnnotationLabel(p.font(), a, as);
        }
    }

    CircularAnnotationLabel::prepareLabels(labelList);
    evaluateLabelPositions();

    foreach(CircularAnnotationItem* item, circItems) {
        item->paint(&p, NULL, this);
    }
    foreach(CircularAnnotationLabel* label, labelList) {
        label->setLabelPosition();
        label->paint(&p, NULL, this);
    }
}

#define REGION_MIN_LEN 3
void CircularViewRenderArea::buildAnnotationItem(DrawAnnotationPass pass, Annotation* a, bool selected /* = false */, const AnnotationSettings* as /* = NULL */) {
    if (!as->visible && (pass == DrawAnnotationPass_DrawFill || !selected)) {
        return;
    }

    ADVSequenceObjectContext* ctx = view->getSequenceContext();

    int seqLen = ctx->getSequenceLen();

    const QList<LRegion>& location = a->getLocation();

    LRegion generalLocation(location.first().startPos, location.last().startPos - location.first().startPos + location.last().len);

    int yLevel = 0;
    bool yFind = false;
    for(;yLevel<regionY.count();yLevel++) {
        bool intersects = false;
        foreach(const LRegion& r, regionY[yLevel]) {
            if(r.intersects(generalLocation)) {
                intersects = true;
                break;
            }
        }
        if(!intersects) {
            QList<LRegion>& rY = regionY[yLevel];
            rY.append(generalLocation);
            yFind = true;
            break;
        }
    }
    if(!yFind) { 
        QList<LRegion> newLevel;
        newLevel.append(generalLocation);
        regionY.append(newLevel);
    }
    annotationYLevel[a] = yLevel;

    QList<CircurlarAnnotationRegionItem*> regions;

    foreach(const LRegion& r, location) {
        float startAngle = (float)r.startPos / (float)seqLen * 360;
        float spanAngle = (float)r.len / (float)seqLen * 360;

        //cut annotation border if dna is linear
        if(!circularView->isCircularTopology()) {
            spanAngle = qMin(spanAngle, (float)(360-startAngle));
        }
        
        startAngle+=rotationDegree;

        QVector<Qualifier> quals;
        a->findQualifiers("SPLIT", quals);
        if(!quals.isEmpty()) {
            int totalLen = quals.at(0).getQualifierValue().toInt();
            if(totalLen>=0) {
                if(totalLen>seqLen) {
                    totalLen = seqLen;
                }
                spanAngle = (float)totalLen / (float)seqLen * 360;
            }
            else {
                return;
            }
        }
        
        QPainterPath path;
        QRect outerRect(-outerEllipseSize/2 - yLevel * ellipseDelta/2, -outerEllipseSize/2 - yLevel * ellipseDelta/2, outerEllipseSize + yLevel * ellipseDelta, outerEllipseSize + yLevel * ellipseDelta);
        QRect innerRect(-innerEllipseSize/2 - yLevel * ellipseDelta/2, -innerEllipseSize/2 - yLevel * ellipseDelta/2, innerEllipseSize + yLevel * ellipseDelta, innerEllipseSize + yLevel * ellipseDelta);
        QRect middleRect(-middleEllipseSize/2 - yLevel * ellipseDelta/2, -middleEllipseSize/2 - yLevel * ellipseDelta/2, middleEllipseSize + yLevel * ellipseDelta, middleEllipseSize + yLevel * ellipseDelta);
        arrowLength = qMin(arrowLength, ARROW_LENGTH);
        float dAlpha = 360 * arrowLength / (float)PI / (outerEllipseSize + innerEllipseSize + yLevel*ellipseDelta);
        bool isShort = r.len / (float)seqLen * 360 < dAlpha;

        float regionLen = spanAngle*PI/180 * outerRect.height()/2;
        if(regionLen < REGION_MIN_LEN) {
            spanAngle = (float)REGION_MIN_LEN / (PI*outerRect.height()) * 360;
        }
        
        if(isShort) {
            path.moveTo(outerRect.width()/2 * cos(-startAngle / 180.0 * PI),-outerRect.height()/2 * sin(-startAngle / 180.0 * PI));
            path.arcTo(outerRect, -startAngle, -spanAngle);
            path.arcTo(innerRect, -startAngle-spanAngle, spanAngle);
            path.closeSubpath();
        } else {
            if(a->isOnComplementStrand()) {
                path.moveTo(outerRect.width()/2 * cos((startAngle + dAlpha) / 180.0 * PI),
                    outerRect.height()/2 * sin((startAngle + dAlpha) / 180.0 * PI));
                path.lineTo((outerRect.width()/2 + arrowHeightDelta) * cos((startAngle + dAlpha) / 180.0 * PI),
                    (outerRect.width()/2 + arrowHeightDelta) * sin((startAngle + dAlpha) / 180.0 * PI));
                path.lineTo(middleRect.width()/2 * cos(startAngle / 180.0 * PI),
                    middleRect.height()/2 * sin(startAngle/ 180.0 * PI));
                path.lineTo((innerRect.width()/2 - arrowHeightDelta) * cos((startAngle + dAlpha) / 180.0 * PI),
                    (innerRect.width()/2 - arrowHeightDelta) * sin((startAngle + dAlpha) / 180.0 * PI));
                path.arcTo(innerRect, -(startAngle + dAlpha), -(spanAngle - dAlpha));
                path.arcTo(outerRect, -startAngle - spanAngle, spanAngle - dAlpha);
                path.closeSubpath();
            } else {
                path.moveTo(outerRect.width()/2 * cos(startAngle / 180.0 * PI),
                    outerRect.height()/2 * sin(startAngle / 180.0 * PI));
                path.arcTo(outerRect, -startAngle, -(spanAngle - dAlpha));
                path.lineTo((outerRect.width()/2 + arrowHeightDelta) * cos((startAngle + spanAngle - dAlpha) / 180.0 * PI),(outerRect.height()/2 + arrowHeightDelta) * sin((startAngle + spanAngle - dAlpha) / 180.0 * PI));
                path.lineTo(middleRect.width()/2 * cos((startAngle + spanAngle) / 180.0 * PI),middleRect.height()/2 * sin((startAngle + spanAngle)/ 180.0 * PI));
                path.lineTo((innerRect.width()/2 - arrowHeightDelta) * cos((-startAngle - (spanAngle - dAlpha)) / 180.0 * PI),(innerRect.height()/2 - arrowHeightDelta) * sin((startAngle + spanAngle - dAlpha) / 180.0 * PI));
                path.arcTo(innerRect, -startAngle - (spanAngle - dAlpha), spanAngle - dAlpha);
                path.closeSubpath();
            }
        }
        regions.append(new CircurlarAnnotationRegionItem(path, isShort, location.indexOf(r)));
    }

    CircularAnnotationItem* item = new CircularAnnotationItem(a, regions, this);
    circItems[a] = item;
}

void CircularViewRenderArea::buildAnnotationLabel(const QFont& font, Annotation* a, const AnnotationSettings* as) {
	
    if (!as->visible) {
        return;
    }

    if(!circItems.contains(a)) {
        return;
    }

    ADVSequenceObjectContext* ctx = view->getSequenceContext();

    int seqLen = ctx->getSequenceLen();
    const QList<LRegion>& location = a->getLocation();
    for(int r=0;r<location.count();r++) {
        CircularAnnotationLabel* label = new CircularAnnotationLabel(a, r, seqLen, font, this);
        labelList.append(label);
        CircurlarAnnotationRegionItem* ri = circItems[a]->getRegions()[r];
        ri->setLabel(label);
    }
}

LRegion CircularViewRenderArea::getAnnotationYRange(Annotation* a, const LRegion& r, const AnnotationSettings* as) const{
    Q_UNUSED(a);
    Q_UNUSED(r);
    Q_UNUSED(as);
    return LRegion(0,0);
}

void CircularViewRenderArea::resizeEvent( QResizeEvent *e ) {
    view->addUpdateFlags(GSLV_UF_ViewResized);
    QWidget::resizeEvent(e);
}

#define MARKER_LEN 30
#define ARR_LEN 4
#define ARR_WIDTH 10
void CircularViewRenderArea::drawMarker(QPainter& p) {
    int yLevel = regionY.count() - 1;
    QPen markerPen;
    markerPen.setWidth(1);
    markerPen.setColor(Qt::gray);
    p.setPen(markerPen);

    QPainterPath arr1, arr2;

    arr1.moveTo((rulerEllipseSize/2.0 - MARKER_LEN)*cos(mouseAngle),
        (rulerEllipseSize/2.0 - MARKER_LEN)*sin(mouseAngle));
    QPointF point11((rulerEllipseSize/2.0 - NOTCH_SIZE)*cos(mouseAngle),
        (rulerEllipseSize/2.0 - NOTCH_SIZE)*sin(mouseAngle));
    arr1.lineTo(point11);
    arr1.lineTo(point11 - QPointF(ARR_LEN*sin(mouseAngle) + ARR_WIDTH/2*cos(mouseAngle), 
        -ARR_LEN*cos(mouseAngle) + ARR_WIDTH/2*sin(mouseAngle)));
    arr1.moveTo(point11);
    arr1.lineTo(point11 + QPointF(ARR_LEN*sin(mouseAngle) - ARR_WIDTH/2*cos(mouseAngle), 
        -ARR_LEN*cos(mouseAngle) - ARR_WIDTH/2*sin(mouseAngle)));

    arr2.moveTo((outerEllipseSize/2 + yLevel * ellipseDelta/2 + MARKER_LEN)*cos(mouseAngle),
        (outerEllipseSize/2 + yLevel * ellipseDelta/2 + MARKER_LEN)*sin(mouseAngle));
    QPointF point21((outerEllipseSize/2 + yLevel * ellipseDelta/2 + ARROW_HEIGHT_DELTA)*cos(mouseAngle),
        (outerEllipseSize/2 + yLevel * ellipseDelta/2 + ARROW_HEIGHT_DELTA)*sin(mouseAngle));
    arr2.lineTo(point21);
    arr2.lineTo(point21 + QPointF(ARR_LEN*sin(mouseAngle) + ARR_WIDTH/2*cos(mouseAngle), 
        -ARR_LEN*cos(mouseAngle) + ARR_WIDTH/2*sin(mouseAngle)));
    arr2.moveTo(point21);
    arr2.lineTo(point21 + QPointF(-ARR_LEN*sin(mouseAngle) + ARR_WIDTH/2*cos(mouseAngle), 
        ARR_LEN*cos(mouseAngle) + ARR_WIDTH/2*sin(mouseAngle)));

    p.drawPath(arr1);
    p.drawPath(arr2);
}

#define LABEL_PAD 30
void CircularViewRenderArea::evaluateLabelPositions() {
    labelEmptyPositions.clear();
    QFont f;
    QFontMetrics fm(f);
    int labelHeight = fm.height();
    int lvlsNum = regionY.count();
    int outerRadius = outerEllipseSize/2 + (lvlsNum-1)*ellipseDelta/2;
    int cw = fm.width('O');

    int areaHeight = height();
    /*int minAreaHeight = outerEllipseSize + ellipseDelta*lvlsNum;
    areaHeight = qMax(areaHeight, minAreaHeight);*/

    for(int zPos=-areaHeight/2 + labelHeight; zPos<areaHeight/2 - labelHeight; zPos+=labelHeight) {
        int x = sqrt(float(outerRadius*outerRadius - zPos*zPos));
        if(width()/2-x>0) {
            QRect l_rect(-x - LABEL_PAD, zPos, width()/2-(x+LABEL_PAD), labelHeight);
            QRect r_rect(x + LABEL_PAD, zPos, width()/2-(x+LABEL_PAD), labelHeight);
            labelEmptyPositions << l_rect << r_rect;
        }
    }
    //inner points
    labelEmptyInnerPositions.clear();
    int innerRadius = rulerEllipseSize/2 - LABEL_PAD;
    for (int zPos=-innerRadius+labelHeight; zPos<-2*labelHeight; zPos+=labelHeight) {
        int x = sqrt(float(innerRadius*innerRadius - zPos*zPos));
        if(2*x>=cw) {
            QRect r_rect(x,zPos,2*x,labelHeight);
            QRect l_rect(-x,zPos,2*x,labelHeight);
            labelEmptyInnerPositions << r_rect << l_rect;
        }
    }
    for (int zPos=innerRadius; zPos>2*labelHeight; zPos-=labelHeight) {
        int x = sqrt(float(innerRadius*innerRadius - zPos*zPos));
        if(2*x>=cw) {
            QRect r_rect(x,zPos,2*x,labelHeight);
            QRect l_rect(-x,zPos,2*x,labelHeight);
            labelEmptyInnerPositions << r_rect << l_rect;
        }
    }
}

}//namespace
