/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <memory>

#include <QtNetwork/QTcpSocket>

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <distributed_computing/LocalTaskManager.h>
#include "DirectSocketScanner.h"
#include "DirectSocketUtils.h"
#include "DirectSocketRemoteTaskServer.h"

#define DIRECT_SOCKET_SERVER_LOG_CAT "Direct socket server"


namespace GB2 {

static LogCategory log( DIRECT_SOCKET_SERVER_LOG_CAT );

DirectSocketRemoteTaskServer::DirectSocketRemoteTaskServer()
: port( DIRECT_SOCKET_TCP_SERVER_DEFAULT_PORT ) {
    connect(&server, SIGNAL(newConnection()), SLOT(sl_newConnection()));
    
    udpSocket.bind( DIRECT_SOCKET_UDP_PORT );
    connect( &udpSocket, SIGNAL( readyRead() ), SLOT( sl_udpSocketReadyRead() ) );
    
    setEnabled(true);
}

DirectSocketRemoteTaskServer::~DirectSocketRemoteTaskServer()
{
    log.message( LogLevel_TRACE, tr( "Direct socket server destroyed" ) );
}

bool DirectSocketRemoteTaskServer::setEnabled( bool enable )
{
    if(enable)
    {
        bool ok = server.listen(QHostAddress::Any, port );
        if( !ok ) {
            log.trace( tr( "Direct socket server: cannot listen on %1 port" ).arg( QString::number( port ) ) );
            return false;
        }
        log.message( LogLevel_TRACE, tr( "Direct socket server enabled on %1 port" ).arg( QString::number( port ) ) );
    }
    else
    {
        server.close();
        log.message( LogLevel_TRACE, tr( "Direct socket server disabled" ) );
    }
    return true;
}

bool DirectSocketRemoteTaskServer::isEnabled() const
{
    return server.isListening();
}

int DirectSocketRemoteTaskServer::getPort() const {
    return port;
}

void DirectSocketRemoteTaskServer::changePort( int newPort ) {
    if( server.isListening() ) {
        server.close();
        server.listen( QHostAddress::Any, newPort );
    }
    port = newPort;
    
}

void DirectSocketRemoteTaskServer::sl_udpSocketReadyRead() {
    while( udpSocket.hasPendingDatagrams() ) {
        QByteArray data;
        QHostAddress senderAddr;
        quint16 senderPort = 0;
        data.resize( udpSocket.pendingDatagramSize() );
        int ret = udpSocket.readDatagram( data.data(), data.size(), &senderAddr, &senderPort );
        if( -1 == ret || !QString( data ).startsWith( DirectSocketScanner::DIRECT_SOCKET_SCANNER_MSG ) ) {
            continue;
        }
        assert( DirectSocketScanner::SCANNER_UDP_PORT == senderPort );
        
        data += " " + QString::number( port ).toAscii();
        udpSocket.writeDatagram( data, senderAddr, DirectSocketScanner::SCANNER_UDP_PORT );
    }
}

void DirectSocketRemoteTaskServer::sl_newConnection()
{
    log.message( LogLevel_TRACE, tr( "Direct socket server: connection established" ) );
    
    QTcpSocket * socket = server.nextPendingConnection();
    QVariantList result;
    RemoteTaskError error(false, "unknown error");
    result << error.serialize();
    result << QVariant();
    
    connect( socket, SIGNAL( disconnected() ), socket, SLOT( deleteLater() ) );
    QVariant data;
    if( !DirectSocketUtils::readFromSocket( *socket, &data ) )
    {
        error = RemoteTaskError( false, "cannot read from socket" );
        result[0] = error.serialize();
        DirectSocketUtils::writeToSocket( *socket, result );
        return;
    }
    
    if(data.canConvert(QVariant::List))
    {
        QVariantList args = data.toList();
        QString methodName = args[0].toString();
        
        log.trace( tr( "Direct socket server: %1 request" ).arg( methodName ) );
        
        if("runRemoteTask" == methodName)
        {
            qint64 taskId = -1;
            error = AppContext::getLocalTaskManager()->
                        runTask(args[1].toString(), args[2], &taskId);
            result[1] = taskId;
        }
        else if("cancelRemoteTask" == methodName)
        {
            bool convertOk = false;
            qint64 taskId = args[1].toLongLong(&convertOk);
            if(convertOk)
            {
                error = AppContext::getLocalTaskManager()->cancelTask(taskId);
            }
            else
            {
                error = RemoteTaskError(false, "invalid request");
            }
        }
        else if("deleteRemoteTask" == methodName)
        {
            bool convertOk = false;
            qint64 taskId = args[1].toLongLong(&convertOk);
            if(convertOk)
            {
                error = AppContext::getLocalTaskManager()->deleteTask(taskId);
            }
            else
            {
                error = RemoteTaskError(false, "invalid request");
            }
        }
        else if("getRemoteTaskCancelFlag" == methodName)
        {
            bool convertOk = false;
            qint64 taskId = args[1].toLongLong(&convertOk);
            if(convertOk)
            {
                bool cancelFlag = false;
                error = AppContext::getLocalTaskManager()->
                            getTaskCancelFlag(taskId, &cancelFlag);
                result[1] = cancelFlag;
            }
            else
            {
                error = RemoteTaskError(false, "invalid request");
            }
        }
        else if("getRemoteTaskState" == methodName)
        {
            bool convertOk = false;
            qint64 taskId = args[1].toLongLong(&convertOk);
            if(convertOk)
            {
                Task::State state = Task::State_Finished;
                error = AppContext::getLocalTaskManager()->
                            getTaskState(taskId, &state);
                result[1] = state;
            }
            else
            {
                error = RemoteTaskError(false, "invalid request");
            }
        }
        else if("getRemoteTaskProgress" == methodName)
        {
            bool convertOk = false;
            qint64 taskId = args[1].toLongLong(&convertOk);
            if(convertOk)
            {
                int progress = 0;
                error = AppContext::getLocalTaskManager()->
                            getTaskProgress(taskId, &progress);
                result[1] = progress;
            }
            else
            {
                error = RemoteTaskError(false, "invalid request");
            }
        }
        else if("getRemoteTaskResult" == methodName)
        {
            bool convertOk = false;
            qint64 taskId = args[1].toLongLong(&convertOk);
            if(convertOk)
            {
                error = AppContext::getLocalTaskManager()->
                            getTaskResult(taskId, &result[1]);
            }
            else
            {
                error = RemoteTaskError(false, "invalid request");
            }
        }
        else if( "getRemoteTaskError" == methodName )
        {
            bool convertOk = false;
            qint64 taskId = args[1].toLongLong( &convertOk );
            if( convertOk )
            {
                QString errMsg;
                error = AppContext::getLocalTaskManager()->getTaskError( taskId, &errMsg );
                result[1] = errMsg;
            }
            else
            {
                error = RemoteTaskError( false, "invalid request" );
            }
        }
        else if("getUuid" == methodName)
        {
            error = RemoteTaskError(true, "");
            result[1] = AppContext::getLocalTaskManager()->getUuid().toString();
        }
        else if( "getServicesList" == methodName )
        {
            error = RemoteTaskError( true, "" );
            result[1] = AppContext::getLocalTaskManager()->getServicesList();
        }
        else if( "getHostName" == methodName )
        {
            error = RemoteTaskError( true, "" );
            result[1] = AppContext::getLocalTaskManager()->getHostName();
        }
        else
        {
            error = RemoteTaskError(false, "invalid request");
        }
    }
    else
    {
        error = RemoteTaskError(false, "invalid request");
    }
    result[0] = error.serialize();
    DirectSocketUtils::writeToSocket( *socket, result );
}

} // namespace GB2
