/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ExportProjectViewItems.h"

#include "ExportTasks.h"
#include "ExportSequencesDialog.h"
#include "ExportSequences2MSADialog.h"
#include "ExportMSA2SequencesDialog.h"
#include "ExportChromatogramDialog.h"
#include "ExportUtils.h"

#include "ImportAnnotationsFromCSVDialog.h"
#include "ImportAnnotationsFromCSVTask.h"

#include <core_api/AppContext.h>
#include <core_api/MainWindow.h>
#include <core_api/ProjectView.h>
#include <core_api/SelectionModel.h>
#include <core_api/L10n.h>
#include <core_api/GUrlUtils.h>
#include <core_api/DocumentUtils.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/MAlignmentObject.h>
#include <gobjects/DNAChromatogramObject.h>

#include <selection/DocumentSelection.h>
#include <selection/GObjectSelection.h>
#include <selection/SelectionUtils.h>

#include <util_algorithm/MSAUtils.h>
#include <util_gui/GUIUtils.h>
#include <util_gui/DialogUtils.h>

#include <QtGui/QMessageBox>
#include <QtGui/QFileDialog>
#include <QtGui/QMainWindow>

namespace GB2 {

ExportProjectViewItemsContoller::ExportProjectViewItemsContoller(QObject* p) : QObject(p) {
    exportSequencesToSequenceFormatAction = new QAction(tr("Export sequences"), this);
    connect(exportSequencesToSequenceFormatAction, SIGNAL(triggered()), SLOT(sl_saveSequencesToSequenceFormat()));

    exportSequencesAsAlignmentAction = new QAction(tr("Export sequences as alignment"), this);
    connect(exportSequencesAsAlignmentAction, SIGNAL(triggered()), SLOT(sl_saveSequencesAsAlignment()));

    exportAlignmentAsSequencesAction = new QAction(tr("Export alignment to sequence format"), this);
    connect(exportAlignmentAsSequencesAction, SIGNAL(triggered()), SLOT(sl_saveAlignmentAsSequences()));

    importAnnotationsFromCSVAction = new QAction(tr("Import annotations from CSV file"), this);
    connect(importAnnotationsFromCSVAction, SIGNAL(triggered()), SLOT(sl_importAnnotationsFromCSV()));

    exportDNAChromatogramAction = new QAction(tr("Export chromatogram to SCF"), this);
    connect(exportDNAChromatogramAction, SIGNAL(triggered()), SLOT(sl_exportChromatogramToSCF()));

    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);
    connect(pv, SIGNAL(si_onDocTreePopupMenuRequested(QMenu&)), SLOT(sl_addToProjectViewMenu(QMenu&)));
}


void ExportProjectViewItemsContoller::sl_addToProjectViewMenu(QMenu& m) {
    addExportMenu(m);
    addImportMenu(m);
}

#define ACTION_PROJECT__EXPORT_MENU "action_project__export_menu"
#define ACTION_PROJECT__IMPORT_MENU "action_project__import_menu"

void ExportProjectViewItemsContoller::addExportMenu(QMenu& m) {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);
    QMenu* sub = NULL;

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::SEQUENCE, &ms, UOF_LoadedOnly);
    if (!set.isEmpty()) {
        sub = new QMenu(tr("Export"));
        sub->addAction(exportSequencesToSequenceFormatAction);
        sub->addAction(exportSequencesAsAlignmentAction);
    } else {
        set = SelectionUtils::findObjects(GObjectTypes::MULTIPLE_ALIGNMENT, &ms, UOF_LoadedOnly);
        if (set.size() == 1) {
            sub = new QMenu(tr("Export"));
            sub->addAction(exportAlignmentAsSequencesAction);
        }
    } 
    
    set = SelectionUtils::findObjects(GObjectTypes::CHROMATOGRAM, &ms, UOF_LoadedOnly);
    if (set.size() == 1) {
        if (sub == NULL) {
            sub = new QMenu(tr("Export"));
        }
        sub->addAction(exportDNAChromatogramAction);
    }

    if (sub!=NULL) {
        sub->setObjectName(ACTION_PROJECT__EXPORT_MENU);
        QAction* beforeAction = GUIUtils::findActionAfter(m.actions(), ACTION_PROJECT__ADD_MENU);
        m.insertMenu(beforeAction, sub);
    }
}

void ExportProjectViewItemsContoller::addImportMenu(QMenu& m) {
    QMenu* importMenu = new QMenu(tr("Import"));
    importMenu->setObjectName(ACTION_PROJECT__IMPORT_MENU);
    importMenu->addAction(importAnnotationsFromCSVAction);
    QAction* beforeAction = GUIUtils::findActionAfter(m.actions(), ACTION_PROJECT__ADD_MENU);
    m.insertMenu(beforeAction, importMenu);
}

static bool hasComplementForAll(const QSet<GObject*>& set) {
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        if (o == NULL || GObjectUtils::findComplementTT(so) == NULL) {
            return false;
        } 
    }
    return true;
}

static bool hasAminoForAll(const QSet<GObject*>& set) {
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        if (o == NULL || GObjectUtils::findAminoTT(so, false, NULL) == NULL) {
            return false;
        } 
    }
    return true;
}

static bool hasNucleicForAll(const QSet<GObject*>& set) {
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        if (o == NULL || GObjectUtils::findBackTranslationTT(so) == NULL) {
            return false;
        } 
    }
    return true;
}


void ExportProjectViewItemsContoller::sl_saveSequencesToSequenceFormat() {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::SEQUENCE, &ms, UOF_LoadedOnly);
    if (set.isEmpty()) {
        QMessageBox::critical(NULL, L10N::errorTitle(), tr("No sequence objects selected!"));
        return;
    }
    bool allowMerge = set.size() > 1;
    bool allowComplement = hasComplementForAll(set);
    bool allowTranslate = hasAminoForAll(set);
    bool allowBackTranslate = hasNucleicForAll(set);

    
    ExportSequencesDialog d(allowMerge, allowComplement, allowTranslate, allowBackTranslate, QString(),  BaseDocumentFormats::PLAIN_FASTA, AppContext::getMainWindow()->getQMainWindow());
    d.setWindowTitle(exportSequencesToSequenceFormatAction->text());
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    ExportSequencesTaskSettings s;
    ExportUtils::loadDNAExportSettingsFromDlg(s,d);

    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        assert(so!=NULL);
        s.names.append(so->getGObjectName());
        s.alphabets.append(so->getAlphabet());
        s.sequences.append(so->getSequence());
        s.complTranslations.append(GObjectUtils::findComplementTT(so));
        s.aminoTranslations.append(d.translate ? GObjectUtils::findAminoTT(so, false, d.useSpecificTable ? d.translationTable : NULL) : NULL);
        s.nucleicTranslations.append(d.backTranslate ? GObjectUtils::findBackTranslationTT(so, d.translationTable) : NULL);
        //FIXME meta info is lost
    }

    Task* t = ExportUtils::wrapExportTask(new ExportSequencesTask(s), d.addToProject);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void ExportProjectViewItemsContoller::sl_saveSequencesAsAlignment() {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QList<GObject*> sequenceObjects = SelectionUtils::findObjectsKeepOrder(GObjectTypes::SEQUENCE, &ms, UOF_LoadedOnly);
    if (sequenceObjects.isEmpty()) {
        QMessageBox::critical(NULL, L10N::errorTitle(), tr("No sequence objects selected!"));
        return;
    }

    QString err;
    MAlignment ma = MSAUtils::seq2ma(sequenceObjects, err);
    if (!err.isEmpty()) {
        QMessageBox::critical(NULL, L10N::errorTitle(), err);
        return;
    }
    
    QString fileExt = AppContext::getDocumentFormatRegistry()->getFormatById(BaseDocumentFormats::CLUSTAL_ALN)->getSupportedDocumentFileExtensions().first();
    GUrl seqUrl = sequenceObjects.first()->getDocument()->getURL();
    GUrl defaultUrl = GUrlUtils::rollFileName(seqUrl.dirPath() + "/" + seqUrl.baseFileName() + "." + fileExt, DocumentUtils::getNewDocFileNameExcludesHint());
    
    ExportSequences2MSADialog d(AppContext::getMainWindow()->getQMainWindow(), defaultUrl.getURLString());
    d.setWindowTitle(exportSequencesAsAlignmentAction->text());
    int rc = d.exec();
    if (rc != QDialog::Accepted) {
        return;
    }
    Task* t = ExportUtils::wrapExportTask(new ExportAlignmentTask(ma, d.url, d.format), d.addToProjectFlag);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void ExportProjectViewItemsContoller::sl_saveAlignmentAsSequences() {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::MULTIPLE_ALIGNMENT, &ms, UOF_LoadedOnly);
    if (set.size()!=1) {
        QMessageBox::critical(NULL, L10N::errorTitle(), tr("Select one alignment object to export"));
        return;
    }
    GObject* obj = set.toList().first();
    MAlignment ma = qobject_cast<MAlignmentObject*>(obj)->getMAlignment();
    ExportMSA2SequencesDialog d(AppContext::getMainWindow()->getQMainWindow());
    d.setWindowTitle(exportAlignmentAsSequencesAction->text());
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    Task* t = ExportUtils::wrapExportTask(new ExportMSA2SequencesTask(ma, d.url, d.trimGapsFlag, d.format), d.addToProjectFlag);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}


void ExportProjectViewItemsContoller::sl_importAnnotationsFromCSV() {
    ImportAnnotationsFromCSVDialog d(AppContext::getMainWindow()->getQMainWindow());
    int rc = d.exec();
    if (rc != QDialog::Accepted) {
        return;
    }
	ImportAnnotationsFromCSVTaskConfig taskConfig;
	d.toTaskConfig(taskConfig);
    ImportAnnotationsFromCSVTask* task = new ImportAnnotationsFromCSVTask(taskConfig);
    AppContext::getTaskScheduler()->registerTopLevelTask(task);
}

void ExportProjectViewItemsContoller::sl_exportChromatogramToSCF()
{
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::CHROMATOGRAM, &ms, UOF_LoadedOnly);
    if (set.size() != 1 ) {
        QMessageBox::warning(NULL, L10N::errorTitle(), tr("Select one chromatogram object to export"));
        return;
    }
    GObject* obj = set.toList().first();
    DNAChromatogramObject* chromaObj = qobject_cast<DNAChromatogramObject*>(obj);
    assert(chromaObj != NULL);
    
    
    
    ExportChromatogramDialog d(QApplication::activeWindow(), chromaObj->getDocument()->getURL());
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
       
    ExportChromatogramTaskSettings settings;
    settings.url = d.url;
    settings.complement = d.complemented;
    settings.reverse = d.reversed;
    settings.loadDocument = d.addToProjectFlag;

    Task* task = ExportUtils::wrapExportTask(new ExportDNAChromatogramTask(chromaObj, settings), d.addToProjectFlag);
    AppContext::getTaskScheduler()->registerTopLevelTask(task);

}
} //namespace
