/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ExportSequenceViewItems.h"
#include "ExportUtils.h"
#include "ExportTasks.h"
#include "ExportSequencesDialog.h"
#include "ExportAnnotations2CSVDialog.h"
#include "ExportAnnotations2CSVTask.h"
#include "ExportSequences2MSADialog.h"

#include <core_api/AppContext.h>
#include <core_api/DNATranslation.h>
#include <core_api/DocumentFormats.h>
#include <core_api/GUrlUtils.h>
#include <core_api/DocumentUtils.h>
#include <core_api/L10n.h>

#include <gobjects/GObjectTypes.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/MAlignmentObject.h>

#include <selection/GObjectSelection.h>
#include <selection/DocumentSelection.h>
#include <selection/SelectionUtils.h>
#include <selection/AnnotationSelection.h>
#include <selection/DNASequenceSelection.h>

#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVConstants.h>


#include <util_gui/DialogUtils.h>
#include <util_gui/GUIUtils.h>
#include <util_text/TextUtils.h>

#include <QtGui/QMainWindow>
#include <QtGui/QMessageBox>

namespace GB2 {


//////////////////////////////////////////////////////////////////////////
// ExportSequenceViewItemsController

ExportSequenceViewItemsController::ExportSequenceViewItemsController(QObject* p) 
: GObjectViewWindowContext(p, ANNOTATED_DNA_VIEW_FACTORY_ID)
{
}


void ExportSequenceViewItemsController::initViewContext(GObjectView* v) {
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(v);
    ADVExportContext* vc= new ADVExportContext(av);
    addViewResource(av, vc);
}


void ExportSequenceViewItemsController::buildMenu(GObjectView* v, QMenu* m) {
    QList<QObject*> resources = viewResources.value(v);
    assert(resources.size() == 1);
    QObject* r = resources.first();
    ADVExportContext* vc = qobject_cast<ADVExportContext*>(r);
    assert(vc!=NULL);
    vc->buildMenu(m);
}


//////////////////////////////////////////////////////////////////////////
// ADV view context


ADVExportContext::ADVExportContext(AnnotatedDNAView* v) : view(v) {
    sequence2SequenceAction = new QAction(tr("Export selected sequence region"), this);
    connect(sequence2SequenceAction, SIGNAL(triggered()), SLOT(sl_saveSelectedSequences()));

    annotations2SequenceAction = new QAction(tr("Export sequence of selected annotations"), this);
    connect(annotations2SequenceAction, SIGNAL(triggered()), SLOT(sl_saveSelectedAnnotationsSequence()));

    annotations2CSVAction = new QAction(tr("Export annotations to CSV format"), this);
    connect(annotations2CSVAction, SIGNAL(triggered()), SLOT(sl_saveSelectedAnnotationsToCSV()));

    annotationsToAlignmentAction = new QAction(QIcon(":core/images/msa.png"), tr("Make an alignment of selected annotations"), this);
    connect(annotationsToAlignmentAction, SIGNAL(triggered()), SLOT(sl_saveSelectedAnnotationsToAlignment()));

    sequenceToAlignmentAction = new QAction(QIcon(":core/images/msa.png"), tr("Make an alignment of selected sequences"), this);
    connect(sequenceToAlignmentAction, SIGNAL(triggered()), SLOT(sl_saveSelectedSequenceToAlignment()));

    sequenceToAlignmentWithTranslationAction = new QAction(QIcon(":core/images/msa.png"), tr("Make an alignment of translated sequence"), this);
    connect(sequenceToAlignmentWithTranslationAction, SIGNAL(triggered()), SLOT(sl_saveSelectedSequenceToAlignmentWithTranslation()));


    connect(view->getAnnotationsSelection(), 
        SIGNAL(si_selectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>& )), 
        SLOT(sl_onAnnotationSelectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>&)));

    connect(view, SIGNAL(si_sequenceAdded(ADVSequenceObjectContext*)), SLOT(sl_onSequenceContextAdded(ADVSequenceObjectContext*)));
    connect(view, SIGNAL(si_sequenceRemoved(ADVSequenceObjectContext*)), SLOT(sl_onSequenceContextRemoved(ADVSequenceObjectContext*)));
    foreach(ADVSequenceObjectContext* sCtx, view->getSequenceContexts()) {
        sl_onSequenceContextAdded(sCtx);
    }

}

void ADVExportContext::sl_onSequenceContextAdded(ADVSequenceObjectContext* c) {
    connect(c->getSequenceSelection(), 
        SIGNAL(si_selectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)), 
        SLOT(sl_onSequenceSelectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)));

    updateActions();
}

void ADVExportContext::sl_onSequenceContextRemoved(ADVSequenceObjectContext* c) {
    c->disconnect(this);
    updateActions();
}

void ADVExportContext::sl_onAnnotationSelectionChanged(AnnotationSelection* , const QList<Annotation*>& , const QList<Annotation*>&) {
    updateActions();
}

void ADVExportContext::sl_onSequenceSelectionChanged(LRegionsSelection* , const QList<LRegion>& , const QList<LRegion>& ) {
    updateActions();
}


static bool allNucleic(const QList<ADVSequenceObjectContext*>& seqs) {
    foreach(const ADVSequenceObjectContext* s, seqs) {
        if (!s->getAlphabet()->isNucleic()) {
            return false;
        }
    }
    return true;
}

void ADVExportContext::updateActions() {
    bool hasSelectedAnnotations = !view->getAnnotationsSelection()->isEmpty();
    int nSequenceSelections = 0;
    foreach(ADVSequenceObjectContext* c, view->getSequenceContexts()) {
        nSequenceSelections += c->getSequenceSelection()->isEmpty() ? 0 : 1;
    }

    sequence2SequenceAction->setEnabled(nSequenceSelections>=1);
    annotations2SequenceAction->setEnabled(hasSelectedAnnotations);
    annotations2CSVAction->setEnabled(hasSelectedAnnotations);

    bool hasMultipleAnnotationsSelected = view->getAnnotationsSelection()->getSelection().size() > 1;
    annotationsToAlignmentAction->setEnabled(hasMultipleAnnotationsSelected);

    bool hasMultiSequenceSelection = nSequenceSelections > 1;
    sequenceToAlignmentAction->setEnabled(hasMultiSequenceSelection);

    bool _allNucleic = allNucleic(view->getSequenceContexts());
    sequenceToAlignmentWithTranslationAction->setEnabled(hasMultiSequenceSelection && _allNucleic);
}

void ADVExportContext::buildMenu(QMenu* m) {
    QMenu* sub = GUIUtils::findSubMenu(m, ADV_MENU_EXPORT);

    if (view->getSequenceContexts().size() > 1) {
        sub->addAction(sequenceToAlignmentAction);
    }
    if (sequenceToAlignmentWithTranslationAction->isEnabled()) {
        sub->addAction(sequenceToAlignmentWithTranslationAction);
    }
    sub->addAction(annotationsToAlignmentAction);

    sub->addAction(sequence2SequenceAction);
    sub->addAction(annotations2SequenceAction);
    sub->addAction(annotations2CSVAction);
}


void ADVExportContext::sl_saveSelectedAnnotationsSequence() {
    AnnotationSelection* as = view->getAnnotationsSelection();
    AnnotationGroupSelection* ags = view->getAnnotationsGroupSelection();

    QSet<Annotation*> annotations;

    const QList<AnnotationSelectionData>& aData = as->getSelection();
    foreach(const AnnotationSelectionData& ad, aData) {
        annotations.insert(ad.annotation);
    }

    const QList<AnnotationGroup*>& groups =  ags->getSelection();
    foreach(AnnotationGroup* g, groups) {
        g->findAllAnnotationsInGroupSubTree(annotations);
    }

    if (annotations.isEmpty()) {
        QMessageBox::warning(view->getWidget(), L10N::warningTitle(), tr("No annotations selected!"));
        return;
    }
    bool merge = true;
    QSetIterator<Annotation*> iter(annotations);
    const bool strand = iter.peekNext()->isOnComplementStrand();

    bool allowComplement = true;
    bool allowTranslation = true;
    bool allowBackTranslation = true;

    while (iter.hasNext()) {
        Annotation* curr = iter.next();
        if (curr->isOnComplementStrand() != strand) {
            merge = false;
        }
        ADVSequenceObjectContext* seqCtx = view->getSequenceContext(curr->getGObject());
        if (seqCtx == NULL) {
            continue;
        }
        DNASequenceObject* dnaSeq = seqCtx->getSequenceObject();
        if (GObjectUtils::findComplementTT(dnaSeq) == NULL) {
            allowComplement = false;
        }
        if (GObjectUtils::findAminoTT(dnaSeq, false) == NULL) {
            allowTranslation = false;
        }
        if (GObjectUtils::findBackTranslationTT(dnaSeq) == NULL) {
            allowBackTranslation = false;
        }
    }

    QString fileExt = AppContext::getDocumentFormatRegistry()->getFormatById(BaseDocumentFormats::PLAIN_FASTA)->getSupportedDocumentFileExtensions().first();
    GUrl seqUrl = view->getSequenceInFocus()->getSequenceGObject()->getDocument()->getURL();
    GUrl defaultUrl = GUrlUtils::rollFileName(seqUrl.dirPath() + "/" + seqUrl.baseFileName() + "_annotation." + fileExt, DocumentUtils::getNewDocFileNameExcludesHint());

    ExportSequencesDialog d(merge, allowComplement, allowTranslation, allowBackTranslation, defaultUrl.getURLString(), BaseDocumentFormats::PLAIN_FASTA, AppContext::getMainWindow()->getQMainWindow());
    d.setWindowTitle(annotations2SequenceAction->text());
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    QHash<QString, TriState> usedNames;
    foreach(Annotation* a, annotations) {
        const QString& name = a->getAnnotationName();
        usedNames[name] = usedNames.contains(name) ? TriState_Yes : TriState_No;
    }

    ExportSequencesTaskSettings s;
    ExportUtils::loadDNAExportSettingsFromDlg(s,d);

    QList<Annotation*> sortedAnnotations = annotations.toList();
    qStableSort(sortedAnnotations.begin(), sortedAnnotations.end(), strand ? annotationGreaterThanByRegion : annotationLessThanByRegion);

    foreach(Annotation* a, sortedAnnotations) {
        ADVSequenceObjectContext* seqCtx = view->getSequenceContext(a->getGObject());
        if (seqCtx == NULL) {
            continue;
        } 
        DNATranslation* complTT = a->isOnComplementStrand() ? seqCtx->getComplementTT() : NULL;
        DNATranslation* aminoTT = d.translate ? (d.useSpecificTable ? GObjectUtils::findAminoTT(seqCtx->getSequenceObject(), false, d.translationTable) : seqCtx->getAminoTT()) : NULL;
        DNATranslation* nucleicTT = d.backTranslate ? GObjectUtils::findBackTranslationTT(seqCtx->getSequenceObject(), d.translationTable) : NULL;
        LRegion seqReg(0, seqCtx->getSequenceLen());
        const QByteArray& sequence = seqCtx->getSequenceData();

        const QList<LRegion>& location = a->getLocation();
        bool multi = location.size() > 1;
        for (int i=0; i < location.size(); i++) {
            QString prefix = a->getAnnotationName();
            if (multi) {
                prefix+=QString(" part %1 of %2").arg(QString::number(i+1)).arg(QString::number(location.size()));
            }
            QString name = prefix;
            for (int j=0; usedNames.contains(name) && usedNames.value(name) == TriState_Yes; j++) {
                name = prefix + "|" + QString::number(j);
            }
            usedNames[name] = TriState_Yes;
            s.names.append(name);

            LRegion reg = location[i].intersect(seqReg);
            QByteArray partSeq = sequence.mid(reg.startPos, reg.len);//TODO: mid() creates a copy -> optimize!!
            if (complTT!=NULL) {
                complTT->translate(partSeq.data(), partSeq.length());
                TextUtils::reverse(partSeq.data(), partSeq.length()); //todo: do it not in the main thread, but in task!!
            }
            s.sequences.append(partSeq);
            s.alphabets.append(seqCtx->getAlphabet());
            s.complTranslations.append(complTT);
            s.aminoTranslations.append(aminoTT);
            s.nucleicTranslations.append(nucleicTT);
        }
    }
    assert(s.names.size() > 0);
    Task* t = ExportUtils::wrapExportTask(new ExportSequencesTask(s), d.addToProject);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void ADVExportContext::sl_saveSelectedSequences() {
    ADVSequenceObjectContext* seqCtx = view->getSequenceInFocus();
    DNASequenceSelection* sel  = NULL;
    if (seqCtx!=NULL) {
        //TODO: support multi-export..
        sel = seqCtx->getSequenceSelection();
    }
    if (sel == NULL || sel->isEmpty()) {
        QMessageBox::warning(view->getWidget(), L10N::warningTitle(), tr("No sequence regions selected!"));
        return;
    }

    const QList<LRegion>& regions =  sel->getSelectedRegions();
    bool merge = regions.size() > 1;
    bool complement = seqCtx->getComplementTT()!=NULL;
    bool amino = seqCtx->getAminoTT()!=NULL;
    bool nucleic = GObjectUtils::findBackTranslationTT(seqCtx->getSequenceObject())!=NULL;

    QString fileExt = AppContext::getDocumentFormatRegistry()->getFormatById(BaseDocumentFormats::PLAIN_FASTA)->getSupportedDocumentFileExtensions().first();
    GUrl seqUrl = seqCtx->getSequenceGObject()->getDocument()->getURL();
    GUrl defaultUrl = GUrlUtils::rollFileName(seqUrl.dirPath() + "/" + seqUrl.baseFileName() + "_region." + fileExt, DocumentUtils::getNewDocFileNameExcludesHint());

    ExportSequencesDialog d(merge, complement, amino, nucleic, defaultUrl.getURLString(), BaseDocumentFormats::PLAIN_FASTA, AppContext::getMainWindow()->getQMainWindow());
    d.setWindowTitle(sequence2SequenceAction->text());
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    const QByteArray& sequence = seqCtx->getSequenceData();
    DNAAlphabet* al = seqCtx->getAlphabet();

    ExportSequencesTaskSettings s;
    ExportUtils::loadDNAExportSettingsFromDlg(s,d);
    foreach(const LRegion& r, regions) {
        QString prefix = QString("region [%1 %2]").arg(QString::number(r.startPos+1)).arg(QString::number(r.endPos()));
        QString name = prefix;
        for (int i=0; s.names.contains(name); i++) {
            name = prefix + "|" + QString::number(i);
        }
        s.names.append(name);
        s.alphabets.append(al);
        s.sequences.append(QByteArray(sequence.constData() + r.startPos, r.len)); //todo: optimize to avoid copying!!
        s.complTranslations.append(seqCtx->getComplementTT());
        s.aminoTranslations.append(d.translate ? (d.useSpecificTable ? GObjectUtils::findAminoTT(seqCtx->getSequenceObject(), false, d.translationTable) : seqCtx->getAminoTT()) : NULL);
        s.nucleicTranslations.append(d.backTranslate ? GObjectUtils::findBackTranslationTT(seqCtx->getSequenceObject(), d.translationTable) : NULL);
    }
    Task* t = ExportUtils::wrapExportTask(new ExportSequencesTask(s), d.addToProject);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

static bool annotationLessThan(Annotation *first, Annotation *second) {
    QListIterator<AnnotationGroup *> firstIterator(first->getGroups());
    QListIterator<AnnotationGroup *> secondIterator(second->getGroups());
    while(firstIterator.hasNext() && secondIterator.hasNext()) {
        if (firstIterator.peekNext()->getGroupName() < secondIterator.peekNext()->getGroupName()) {
            return true;
        }
        if (firstIterator.peekNext()->getGroupName() > secondIterator.peekNext()->getGroupName()) {
            return false;
        }
        firstIterator.next();
        secondIterator.next();
    }
    if (secondIterator.hasNext()) {
        if(first->getAnnotationName() < secondIterator.peekNext()->getGroupName()) {
            return true;
        }
        if(first->getAnnotationName() > secondIterator.peekNext()->getGroupName()) {
            return false;
        }
        secondIterator.next();
    }
    if (firstIterator.hasNext()) {
        if(firstIterator.peekNext()->getGroupName() < second->getAnnotationName()) {
            return true;
        }
        if(firstIterator.peekNext()->getGroupName() > second->getAnnotationName()) {
            return false;
        }
        firstIterator.next();
    }
    if (secondIterator.hasNext()) {
        return true;
    }
    if (firstIterator.hasNext()) {
        return false;
    }
    return (first->getAnnotationName() < second->getAnnotationName());
}

void ADVExportContext::sl_saveSelectedAnnotationsToCSV() {
    QSet<Annotation *> annotationSet;
    AnnotationSelection* as = view->getAnnotationsSelection();
    foreach(const AnnotationSelectionData &data, as->getSelection()) {
        annotationSet.insert(data.annotation);
    }
    foreach(AnnotationGroup *group, view->getAnnotationsGroupSelection()->getSelection()) {
        group->findAllAnnotationsInGroupSubTree(annotationSet);
    }

    if (annotationSet.isEmpty()) {
        QMessageBox::warning(view->getWidget(), L10N::warningTitle(), tr("No annotations selected!"));
        return;
    }

    ExportAnnotations2CSVDialog d(AppContext::getMainWindow()->getQMainWindow());
    d.setWindowTitle(annotations2CSVAction->text());
    if (QDialog::Accepted != d.exec()) {
        return;
    }

    //TODO: lock documents or use shared-data objects
    QList<Annotation *> annotationList = annotationSet.toList();
    qStableSort(annotationList.begin(), annotationList.end(), annotationLessThan);

    ADVSequenceObjectContext *sequenceContext = view->getSequenceInFocus();
    AppContext::getTaskScheduler()->registerTopLevelTask(
        new ExportAnnotations2CSVTask(annotationList, sequenceContext->getSequenceData(),
        sequenceContext->getComplementTT(), d.getExportSequence(), d.getFileName()));
}


//////////////////////////////////////////////////////////////////////////
// alignment part


#define MAX_ALI_MODEL (10*1000*1000)

QString ADVExportContext::prepareMAFromAnnotations(MAlignment& ma) {
    assert(ma.isEmpty());
    const QList<AnnotationSelectionData>& selection = view->getAnnotationsSelection()->getSelection();
    if (selection.size() < 2) {
        return tr("At least 2 sequences required");        
    }
    // check that all sequences are present and have the same alphabets
    DNAAlphabet* al = NULL;
    DNATranslation* complTT = NULL;
    foreach(const AnnotationSelectionData& a, selection) {
        AnnotationTableObject* ao = a.annotation->getGObject();
        ADVSequenceObjectContext* seqCtx = view->getSequenceContext(ao);
        if (seqCtx == NULL) {
            return tr("No sequence object found");
        }
        if (al == NULL ) {
            al = seqCtx->getAlphabet();
            complTT = seqCtx->getComplementTT();
        } else {
            DNAAlphabet* al2 = seqCtx->getAlphabet();
            //BUG524: support alphabet reduction
            if (al->getType() != al2->getType()) {
                return tr("Different alphabets");                
            } else if (al != al2) {
                al = al->getMap().count(true) >= al2->getMap().count(true) ? al : al2;
            }
        }
    }

    int maxLen = 0;
    ma.setAlphabet(al);
    QSet<QString> names;
    //TODO: check if amino translation is needed
    foreach(const AnnotationSelectionData& a, selection) {
        QString rowName = ExportUtils::genUniqueName(names, a.annotation->getAnnotationName());
        AnnotationTableObject* ao = a.annotation->getGObject();
        ADVSequenceObjectContext* seqCtx = view->getSequenceContext(ao);
        const QByteArray& sequence = seqCtx->getSequenceData();

        maxLen = qMax(maxLen, a.getSelectedRegionsLen());
        if (maxLen * ma.getNumRows() > MAX_ALI_MODEL) {
            return tr("Alignment is too large");
        }

		QByteArray rowSequence;
        AnnotationSelection::getAnnotationSequence(rowSequence, a, MAlignment_GapChar, sequence, complTT, NULL);
		ma.addRow(MAlignmentRow(rowName, rowSequence));
        names.insert(rowName);
    }
    return "";
}

QString ADVExportContext::prepareMAFromSequences(MAlignment& ma, bool translate) {
    assert(ma.isEmpty());

    DNAAlphabet* al = translate ? AppContext::getDNAAlphabetRegistry()->findById(BaseDNAAlphabetIds::AMINO_DEFAULT) : NULL;

    //derive alphabet
    int nItems = 0;
    foreach(ADVSequenceObjectContext* c, view->getSequenceContexts()) {
        if (c->getSequenceSelection()->isEmpty()) {
            continue;
        }
        nItems++;
        DNAAlphabet* seqAl = c->getAlphabet();
        if (al == NULL) {
            al = seqAl;
        } else if (al != seqAl) {
            if (al->isNucleic() && seqAl->isAmino()) {
                translate = true;
                al = seqAl;
            } else if (al->isAmino() && seqAl->isNucleic()) {
                translate = true;
            } else {
                return tr("Can't derive alignment alphabet");
            }
        }
    }

    if (nItems < 2) { 
        return tr("At least 2 sequences required");        
    }

    //cache sequences
    QSet<QString> names;
    QList<MAlignmentRow> rows;
    int maxLen = 0;
    foreach(ADVSequenceObjectContext* c, view->getSequenceContexts()) {
        if (c->getSequenceSelection()->isEmpty()) {
            continue;
        }
        DNAAlphabet* seqAl = c->getAlphabet();
        LRegion r = c->getSequenceSelection()->getSelectedRegions().first();
        const QByteArray& seq = c->getSequenceData();
        maxLen = qMax(maxLen, r.len);
        if (maxLen * rows.size() > MAX_ALI_MODEL) {
            return tr("Alignment is too large");
        }
        QByteArray mid = seq.mid(r.startPos, r.len);
        if (translate && seqAl->isNucleic()) {
            DNATranslation* aminoTT = c->getAminoTT();
            if (aminoTT!=NULL) {
                int len = aminoTT->translate(mid.data(), mid.size());
                mid.resize(len);
            }
        }
        MAlignmentRow row(ExportUtils::genUniqueName(names, c->getSequenceGObject()->getGObjectName()), mid);
        names.insert(row.getName());
        rows.append(row);
    }

    ma.setAlphabet(al);
	foreach(const MAlignmentRow& row, rows) {
		ma.addRow(row);
	}
    return "";
}



void ADVExportContext::selectionToAlignment(const QString& title, bool annotations, bool translate) {
    MAlignment ma(MA_OBJECT_NAME);
    QString err = annotations ? prepareMAFromAnnotations(ma) : prepareMAFromSequences(ma, translate);
    if (!err.isEmpty()) {
        QMessageBox::critical(NULL, L10N::errorTitle(), err);
        return;
    }

    DocumentFormatConstraints c;
    c.addFlagToSupport(DocumentFormatFlag_SupportWriting);
    c.supportedObjectTypes += GObjectTypes::MULTIPLE_ALIGNMENT;

    ExportSequences2MSADialog d(view->getWidget());
    d.setWindowTitle(title);
    int rc = d.exec();
    if (rc != QDialog::Accepted) {
        return;
    }
    Task* t = ExportUtils::wrapExportTask(new ExportAlignmentTask(ma, d.url, d.format), d.addToProjectFlag);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}


void ADVExportContext::sl_saveSelectedAnnotationsToAlignment() {
    selectionToAlignment(annotationsToAlignmentAction->text(), true, false);   
}

void ADVExportContext::sl_saveSelectedSequenceToAlignment() {
    selectionToAlignment(sequenceToAlignmentAction->text(), false, false);   
}

void ADVExportContext::sl_saveSelectedSequenceToAlignmentWithTranslation() {
    selectionToAlignment(sequenceToAlignmentWithTranslationAction->text(), false, true);   
}

} //namespace
