/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ExportTasks.h"

#include <core_api/DNAAlphabet.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <core_api/AppContext.h>
#include <core_api/DNATranslation.h>
#include <document_format/DNATranslationImpl.h>
#include <document_format/SCFFormat.h>
#include <core_api/Counter.h>
#include <gobjects/DNAChromatogramObject.h>
#include <gobjects/GObjectRelationRoles.h>
#include <util_tasks/LoadDocumentTask.h>
#include <util_tasks/AddDocumentTask.h>
#include <util_algorithm/MSAUtils.h>
#include <util_text/TextUtils.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/MAlignmentObject.h>

namespace GB2 {

//////////////////////////////////////////////////////////////////////////
//ExportSequencesTask	

ExportSequencesTask::ExportSequencesTask(const ExportSequencesTaskSettings& s) 
: AbstractExportTask("", TaskFlag_None), config(s)
{
    GCOUNTER( cvar, tvar, "DNAExportSequenceTask" );
    setTaskName(tr("Export sequence to '%1'").arg(QFileInfo(s.fileName).fileName()));
    setVerboseLogMode(true);

    assert(config.names.size()!=0);
    assert(config.names.size() == config.sequences.size());
    assert(config.names.size() == config.alphabets.size());
}

void ExportSequencesTask::run() {
    //prepare complement if needed
    QStringList             namesPre;
    QList<DNAAlphabet*>     alphabetsPre; 
    QList<QByteArray>       sequencesPre;
    QList<DNATranslation*>  aminoTTPre;
    QList<DNATranslation*>  nucleicTTPre;
    
    DocumentFormatRegistry* r = AppContext::getDocumentFormatRegistry();
    DocumentFormat* f = r->getFormatById(config.formatId);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(config.fileName));
    doc.reset(f->createNewDocument(iof, config.fileName)); 

    for (int i=0; i < config.names.size(); i++) {
        if (config.strand == TriState_Yes || config.strand == TriState_Unknown) {
            namesPre.append(config.names[i]);
            alphabetsPre.append(config.alphabets[i]);
            sequencesPre.append(config.sequences[i]);
            aminoTTPre.append(config.aminoTranslations[i]);
            nucleicTTPre.append(config.nucleicTranslations[i]);
        } 
        if (config.strand == TriState_No || config.strand == TriState_Unknown) { 
            DNATranslation* complTT = config.complTranslations[i];
            QByteArray seq;
            if (complTT == NULL) {
                stateInfo.setError( tr("Complement translator not found") );
                return;
            }

            char* data = config.sequences[i].data();
            int len = config.sequences[i].length();
            complTT->translate(data, len);
            TextUtils::reverse(data, len);
            namesPre.append(config.names[i] + "|rev-compl"); //TODO: check if unique!
            alphabetsPre.append(config.alphabets[i]);
            sequencesPre.append(config.sequences[i]);
            aminoTTPre.append(config.aminoTranslations[i]);
            nucleicTTPre.append(config.nucleicTranslations[i]);
            config.sequences[i].clear(); //release resources
        }
    }

    //translate data if needed
    QStringList             names;
    QList<DNAAlphabet*>     alphabets; 
    QList<QByteArray>       sequences;
    for (int i=0; i < namesPre.size(); i++) {
        DNATranslation* aminoTT = aminoTTPre[i];
        DNATranslation* nucleicTT = nucleicTTPre[i];
        if (aminoTT != NULL) {
            for (int j=0, nStrands = config.allAminoStrands ? 3 : 1; j < nStrands; j++) {
                QByteArray seq = sequencesPre[i];
                int len = (seq.length() - j)/3;
                QByteArray res(len, '\0');
                if (res.isNull() && len != 0) {
                    stateInfo.setError( tr("Out of memory") );
                    return;
                }
                assert(aminoTT->isThree2One());
                aminoTT->translate(seq.constData() + j, seq.length() - j, res.data(), res.length());

                sequences.append(res);
                names.append(namesPre[i] + QString("|transl") + (nStrands == 1 ? QString("") : " " + QString::number(j))); //TODO: check if unique!
                alphabets.append(aminoTT->getDstAlphabet());
            }
        } else if (nucleicTT != NULL) {
            DNATranslation* nucleicTT = nucleicTTPre[i];
            if (nucleicTT != NULL) {
                QByteArray seq = sequencesPre[i];
                int len = seq.length() * 3;
                QByteArray res(len, '\0');
                if (res.isNull() && len != 0) {
                    stateInfo.setError( tr("Out of memory") );
                    return;
                }
                assert(nucleicTT->isOne2Three());
                DNATranslation1to3Impl* trans = (DNATranslation1to3Impl*)nucleicTT;
                trans->translate(seq.constData(), seq.length(), res.data(), res.length(), config.mostProbable ? USE_MOST_PROBABLE_CODONS : USE_FREQUENCE_DISTRIBUTION);

                sequences.append(res);
                names.append(namesPre[i] + QString("|revtransl")); //TODO: check if unique!
                alphabets.append(nucleicTT->getDstAlphabet());
            }
        } else {
            names.append(namesPre[i]);
            alphabets.append(alphabetsPre[i]);
            sequences.append(sequencesPre[i]);
        }
        sequencesPre[i].clear(); //release resources
    }

    //process merge property and construct the object
    int n = names.size();
    assert(n > 0);
    if (config.merge) {
        int size = 0;
        for (int i=0; i<n; i++) {
            const QByteArray& seq = sequences[i];
            size += seq.size();
            if (i != n-1) {
                size+=config.mergeGap;
            }
        }
        QByteArray mergedSequence;
        mergedSequence.reserve(size);
        DNAAlphabet* al = alphabets[0];
        const QString& name = names[0];
        QByteArray gapSequence(config.mergeGap, al->getDefaultSymbol());
        for (int i=0; i < n; i++) {
            mergedSequence+=sequences[i];
            sequences[i].clear(); //release resource
            if (i != n-1) {
                mergedSequence+=gapSequence;
            }
        }
        doc->addObject(new DNASequenceObject(name, DNASequence(mergedSequence, al)));
    } else {
        for (int i=0; i < names.size(); i++) {
            DNAAlphabet* al = alphabets[i];
            const QString& name = names[i];
            const QByteArray& seq = sequences[i];
            doc->addObject(new DNASequenceObject(name, DNASequence(seq, al)));
        }
    }
    //store the document
    f->storeDocument(doc.get(), stateInfo);
}

//////////////////////////////////////////////////////////////////////////
// AddDocumentAndOpenViewTask

AddDocumentAndOpenViewTask::AddDocumentAndOpenViewTask(AbstractExportTask* t) 
: Task("Export sequence to document", TaskFlags_NR_FOSCOE)
{
    exportTask = t;
	addSubTask(exportTask);
}


QList<Task*> AddDocumentAndOpenViewTask::onSubTaskFinished( Task* subTask ) {
    QList<Task*> subTasks;
    if (subTask == exportTask) {
        Document* doc = exportTask->getDocument();
        const GUrl& fullPath = doc->getURL();
        DocumentFormat* format = doc->getDocumentFormat();
        IOAdapterFactory * iof = doc->getIOAdapterFactory();
        Document* clonedDoc = new Document(format, iof, fullPath);
        clonedDoc->loadFrom(doc); // doc was loaded in a separate thread -> clone all GObjects
        assert(!clonedDoc->isTreeItemModified());
        assert(clonedDoc->isLoaded());
        subTasks.append(new AddDocumentTask(clonedDoc));
        subTasks.append(new LoadUnloadedDocumentAndOpenViewTask(clonedDoc));           
    }
	//TODO: provide a report if subtask fails
    return subTasks;
}

//////////////////////////////////////////////////////////////////////////
// DNAExportAlignmentTask
ExportAlignmentTask::ExportAlignmentTask(const MAlignment& _ma, const QString& _fileName, DocumentFormatId _f)
: AbstractExportTask("", TaskFlag_None), ma(_ma), fileName(_fileName), format(_f)
{
    GCOUNTER( cvar, tvar, "ExportAlignmentTask" );
    setTaskName(tr("Export alignment to '%1'").arg(QFileInfo(fileName).fileName()));
    setVerboseLogMode(true);

    assert(!ma.isEmpty());
}

void ExportAlignmentTask::run() {
    DocumentFormatRegistry* r = AppContext::getDocumentFormatRegistry();
    DocumentFormat* f = r->getFormatById(format);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(fileName));
    doc.reset(f->createNewDocument(iof, fileName));
    doc->addObject(new MAlignmentObject(ma));
    f->storeDocument(doc.get(), stateInfo);
}


//////////////////////////////////////////////////////////////////////////
// export alignment  2 sequence format

ExportMSA2SequencesTask::ExportMSA2SequencesTask(const MAlignment& _ma, const QString& _url, bool _trimAli, DocumentFormatId _format) 
: AbstractExportTask(tr("Export alignment to sequence: %1").arg(_url), TaskFlag_None), 
ma(_ma), url(_url), trimAli(_trimAli), format(_format)
{
    GCOUNTER( cvar, tvar, "ExportMSA2SequencesTask" );
    setVerboseLogMode(true);
}

void ExportMSA2SequencesTask::run() {
    DocumentFormatRegistry* r = AppContext::getDocumentFormatRegistry();
    DocumentFormat* f = r->getFormatById(format);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
    doc.reset(f->createNewDocument(iof, url));

    QList<DNASequence> lst = MSAUtils::ma2seq(ma, trimAli);
    QSet<QString> usedNames;
    foreach(const DNASequence& s, lst) {
        QString name = s.getName();
        if (usedNames.contains(name)) {
            name = TextUtils::variate(name, " ", usedNames, false, 1);
        }
        doc->addObject(new DNASequenceObject(name, s));
        usedNames.insert(name);
    }
    f->storeDocument(doc.get(), stateInfo);
}

//////////////////////////////////////////////////////////////////////////
// export chromatogram to SCF

ExportDNAChromatogramTask::ExportDNAChromatogramTask( DNAChromatogramObject* _obj, const ExportChromatogramTaskSettings& _settings)
 : AbstractExportTask(tr("Export chromatogram to SCF"), TaskFlags_NR_FOSCOE), 
   cObj(_obj), settings(_settings), loadTask(NULL)
{
    GCOUNTER( cvar, tvar, "ExportDNAChromatogramTask" );
    setVerboseLogMode(true);
}

void ExportDNAChromatogramTask::prepare()
{
    Document* d = cObj->getDocument();
    assert(d != NULL);
    if (d == NULL ) {
        stateInfo.setError("Chromatogram object document is not found!");
        return;
    }

    QList<GObjectRelation> relatedObjs = cObj->findRelatedObjectsByRole(GObjectRelationRole::SEQUENCE);
    assert(relatedObjs.count() == 1);
    if (relatedObjs.count() != 1) {
        stateInfo.setError("Sequence related to chromatogram is not found!");
    }
    QString seqObjName = relatedObjs.first().ref.objName;

    GObject* resObj = d->findGObjectByName(seqObjName);
    DNASequenceObject * sObj = qobject_cast<DNASequenceObject*>(resObj);
    assert(sObj != NULL);

    DNAChromatogram cd = cObj->getChromatogram();
    DNASequence dna = sObj->getSequence();
    
    
    

    if (settings.reverse) {
        TextUtils::reverse(dna.seq.data(), dna.seq.length());
        reverseVector(cd.A);
        reverseVector(cd.C);
        reverseVector(cd.G);
        reverseVector(cd.T);
        int offset = 0;
        if (cObj->getDocument()->getDocumentFormatId() == BaseDocumentFormats::ABIF) {
            int baseNum = cd.baseCalls.count();
            int seqLen = cd.seqLength;
            // this is required for base <-> peak correspondence 
            if (baseNum > seqLen) {
                cd.baseCalls.remove(baseNum - 1);
                cd.prob_A.remove(baseNum - 1);
                cd.prob_C.remove(baseNum - 1);
                cd.prob_G.remove(baseNum - 1);
                cd.prob_T.remove(baseNum - 1);
            }
        } else if (cObj->getDocument()->getDocumentFormatId() == BaseDocumentFormats::SCF) {
            // SCF format particularities
            offset = -1;
        }

        for (int i = 0; i < cd.seqLength; ++i) {
            cd.baseCalls[i] = cd.traceLength - cd.baseCalls[i] + offset;
        } 
        reverseVector(cd.baseCalls);
        reverseVector(cd.prob_A);
        reverseVector(cd.prob_C);
        reverseVector(cd.prob_G);
        reverseVector(cd.prob_T);
    }

    if (settings.complement) {
        DNATranslation* tr = AppContext::getDNATranslationRegistry()->lookupTranslation(BaseDNATranslationIds::NUCL_DNA_DEFAULT_COMPLEMENT);
        tr->translate(dna.seq.data(), dna.length());
        qSwap(cd.A,cd.T);
        qSwap(cd.C, cd.G);
        qSwap(cd.prob_A, cd.prob_T);
        qSwap(cd.prob_C, cd.prob_G);
    
    }
    
    SCFFormat::exportDocumentToSCF(settings.url, cd, dna, stateInfo);
    if (stateInfo.hasErrors()) {
        return;
    }

    if (settings.loadDocument) {
        IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
        loadTask = new LoadDocumentTask(BaseDocumentFormats::SCF, settings.url, iof );
        addSubTask( loadTask );
    }
    
}

Document* ExportDNAChromatogramTask::getDocument() const
{
    Document* doc = loadTask->getDocument();
    return doc;
}

}//namespace

