/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <util_tasks/LoadDocumentTask.h>
#include <util_algorithm/FindAlgorithmTask.h>
#include <core_api/DocumentModel.h>
#include <core_api/AppContext.h>
#include <gobjects/DNASequenceObject.h>
#include <document_format/DocumentFormatUtils.h>
#include <util_algorithm/SArrayIndex.h>
#include <util_algorithm/SArrayBasedFindTask.h>
#include <util_text/TextUtils.h>
#include <core_api/Log.h>
#include <core_api/Counter.h>


#include "GenomeAlignerTask.h"

namespace GB2 {


static LogCategory log(ULOG_CAT_GA);

const QString GenomeAlignerTask::taskName(tr("UGENE genome aligner"));
const QString GenomeAlignerTask::OPTION_ALIGN_REVERSED("option_align_reversed");
const QString GenomeAlignerTask::OPTION_MISMATCHES("mismatches_allowed");
const QString GenomeAlignerTask::OPTION_ADD_REF_TO_ALIGNMENT("add_ref_to_alignment");


GenomeAlignerTask::GenomeAlignerTask( const DnaAssemblyToRefTaskSettings& settings )
: DnaAssemblyToReferenceTask(settings),loadRefTask(NULL),createIndexTask(NULL), refSeqObj(NULL), windowSize(0)
{
    GCOUNTER(cvar,tvar, "GenomeAlignerTask");  
}

void GenomeAlignerTask::prepare()
{
    assert(!settings.shortReads.isEmpty());
    if (settings.shortReads.isEmpty()) {
        setError(tr("Short reads list is empry!"));
        return;
    }
    result.setAlphabet(settings.shortReads.first().alphabet);
    QList<DocumentFormat*> detectedFormats = DocumentFormatUtils::detectFormat(settings.refSeqUrl);    
    if (!detectedFormats.isEmpty()) {
        IOAdapterFactory* factory = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
        DocumentFormat* format = detectedFormats.first();
        loadRefTask = new LoadDocumentTask(format->getFormatId(), settings.refSeqUrl, factory);
        addSubTask(loadRefTask);
    } else {
        setError(QString("Unknown format: %1").arg(settings.refSeqUrl.fileName()));
    }
    
    QVector<int> sizes;
    foreach ( const DNASequence& s, settings.shortReads) {
        sizes.append(s.length());
    }
    qSort(sizes);
    windowSize = sizes.first();

    shortReads += settings.shortReads;
    
    // add reversed short reads to alignment?
    bool alignReversed = settings.getCustomValue(OPTION_ALIGN_REVERSED, true).toBool();

    if (alignReversed) {
        foreach (const DNASequence& s, settings.shortReads) {
            QByteArray reversed(s.seq);
            TextUtils::reverse(reversed.data(), reversed.count());
            DNASequence rS(QString("%1 rev").arg(s.getName()), reversed, NULL);
            if (rS.seq != s.seq) {
                shortReads.append(rS);
            }
        }
    }
    log.trace(QString("Total number of reads to align: %1").arg(shortReads.count()));
 
}

QList<Task*> GenomeAlignerTask::onSubTaskFinished( Task* subTask )
{
    QList<Task*> subTasks;
    if (hasErrors()) {
        return subTasks;
    }
    if (subTask == loadRefTask) {
        Document* doc = loadRefTask->getDocument();
        QList<GObject*> seqObjects = doc->findGObjectByType(GObjectTypes::SEQUENCE);

        assert(seqObjects.count() == 1);
        if (seqObjects.count() == 0) {
            setError(QString("Ref sequence is not found in %1").arg(settings.refSeqUrl.fileName()) );
            return subTasks;
        }
        refSeqObj = qobject_cast<DNASequenceObject*>( seqObjects.first() );
        LRegion refRegion = refSeqObj->getSequenceRange();
        assert(refSeqObj != NULL);
		int nMismatches = settings.getCustomValue(GenomeAlignerTask::OPTION_MISMATCHES, 0).toInt();
        if (nMismatches > 0) {
            windowSize = windowSize / (nMismatches + 1);
        }
        createIndexTask = new CreateSArrayIndexTask(refSeqObj, windowSize);
        subTasks.append(createIndexTask);

    }  else if (subTask == createIndexTask) {
        
        assert(createIndexTask->getBitTable() == NULL);
        SArrayIndex* index = createIndexTask->index;
        foreach(const DNASequence& seq, shortReads) {
            SArrayBasedSearchSettings s;
            s.query = seq.seq;
            s.useBitMask = false;
            /*s.useBitMask = bitTable;
            s.bitMask = bitMask;
            s.bitMaskCharBitsNum = bitCharLen;
            */			
            s.nMismatches = settings.getCustomValue(GenomeAlignerTask::OPTION_MISMATCHES, 0).toInt();;
            s.unknownChar = createIndexTask->getUnknownChar();
            SArrayBasedFindTask* findTask = new SArrayBasedFindTask(index, s);   
            findTasks.append(findTask);           
            subTasks.append(findTask);
        }
    }

    return subTasks;
}

Task::ReportResult GenomeAlignerTask::report()
{
    if (hasErrors()) {
        return ReportResult_Finished;
    }
    
    int count = findTasks.count();
    for (int i = 0; i < count; ++i) {
         SArrayBasedFindTask* findTask = findTasks.at(i);
         QList<int> findResults = findTask->getResults();
         const DNASequence& seq = shortReads.at(i); 
         foreach (int offset, findResults) {
             alignedShortReads.insert(offset - 1,seq);
         }
     }
    
    if (alignedShortReads.isEmpty()) {
        setError("Reference assembly failed - no possible alignment found");
        return ReportResult_Finished;
    }
    
    result.setName(QString("Aligned %1").arg(refSeqObj->getGObjectName()));
    bool addRef = settings.getCustomValue(OPTION_ADD_REF_TO_ALIGNMENT, true).toBool();
    if (addRef) {
        result.addRow(MAlignmentRow(refSeqObj->getGObjectName(), refSeqObj->getSequence()));
    }
    foreach (int offset, alignedShortReads.keys()) {
        const DNASequence& shortRead = alignedShortReads.value(offset);
        result.addRow(MAlignmentRow(shortRead.getName(), shortRead.seq, offset));
    }
    return ReportResult_Finished;

}

void GenomeAlignerTask::run()
{
    
}




} // GB2
