/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QDebug>

#include <core_api/Log.h>
#include <workflow_support/WorkflowUtils.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow/IntegralBusType.h>
#include <workflow/Datatype.h>

#include "WorkflowEditorDelegates.h"
#include "WorkflowEditor.h"
#include "ActorCfgModel.h"

namespace GB2 {

static LogCategory log(ULOG_CAT_WD);

/*****************************
 * ActorCfgModel
 *****************************/
static const int KEY_COLUMN = 0;
static const int VALUE_COLUMN = 1;
static const int SCRIPT_COLUMN = 2;

ActorCfgModel::ActorCfgModel(QObject *parent, QList<Iteration>& lst)
: QAbstractTableModel(parent), subject(NULL), iterations(lst), iterationIdx(-1) {
    scriptDelegate = new AttributeScriptDelegate();
}

ActorCfgModel::~ActorCfgModel() {
    delete scriptDelegate;
}

void ActorCfgModel::setActor(Actor* cfg) {
    listValues.clear();
    attrs.clear();
    inputPortsData.reset();
    subject = cfg;
    if (cfg) {
        attrs = cfg->getParameters().values();
        inputPortsData.setData(cfg->getInputPorts());
        setupAttributesScripts();
    }
    reset();
}

static void filterEmptyDescriptors(QList<Descriptor> & descriptors) {
    foreach( const Descriptor & d, descriptors ) {
        if(d.getId().isEmpty()) {
            descriptors.removeAll(d);
        }
    }
}

void dumpDescriptors(const QList<Descriptor> & descriptors) {
    foreach( const Descriptor & d, descriptors ) {
        qDebug() << d.getId() << d.getDisplayName();
    }
}

void ActorCfgModel::setupAttributesScripts() {
    foreach( Attribute * attribute, attrs ) {
        assert(attribute != NULL);
        attribute->getAttributeScript().clearScriptVars();
        
        DataTypePtr attributeType = attribute->getAttributeType();
        // FIXME: add support for all types in scripting
        if(attributeType != CoreDataTypes::STRING_TYPE() && attributeType != CoreDataTypes::NUM_TYPE()) {
            continue;
        }
        
        // find all data from ports with same type
        foreach( const InputPortData & portData, inputPortsData.data ) {
            // in case of no linked actors to this port
            if( portData.from == portData.to ) {
                continue;
            }
            
            QList<Descriptor> matchingTypes = DesignerUtils::findMatchingCandidates(portData.from, attributeType);
            filterEmptyDescriptors(matchingTypes);
            /*foreach( const Descriptor & desc, matchingTypes ) {
                QString descId = desc.getId();
                ActorId actorId = IntegralBusType::parseSlotDesc(descId);
                QString attrId = IntegralBusType::parseAttributeIdFromSlotDesc(descId);
                Actor * a = portData.port->getLinkedActorById(actorId);
                assert(a != NULL);
                QString attrVarName = a->getPortAttributeShortName(attrId);
                attribute->getAttributeScript().
                    setScriptVar(Descriptor(attrVarName, desc.getDisplayName(), desc.getDocumentation()), QVariant());
            }*/
            QString attrVarName = attribute->getDisplayName();
            attribute->getAttributeScript().
                setScriptVar(Descriptor(attrVarName, attrVarName, attribute->getDocumentation()), QVariant());
        }
    }
}

void ActorCfgModel::selectIteration(int i) {
    listValues.clear();
    iterationIdx = i;
    reset();
}

void ActorCfgModel::setIterations(QList<Iteration>& lst) {
    iterations = lst;
    reset();
}

int ActorCfgModel::columnCount(const QModelIndex &) const { 
    return 3; // key, value and script
}

int ActorCfgModel::rowCount( const QModelIndex & parent ) const {
    if(parent.isValid()) {
        return 0;
    }
    return attrs.isEmpty() || parent.isValid() ? 0 : attrs.size() + inputPortsData.size();
}

Qt::ItemFlags ActorCfgModel::flags( const QModelIndex & index ) const {
    int col = index.column();
    int row = index.row();
    switch(col) {
    case KEY_COLUMN:
        return Qt::ItemIsEnabled;
    case VALUE_COLUMN:
        return row < attrs.size() ? Qt::ItemIsEditable | Qt::ItemIsEnabled | Qt::ItemIsSelectable : Qt::ItemIsEnabled;
    case SCRIPT_COLUMN:
        {
            if(row < attrs.size()) {
                Attribute * currentAttribute = attrs.at(row);
                assert(currentAttribute != NULL);
                // FIXME: add support for all types in scripting
                if(currentAttribute->getAttributeType() != CoreDataTypes::STRING_TYPE()) {
                    return Qt::ItemIsEnabled;    
                } else {
                    return Qt::ItemIsEditable | Qt::ItemIsEnabled | Qt::ItemIsSelectable;
                }
            } else {
                return Qt::ItemIsEnabled;
            }
        }
    default:
        assert(false);
    }
    // unreachable code
    return Qt::NoItemFlags;
}

QVariant ActorCfgModel::headerData( int section, Qt::Orientation orientation, int role ) const {
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
        switch(section) {
        case KEY_COLUMN: 
            return WorkflowEditor::tr("Name");
        case VALUE_COLUMN: 
            return WorkflowEditor::tr("Value");
        case SCRIPT_COLUMN:
            return WorkflowEditor::tr("Script");
        default:
            assert(false);
        }
    }
    // unreachable code
    return QVariant();
}

bool ActorCfgModel::setAttributeValue( const Attribute * attr, QVariant & attrValue ) const{
    assert(attr != NULL);
    bool isDefaultVal = true;
    attrValue = attr->getAttributePureValue();
    if (iterationIdx >= 0) {
        int x = iterationIdx;
        if (x >= iterations.size()) {
            //FIXME: handle error
            x = 0;
        }
        const Iteration& it = iterations.at(x);
        if (it.cfg.contains(subject->getId())) {
            const QVariantMap& params = it.cfg[subject->getId()];
            if (params.contains(attr->getId())) {
                attrValue = params.value(attr->getId());
                isDefaultVal = false;
            }
        }
    }
    return isDefaultVal;
}

QVariant ActorCfgModel::dataForActorAttributes(const QModelIndex & index, int role ) const {
    const Attribute *currentAttribute = attrs.at(index.row());
    if (role == DescriptorRole) { // descriptor that will be shown in under editor. 'propDoc' in WorkflowEditor
        return qVariantFromValue<Descriptor>(*currentAttribute);
    }
    
    int col = index.column();
    switch(col) {
    case KEY_COLUMN:
        {
            switch (role) {
            case Qt::DisplayRole: 
                return currentAttribute->getDisplayName();
            case Qt::ToolTipRole: 
                return currentAttribute->getDocumentation();
            case Qt::FontRole:
                if (currentAttribute->isRequiredAttribute()) {
                    QFont fnt; 
                    fnt.setBold(true);
                    return QVariant(fnt);
                }
            default:
                return QVariant();
            }
        }
    case VALUE_COLUMN:
        {
            if (role == ConfigurationEditor::ItemListValueRole) {
                return listValues.value(currentAttribute->getId());
            }
            
            QVariant attributeValue;
            bool isDefaultVal = setAttributeValue(currentAttribute, attributeValue);
            ConfigurationEditor* confEditor = subject->getEditor();
            PropertyDelegate* propertyDelegate = confEditor ? confEditor->getDelegate(currentAttribute->getId()) : NULL;
            switch(role) {
            case Qt::DisplayRole: 
            case Qt::ToolTipRole:
                return propertyDelegate ? propertyDelegate->getDisplayValue(attributeValue) : attributeValue;
            case Qt::ForegroundRole:
                return isDefaultVal ? QVariant(QColor(Qt::gray)) : QVariant();
            case DelegateRole:
                return qVariantFromValue<PropertyDelegate*>(propertyDelegate);
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                return attributeValue;
            default:
                return QVariant();
            }
        }
    case SCRIPT_COLUMN:
        {
            // FIXME: add support for all types in scripting
            if(currentAttribute->getAttributeType() != CoreDataTypes::STRING_TYPE()) {
                if( role == Qt::DisplayRole || role == Qt::ToolTipRole ) {
                    return QVariant(tr("N/A"));
                } else {
                    return QVariant();
                }
            }
            
            // for STRING type
            switch(role) {
            case Qt::DisplayRole:
            case Qt::ToolTipRole:
                return scriptDelegate ? 
                    scriptDelegate->getDisplayValue(qVariantFromValue<AttributeScript>(currentAttribute->getAttributeScript())) 
                    : QVariant();
            case Qt::ForegroundRole:
                return currentAttribute->getAttributeScript().isEmpty() ? QVariant(QColor(Qt::gray)) : QVariant();
            case DelegateRole:
                assert(scriptDelegate != NULL);
                return qVariantFromValue<PropertyDelegate*>(scriptDelegate);
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                return qVariantFromValue<AttributeScript>(currentAttribute->getAttributeScript());
            default:
                return QVariant();
            }
        }
    default:
        assert(false);
    }
    // unreachable code
    return QVariant();
}

QVariant ActorCfgModel::dataForPortAttributes(const QModelIndex & index, int role ) const {
    Descriptor keyDesc;
    Descriptor valueDesc;
    inputPortsData.setKeyValDescriptors( keyDesc, valueDesc, index.row() - attrs.size() );
    
    if( role == DescriptorRole ) { //will be shown under editor
        return qVariantFromValue<Descriptor>(keyDesc);
    }
    
    int col = index.column();
    switch(col) {
    case KEY_COLUMN:
        {
            switch(role) {
            case Qt::DisplayRole: 
                return keyDesc.getDisplayName();
            case Qt::ToolTipRole: 
                return keyDesc.getDocumentation();
            case Qt::ForegroundRole:
                return QVariant(QColor(Qt::gray));
            case Qt::FontRole:  
            default:
                return QVariant();
            }
        }
    case VALUE_COLUMN:
        {
            switch(role) {
            case Qt::DisplayRole:
            case Qt::ToolTipRole:
                return qVariantFromValue<QString>(valueDesc.getDisplayName());
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                return qVariantFromValue<QString>(valueDesc.getDisplayName());
            case Qt::ForegroundRole:
                return QVariant(QColor(Qt::gray));
            default:
                return QVariant(); //TODO: Add here Qt::DelegateRole
            }
        }
    case SCRIPT_COLUMN:
        return QVariant();
    default:
        assert(false);
    }
    
    // unreachable code
    return QVariant();
}

QVariant ActorCfgModel::data( const QModelIndex & index, int role ) const {
    return index.row() < attrs.size() ? dataForActorAttributes(index, role) : dataForPortAttributes(index, role);
}

bool ActorCfgModel::setData( const QModelIndex & index, const QVariant & value, int role ) {
    int col = index.column();
    Attribute* editingAttribute = attrs[index.row()];
    assert(editingAttribute != NULL);
    
    switch(col) {
    case VALUE_COLUMN:
        {
            switch(role) {
            case ConfigurationEditor::ItemListValueRole: 
                {
                    listValues.insert(editingAttribute->getId(), value); 
                    return true;
                }
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                {
                    const QString& key = editingAttribute->getId();
                    if (iterationIdx >= 0) {
                        int x = iterationIdx;
                        if (x >= iterations.size()) {
                            //FIXME: handle error
                            x = 0;
                        }
                        QVariantMap& cfg = iterations[x].cfg[subject->getId()];
                        QVariant old = cfg.contains(key) ? cfg.value(key) : editingAttribute->getAttributePureValue();
                        if (old != value) {
                            cfg.insert(key, value);
                            emit dataChanged(index, index);
                            log.trace("committed property change");
                        }
                    } else {
                        if (editingAttribute->getAttributePureValue() != value) {
                            subject->setParameter(key, value);
                            emit dataChanged(index, index);
                            log.trace("committed property change");
                        }
                    }
                    return true;
                }
            default:
                return false;
            }
        }
    case SCRIPT_COLUMN:
        {
            switch(role) {
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                {
                    AttributeScript attrScript = value.value<AttributeScript>();
                    editingAttribute->getAttributeScript().setScriptText(attrScript.getScriptText());
                    emit dataChanged(index, index);
                    log.trace(QString("user script for '%1' attribute updated").arg(editingAttribute->getDisplayName()));
                    return true;
                }
            default:
                return false;
            }
        }
    default:
        assert(false);
    }
    
    // unreachable code
    return false;
}

/*****************************
 * InputPortsData
 *****************************/
const Descriptor NULL_DESCRIPTOR("", "", "");
const Descriptor EMPTY_VAL_DESCRIPTOR("", "<empty>", "");

void InputPortsData::setData( const QList<Port*>& inputPorts ) {
    foreach( Port * p, inputPorts ) {
        BusPort * port = qobject_cast<BusPort*>(p);
        assert( port != NULL );
        InputPortData portData;
        portData.port = port;
        portData.portDisplayName = QString("Data from port '%1':").arg(portData.port->getDisplayName());
        
        QStrStrMap bindingsMap = port->getParameter(BusPort::BUS_MAP)->getAttributeValue<QStrStrMap>();
        portData.to = DesignerUtils::getToDatatypeForBusport(port);
        portData.from = DesignerUtils::getFromDatatypeForBusport(port, portData.to);
        
        QList<Descriptor> keys = portData.to->getAllDescriptors();
        int keySz = keys.size();
        for( int i = 0; i < keySz; ++i ) {
            Descriptor key = keys.at(i);
            if(portData.to == portData.from) {
                portData.data << QPair<Descriptor, Descriptor>(key, EMPTY_VAL_DESCRIPTOR);
                continue;
            }
            QList<Descriptor> candidates = DesignerUtils::findMatchingCandidates(portData.from, portData.to, key);
            Descriptor currentValue = DesignerUtils::getCurrentMatchingDescriptor(candidates, portData.to, key, bindingsMap);
            portData.data << QPair<Descriptor, Descriptor>(key, currentValue);
        }
        data << portData;
    }
}

void InputPortsData::reset() {
    data.clear();
}

int InputPortsData::size() const {
    int size = 0;
    foreach( const InputPortData & portData, data ) {
        size += portData.data.size() + 1;
    }
    return size;
}

void InputPortsData::dump() const {
    foreach( const InputPortData & pd, data ) {
        qDebug() << pd.port->getId();
        for(int i = 0; i < pd.data.size(); ++i) {
            const QPair<Descriptor, Descriptor> & pair = pd.data.at(i);
            qDebug() << pair.first.getDisplayName() << " " << pair.second.getDisplayName();
        }
    }
}

void InputPortsData::setKeyValDescriptors( Descriptor & key, Descriptor & val, int row ) const {
    int sz = size();
    assert( row >= 0 && row < sz );
    
    int portInd = 0;
    int szSum = 0;
    for(int i = 0; i < data.size(); ++i) {
        int tmp = szSum;
        szSum += data.at(i).data.size() + 1;
        if( row < szSum ) {
            portInd = i;
            row -= tmp;
            break;
        }
    }
    
    const InputPortData & portData = data.at(portInd);
    if( row == 0 ) {
        key = *(portData.port);
        val = NULL_DESCRIPTOR;
    } else {
        const QPair<Descriptor, Descriptor> & pair = portData.data.at(row - 1);
        key = pair.first;
        val = pair.second;
    }
}

} // GB2
