/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowEditor.h"
#include "WorkflowViewController.h"
#include "IterationListWidget.h"
#include "WorkflowEditorDelegates.h"

#include "ActorCfgModel.h"

#include <core_api/Settings.h>
#include <core_api/Log.h>
#include <workflow_support/WorkflowUtils.h>

#include <QtGui/QAction>
#include <QtGui/QHeaderView>

#define MAIN_SPLITTER "main.splitter"
#define TAB_SPLITTER "tab.splitter"

namespace GB2 {

static LogCategory log(ULOG_CAT_WD);

WorkflowEditor::WorkflowEditor(WorkflowView *p) 
: QWidget(p), owner(p), custom(NULL), customWidget(NULL), subject(NULL), actor(NULL)
{
    setupUi(this);

    caption->setMinimumHeight(nameEdit->sizeHint().height());

    iterationList = new IterationListWidget(this);
    iterationBox->layout()->addWidget(iterationList);

    actorModel = new ActorCfgModel(this, iterationList->list());
    table->setModel(actorModel);
    table->horizontalHeader()->setResizeMode(QHeaderView::Interactive);
    
    table->horizontalHeader()->setStretchLastSection(true);
    /*table->horizontalHeader()->setStretchLastSection(false);
    table->horizontalHeader()->setResizeMode(0, QHeaderView::ResizeToContents);
    table->horizontalHeader()->setResizeMode(1, QHeaderView::Interactive);
    table->horizontalHeader()->setResizeMode(2, QHeaderView::Stretch);*/
    
    table->horizontalHeader()->setClickable(false);
    table->verticalHeader()->hide();
    table->verticalHeader()->setDefaultSectionSize(QFontMetrics(QFont()).height() + 6);
    table->setItemDelegate(new SuperDelegate(this));

    reset();
    
    connect(iterationList, SIGNAL(iterationListAboutToChange()), SLOT(finishPropertyEditing()));
    connect(iterationList, SIGNAL(selectionChanged()), SLOT(updateIterationData()));
    connect(iterationList, SIGNAL(listChanged()), SLOT(commitIterations()));
    connect(iterationList, SIGNAL(selectionChanged()), SIGNAL(iterationSelected()));

    connect(nameEdit, SIGNAL(editingFinished()), SLOT(editingLabelFinished()));

    connect(table->selectionModel(), SIGNAL(currentChanged(QModelIndex, QModelIndex)), SLOT(sl_showPropDoc()));
    connect(table->model(), SIGNAL(dataChanged(QModelIndex, QModelIndex)), SLOT(handleDataChanged(QModelIndex, QModelIndex)));
}

void WorkflowEditor::handleDataChanged(const QModelIndex &topLeft, const QModelIndex &bottomRight) {
    if (topLeft == bottomRight) {
        commitIterations();
    }
}

void WorkflowEditor::selectIteration(int id) {
    iterationList->selectIteration(id);
}

Iteration WorkflowEditor::getCurrentIteration() const {
    return iterationList->list().at(iterationList->current());
}

void WorkflowEditor::updateIterationData() {
    if (sender()) {
        finishPropertyEditing();
    }
    actorModel->setIterations(iterationList->list());
    actorModel->selectIteration(iterationList->current());
}

void WorkflowEditor::sl_showPropDoc() {
    QModelIndex current = table->selectionModel()->currentIndex();
    if (current.isValid()) {
        propDoc->setText(DesignerUtils::getRichDoc(current.data(DescriptorRole).value<Descriptor>()));
    } else {
        propDoc->setText("");
    }
}

void WorkflowEditor::editingLabelFinished() {
    actor->setLabel(nameEdit->text());
}

void WorkflowEditor::reset() {
    caption->setText("");
    nameEdit->hide();
    setDescriptor(NULL);
    edit(NULL);
    actor = NULL;
    actorModel->setActor(NULL);
    propDoc->setText("");
}

void WorkflowEditor::resetIterations() {
    //disconnect(iterationList, SIGNAL(listChanged()), this, SLOT(commitIterations()));
    iterationList->setList(owner->getScene()->getIterations());
    //connect(iterationList, SIGNAL(listChanged()), SLOT(commitIterations()));
}

void WorkflowEditor::commitIterations() {
    log.trace("committing iterations data");
    owner->getScene()->setIterations(iterationList->list());
}

void WorkflowEditor::finishPropertyEditing() {
    //table->setCurrentCell(0,0, QItemSelectionModel::NoUpdate);
    table->setCurrentIndex(QModelIndex()/*table->model()->index(0, 0, QModelIndex())*/);
}

void WorkflowEditor::commit() {
    finishPropertyEditing();
    //commitIterations();
}

void WorkflowEditor::editActor(Actor* a) {
    reset();
    actor = a;
    if (a) {
        caption->setText(tr("Task name:"));
        nameEdit->setText(a->getLabel());
        nameEdit->show();
        setDescriptor(a->getProto(), tr("Task parameters can be configured in the \"Parameters\" widget suited below."));
        edit(a);
    }
}

void WorkflowEditor::editPort(Port* p) {
    reset();
    if (p) {
        //caption->setText(formatPortCaption(p));
        QString portDoc = tr("<b>%1 \"%2\"</b> of task \"%3\":<br>%4<br><br>%5")
            .arg(p->isOutput() ? tr("Output port") : tr("Input port"))
            .arg(p->getDisplayName())
            .arg(p->owner()->getLabel())
            .arg(p->getDocumentation())
            .arg(tr("You can observe data slots of the port and configure connections if any in the \"Parameters\" widget suited below."));
        doc->setText(portDoc);
        edit(p);
    }
}

void WorkflowEditor::setDescriptor(Descriptor* d, const QString& hint) {
    QString text = d ? DesignerUtils::getRichDoc(*d) + "<br><br>" + hint : hint;
    if (text.isEmpty()) {
        text = tr("Select any object to inspect.");
    }
    doc->setText(text);
}

void WorkflowEditor::edit(Configuration* cfg) {

    if (customWidget) {
        custom->commit();
        customWidget->hide();
        paramBox->layout()->removeWidget(customWidget);
    }

    subject = cfg;
    custom = cfg ? cfg->getEditor() : NULL;
    customWidget = custom ? custom->getWidget() : NULL;
    
    if (subject && !customWidget) {
        assert(actor);
        actorModel->setActor(actor);
        updateIterationData();
        tableSplitter->show();
    } else {
        tableSplitter->hide();
        if (customWidget) { 
            if (actor) assert(0 && "No support for iterations yet");
            paramBox->layout()->addWidget(customWidget);
        }
    }
}

QVariant WorkflowEditor::saveState() const {
    QVariantMap m;
    m.insert(MAIN_SPLITTER, splitter->saveState());
    m.insert(TAB_SPLITTER, tableSplitter->saveState());
    return m;
}

void WorkflowEditor::restoreState(const QVariant& v) {
    QVariantMap m = v.toMap();
    splitter->restoreState(m.value(MAIN_SPLITTER).toByteArray());
    tableSplitter->restoreState(m.value(TAB_SPLITTER).toByteArray());
}

}//namespace
