/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtCore/QMutexLocker>

#include <gobjects/DNASequenceObject.h>
#include <core_api/AppResources.h>
#include <core_api/DocumentModel.h>
#include <core_api/L10n.h>
#include <core_api/Counter.h>
#include <core_api/Log.h>
#include <gobjects/GObjectTypes.h>

#include <task_local_storage/uHMMSearchTaskLocalStorage.h>

#include "uhmm3phmmer.h"
#include "uhmm3PhmmerTask.h"

#define UHMM3_PHMMER_LOG_CAT "uhmm3_phmmer_log_category"

using namespace GB2;

namespace GB2 {

static LogCategory log( UHMM3_PHMMER_LOG_CAT );

static int countPhmmerMemInMB( qint64 dbLen, int queryLen ) {
    assert( 0 < dbLen && 0 < queryLen );
    return qMax( ( ( double )dbLen * queryLen / ( 1024 * 1024 ) ) * 10, 2.0 );
}

/**************************************
* General hmmer3 phmmer task.
**************************************/

UHMM3PhmmerTask::UHMM3PhmmerTask( const DNASequence & q, const DNASequence & d, const UHMM3PhmmerSettings & set )
: Task( tr( "HMM Phmmer task" ), TaskFlag_None ), query( q ), db( d ), settings( set ), loadQueryTask( NULL ), loadDbTask( NULL ) {
    GCOUNTER( cvar, tvar, "UHMM3PhmmerTask" );
    if( 0 == query.length() ) {
        stateInfo.setError( L10N::badArgument( tr( "query_sequence" ) ) );
        return;
    }
    if( 0 == db.length() ) {
        stateInfo.setError( L10N::badArgument( tr( "sequence_to_search_in" ) ) );
        return;
    }
    setTaskName( tr( "HMM Phmmer search %1 sequence in %2 database" ).arg( query.getName() ).arg( db.getName() ) );
    
    addMemResource();
}

UHMM3PhmmerTask::UHMM3PhmmerTask( const QString & queryFilename, const QString & dbFilename, const UHMM3PhmmerSettings & set ) 
: Task( tr( "HMM Phmmer task" ), TaskFlag_None ), settings( set ), loadQueryTask( NULL ), loadDbTask( NULL ) {
    if( queryFilename.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( tr( "query_sequence_filename" ) ) );
        return;
    }
    if( dbFilename.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( tr( "db_sequence_to_search_in" ) ) );
        return;
    }
    setTaskName( tr( "HMM Phmmer search %1 sequence with %2 database" ).arg( queryFilename ).arg( dbFilename ) );
    
    loadQueryTask = LoadDocumentTask::getDefaultLoadDocTask( queryFilename );
    if( NULL == loadQueryTask ) {
        stateInfo.setError( tr( "cannot_create_load_query_doc_task" ) );
        return;
    }
    addSubTask( loadQueryTask );
    loadDbTask = LoadDocumentTask::getDefaultLoadDocTask( dbFilename );
    if( NULL == loadDbTask ) {
        stateInfo.setError( tr( "cannot_create_load_db_doc_task" ) );
        return;
    }
    addSubTask( loadDbTask );
}

UHMM3PhmmerTask::UHMM3PhmmerTask( const QString & queryFilename, const DNASequence & d, const UHMM3PhmmerSettings & s )
: Task( tr( "HMM Phmmer task" ), TaskFlag_None ), db( d ), settings( s ), loadQueryTask( NULL ), loadDbTask( NULL ) {
    if( queryFilename.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( tr( "query_sequence_filename" ) ) );
        return;
    }
    if( 0 == db.length() ) {
        stateInfo.setError( L10N::badArgument( tr( "sequence_to_search_in" ) ) );
        return;
    }
    setTaskName( tr( "HMM Phmmer search %1 sequence in %2 database" ).arg( queryFilename ).arg( db.getName() ) );
    
    loadQueryTask = LoadDocumentTask::getDefaultLoadDocTask( queryFilename );
    if( NULL == loadQueryTask ) {
        stateInfo.setError( tr( "cannot_create_load_query_doc_task" ) );
        return;
    }
    addSubTask( loadQueryTask );
}

void UHMM3PhmmerTask::addMemResource() {
    assert( !db.isNull() && !query.isNull() );
    
    int howManyMem = countPhmmerMemInMB( db.length(), query.length() );
    taskResources.append( TaskResourceUsage( RESOURCE_MEMORY, howManyMem ) );
    log.trace( QString( "%1 needs %2 of memory" ).arg( getTaskName() ).arg( howManyMem ) );
}

DNASequence UHMM3PhmmerTask::getSequenceFromDocument( Document * doc ) {
    DNASequence ret;
    if( NULL == doc ) {
        stateInfo.setError( tr( "cannot load document from:" ) );
        return ret;
    }
    
    QList< GObject* > objsList = doc->findGObjectByType( GObjectTypes::SEQUENCE );
    if( objsList.isEmpty() ) {
        stateInfo.setError( tr( "no_dna_sequence_objects_in_document" ) );
        return ret;
    }
    DNASequenceObject* seqObj = qobject_cast< DNASequenceObject* >( objsList.first() );
    if( NULL == seqObj ) {
        stateInfo.setError( tr( "cannot_cast_to_dna_object" ) );
        return ret;
    }
    ret = seqObj->getDNASequence();
    if( !ret.length() ) {
        stateInfo.setError( tr( "empty_sequence_given" ) );
        return ret;
    }
    return ret;
}

QList< Task* > UHMM3PhmmerTask::onSubTaskFinished( Task* subTask ) {
    QMutexLocker locker( &loadTasksMtx );
    QList< Task* > ret;
    assert( NULL != subTask );
    if( hasErrors() ) {
        return ret;
    }
    if( subTask->hasErrors() ) {
        stateInfo.setError( subTask->getError() );
        return ret;
    }
    
    if( loadQueryTask == subTask ) {
        query = getSequenceFromDocument( loadQueryTask->getDocument() );
        if( hasErrors() ) {
            stateInfo.setError( getError() + tr( "query sequence" ) );
        }
        loadQueryTask = NULL;
    } else if( loadDbTask == subTask ) {
        db = getSequenceFromDocument( loadDbTask->getDocument() );
        if( hasErrors() ) {
            stateInfo.setError( getError() + tr( "db sequence" ) );
        }
        loadDbTask = NULL;
    } else {
        assert( false && "undefined task finished" );
    }
    
    if( NULL == loadQueryTask && NULL == loadDbTask ) {
        addMemResource();
    }
    return ret;
}

UHMM3SearchResult UHMM3PhmmerTask::getResult() const {
    return result;
}

QList< SharedAnnotationData > UHMM3PhmmerTask::getResultsAsAnnotations( const QString & name ) const {
    assert( !name.isEmpty() );
    QList< SharedAnnotationData > annotations;
    
    foreach( const UHMM3SearchSeqDomainResult & domain, result.domainResList ) {
        AnnotationData * annData = new AnnotationData();
        
        annData->name = name;
        annData->complement = false;
        annData->aminoStrand = TriState_No;
        annData->location << domain.seqRegion;
        annData->qualifiers << Qualifier( "Query sequence", query.getName() );
        domain.writeQualifiersToAnnotation( annData );
        
        annotations << SharedAnnotationData( annData );
    }
    
    return annotations;
}

void UHMM3PhmmerTask::run() {
    if( hasErrors() ) {
        return;
    }
    
    UHMM3SearchTaskLocalStorage::createTaskContext( getTaskId() );
    result = UHMM3Phmmer::phmmer( query.seq.data(), query.length(), db.seq.data(), db.length(), settings, stateInfo );
    UHMM3SearchTaskLocalStorage::freeTaskContext( getTaskId() );
}

/*******************************************
* HMMER3 phmmer search to annotations task.
********************************************/

void UHMM3PhmmerToAnnotationsTask::checkArgs() {
    if( queryfile.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( tr( "query sequence file path" ) ) );
        return;
    }
    if( dbSeq.isNull() ) {
        stateInfo.setError( L10N::badArgument( tr( "db sequence" ) ) );
        return;
    }
    if( NULL == annotationObj.data() ) {
        stateInfo.setError( L10N::badArgument( tr( "annotation object" ) ) );
        return;
    }
    if( annName.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( tr( "annotation name" ) ) );
        return;
    }
    if( annGroup.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( tr( "annotation group" ) ) );
        return;
    }
}

UHMM3PhmmerToAnnotationsTask::UHMM3PhmmerToAnnotationsTask( const QString & qfile, const DNASequence & db,
                                                            AnnotationTableObject * o, const QString & gr, 
                                                            const QString & name, const UHMM3PhmmerSettings & set )
: Task( "HMM Phmmer task", TaskFlags_NR_FOSCOE | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled ), 
queryfile( qfile ), dbSeq( db ), annotationObj( o ), annGroup( gr ), annName( name ), settings( set ),
phmmerTask( NULL ), createAnnotationsTask( NULL ) {
    
    checkArgs();
    if( hasErrors() ) {
        return;
    }
    setTaskName( tr( "HMM Phmmer search %1 sequence with %2 database" ).arg( queryfile ).arg( dbSeq.getName() ) );
    phmmerTask = new UHMM3PhmmerTask( queryfile, dbSeq, settings );
    addSubTask( phmmerTask );
}

QList< Task* > UHMM3PhmmerToAnnotationsTask::onSubTaskFinished( Task * subTask ) {
    assert( NULL != subTask );
    QList< Task* > res;
    if( hasErrors() ) {
        return res;
    }
    if( subTask->hasErrors() ) {
        stateInfo.setError( subTask->getError() );
        return res;
    }
    
    if( annotationObj.isNull() ) {
        stateInfo.setError( tr( "Annotation object removed" ) );
        return res;
    }
    
    if( phmmerTask == subTask ) {
        QList< SharedAnnotationData > annotations = phmmerTask->getResultsAsAnnotations( annName );
        if( annotations.isEmpty() ) {
            return res;
        }
        createAnnotationsTask = new CreateAnnotationsTask( annotationObj, annGroup, annotations );
        res << createAnnotationsTask;
    } else if( createAnnotationsTask != subTask ) {
        assert( false && "undefined_task_finished!" );
    }
    
    return res;
}

QString UHMM3PhmmerToAnnotationsTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("Query sequence") + "</b></td><td>" + QFileInfo( queryfile ).absoluteFilePath() + "</td></tr>";
    
    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task was not finished") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    
    res += "<tr><td><b>" + tr("Result annotation table") + "</b></td><td>" + annotationObj->getDocument()->getName() + "</td></tr>";
    res += "<tr><td><b>" + tr("Result annotation group") + "</b></td><td>" + annGroup + "</td></tr>";
    res += "<tr><td><b>" + tr("Result annotation name") +  "</b></td><td>" + annName + "</td></tr>";
    
    int nResults = createAnnotationsTask == NULL ? 0 : createAnnotationsTask->getAnnotations().size();
    res += "<tr><td><b>" + tr("Results count") +  "</b></td><td>" + QString::number( nResults )+ "</td></tr>";
    res += "</table>";
    return res;
}

} // GB2
