/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <cmath>
#include <cassert>

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>

#include <core_api/AppContext.h>
#include <util_gui/DialogUtils.h>
#include <gobjects/GObjectTypes.h>
#include <util_gui/RemoteMachineMonitorDialogImpl.h>
#include <distributed_computing/DistributedComputingUtil.h>

#include <gobject/uHMMObject.h>

#include "uHMM3SearchLocalTask.h"
#include "uHMM3SearchDialogImpl.h"

namespace GB2 {

const QString UHMM3SearchDialogImpl::DOM_E_PLUS_PREFIX          = "1E+";
const QString UHMM3SearchDialogImpl::DOM_E_MINUS_PREFIX         = "1E";
const QString UHMM3SearchDialogImpl::HMM_FILES_DIR_ID           = "uhmmer3_search_dlg_impl_hmm_dir";
const QString UHMM3SearchDialogImpl::ANNOTATIONS_DEFAULT_NAME   = "hmm_signal";

UHMM3SearchDialogImpl::UHMM3SearchDialogImpl( const DNASequenceObject * seqObj, QWidget * p ) : QDialog( p ) {
    assert( NULL != seqObj );
    
    setupUi( this );
    useScoreTresholdGroup.addButton( useExplicitScoreTresholdButton );
    useScoreTresholdGroup.addButton( useGATresholdsButton );
    useScoreTresholdGroup.addButton( useNCTresholdsButton );
    useScoreTresholdGroup.addButton( useTCTresholdsButton );
    useExplicitScoreTresholdButton->setChecked( true );
    
    model.sequence = seqObj->getDNASequence();
    setModelValues(); // default settings here
    
    // Annotations widget
    CreateAnnotationModel annModel;
    annModel.hideLocation = true;
    annModel.sequenceObjectRef = seqObj;
    annModel.data->name = ANNOTATIONS_DEFAULT_NAME;
    annModel.sequenceLen = seqObj->getSequenceLen();
    annotationsWidgetController = new CreateAnnotationWidgetController( annModel, this );
    QWidget * firstTab = tabWidget->widget( 0 );
    assert( NULL != firstTab );
    QVBoxLayout * curLayout = qobject_cast< QVBoxLayout* >( firstTab->layout() );
    assert( NULL != curLayout );
    QWidget * aw = annotationsWidgetController->getWidget();
    curLayout->insertWidget( 1, aw );
    aw->setMinimumSize( aw->layout()->minimumSize() );
    
    connect( cancelButton, SIGNAL( clicked() ), SLOT( sl_cancelButtonClicked() ) );
    connect( searchButton, SIGNAL( clicked() ), SLOT( sl_okButtonClicked() ) );
    connect( useEvalTresholdsButton, SIGNAL( toggled( bool ) ), SLOT( sl_useEvalTresholdsButtonChanged( bool ) ) );
    connect( useScoreTresholdsButton, SIGNAL( toggled( bool ) ), SLOT( sl_useScoreTresholdsButtonChanged( bool ) ) );
    connect( useExplicitScoreTresholdButton, SIGNAL( toggled( bool ) ), SLOT( sl_useExplicitScoreTresholdButton( bool ) ) );
    connect( maxCheckBox, SIGNAL( stateChanged( int ) ), SLOT( sl_maxCheckBoxChanged( int ) ) );
    connect( domESpinBox, SIGNAL( valueChanged( int ) ), SLOT( sl_domESpinBoxChanged( int ) ) );
    connect( queryHmmFileToolButton, SIGNAL( clicked() ), SLOT( sl_queryHmmFileToolButtonClicked() ) );
    connect( domZCheckBox, SIGNAL( stateChanged( int ) ), SLOT( sl_domZCheckBoxChanged( int ) ) );
    connect( remoteRunPushButton, SIGNAL( clicked() ), SLOT( sl_remoteRunButtonClicked() ) );
}

void UHMM3SearchDialogImpl::setModelValues() {
    const UHMM3SearchSettings & settings = model.searchSettings.inner;
    domESpinBox->setValue( 1 ); assert( 10.0 == settings.domE );
    scoreTresholdDoubleSpin->setValue( 0 ); // because default is OPTION_NOT_SET
    domZDoubleSpinBox->setValue( 0 ); // because default is OPTION_NOT_SET
    nobiasCheckBox->setChecked( (bool)settings.noBiasFilter );
    nonull2CheckBox->setChecked( (bool)settings.noNull2 );
    maxCheckBox->setChecked( (bool)settings.doMax );
    f1DoubleSpinBox->setValue( settings.f1 );
    f2DoubleSpinBox->setValue( settings.f2 );
    f3DoubleSpinBox->setValue( settings.f3 );
    seedSpinBox->setValue( settings.seed );
}

void UHMM3SearchDialogImpl::sl_cancelButtonClicked() {
    reject();
}

void UHMM3SearchDialogImpl::getModelValues() {
    UHMM3SearchSettings & settings = model.searchSettings.inner;
    
    if( useEvalTresholdsButton->isChecked() ) {
        settings.domE =  pow( 10.0, domESpinBox->value() );
        settings.domT = OPTION_NOT_SET;
    } else if( useScoreTresholdsButton->isChecked() ) {
        if( useExplicitScoreTresholdButton->isChecked() ) {
            settings.domT = scoreTresholdDoubleSpin->value();
        } else if( useGATresholdsButton->isChecked() ) {
            settings.useBitCutoffs = p7H_GA;
        } else if( useNCTresholdsButton->isChecked() ) {
            settings.useBitCutoffs = p7H_NC;
        } else if( useTCTresholdsButton->isChecked() ) {
            settings.useBitCutoffs = p7H_TC;
        } else {
            assert( false );
        }
    } else {
        assert( false );
    }
    
    if( domZCheckBox->isChecked() ) {
        settings.domZ = domZDoubleSpinBox->value();
    } else {
        settings.domZ = OPTION_NOT_SET;
    }
    
    settings.noBiasFilter = nobiasCheckBox->isChecked();
    settings.noNull2 = nonull2CheckBox->isChecked();
    settings.doMax = maxCheckBox->isChecked();
    
    settings.f1 = f1DoubleSpinBox->value();
    settings.f2 = f2DoubleSpinBox->value();
    settings.f3 = f3DoubleSpinBox->value();
    
    settings.seed = seedSpinBox->value();
    
    model.hmmfile = queryHmmFileEdit->text();
}

QString UHMM3SearchDialogImpl::checkModel() {
    assert( checkUHMM3SearchSettings( &model.searchSettings.inner ) );
    QString ret;
    
    if( model.hmmfile.isEmpty() ) {
        ret = tr( "HMM profile file path is empty" );
        return ret;
    }
    ret = annotationsWidgetController->validate();
    if( !ret.isEmpty() ) {
        return ret;
    }
    
    return ret;
}

void UHMM3SearchDialogImpl::sl_okButtonClicked() {
    getModelValues();
    QString err = checkModel();
    if( !err.isEmpty() ) {
        QMessageBox::critical( this, tr( "Error: bad arguments!" ), err );
        return;
    }
    
    annotationsWidgetController->prepareAnnotationObject();
    const CreateAnnotationModel & annModel = annotationsWidgetController->getModel();
    UHMM3SWSearchToAnnotationsTask * searchTask = new UHMM3SWSearchToAnnotationsTask( model.hmmfile, model.sequence,
        annModel.getAnnotationObject(), annModel.groupName, annModel.data->name, model.searchSettings );
    AppContext::getTaskScheduler()->registerTopLevelTask( searchTask );
    
    QDialog::accept();
}

void UHMM3SearchDialogImpl::sl_remoteRunButtonClicked() {
    // check dialog model first
    getModelValues();
    QString err = checkModel();
    if( !err.isEmpty() ) {
        QMessageBox::critical( this, tr( "Error: bad arguments!" ), err );
        return;
    }
    
    int ret = 0;
    RemoteMachineSettings * settings = NULL;
    do {
        RemoteMachineMonitor * rmm = AppContext::getRemoteMachineMonitor();
        assert( NULL != rmm );
        RemoteMachineMonitorDialogImpl dlg( QApplication::activeWindow(), rmm->getRemoteMachineMonitorItems(), 
            UHMM3SearchLocalTaskFactory::ID );
        ret = dlg.exec();
        if( QDialog::Rejected == ret ) {
            return;
        }
        assert( QDialog::Accepted == ret );
        
        QList< RemoteMachineMonitorDialogItem > dlgModel = dlg.getModel();
        DistributedComputingUtil::applyChangesForRemoteMachineMonitor( rmm, dlgModel );
        QList< RemoteMachineSettings* > selectedMachines = rmm->getSelectedMachines();
        int howManyMachines = selectedMachines.size();
        if( 0 == howManyMachines ) {
            QMessageBox::critical( this, tr( "Selecting machines error!" ), tr( "You didn't select a machine to run remote task!" ) );
            continue;
        } else if( 1 != howManyMachines ) {
            QMessageBox::critical( this, tr( "Selecting machines error!" ), 
                tr( "Distributed run on many machines is not supported yet. Select 1 machine" ) );
            continue;
        }
        assert( 1 == howManyMachines );
        settings = selectedMachines.first();
        break;
    } while( QDialog::Accepted == ret );
    
    annotationsWidgetController->prepareAnnotationObject();
    const CreateAnnotationModel & annModel = annotationsWidgetController->getModel();
    UHMM3RemoteSearchToAnnotationsTask * searchTask = new UHMM3RemoteSearchToAnnotationsTask( model.hmmfile, model.sequence, 
        model.searchSettings, settings, annModel.getAnnotationObject(), annModel.groupName, annModel.data->name );
    AppContext::getTaskScheduler()->registerTopLevelTask( searchTask );
    
    QDialog::accept();
}

void UHMM3SearchDialogImpl::sl_useEvalTresholdsButtonChanged( bool checked ) {
    domESpinBox->setEnabled( checked );
}

void UHMM3SearchDialogImpl::sl_useScoreTresholdsButtonChanged( bool checked ) {
    useExplicitScoreTresholdButton->setEnabled( checked );
    useGATresholdsButton->setEnabled( checked );
    useNCTresholdsButton->setEnabled( checked );
    useTCTresholdsButton->setEnabled( checked );
    if( !checked ) {
        scoreTresholdDoubleSpin->setEnabled( false );
    } else {
        scoreTresholdDoubleSpin->setEnabled( useExplicitScoreTresholdButton->isChecked() );
    }
}

void UHMM3SearchDialogImpl::sl_useExplicitScoreTresholdButton( bool checked ) {
    scoreTresholdDoubleSpin->setEnabled( checked );
}

void UHMM3SearchDialogImpl::sl_maxCheckBoxChanged( int state ) {
    assert( Qt::PartiallyChecked != state );
    bool unchecked = Qt::Unchecked == state;
    f1Label->setEnabled( unchecked );
    f2Label->setEnabled( unchecked );
    f3Label->setEnabled( unchecked );
    f1DoubleSpinBox->setEnabled( unchecked );
    f2DoubleSpinBox->setEnabled( unchecked );
    f3DoubleSpinBox->setEnabled( unchecked );
}

void UHMM3SearchDialogImpl::sl_domESpinBoxChanged( int newVal ) {
    const QString & prefix = 0 <= newVal ? DOM_E_PLUS_PREFIX : DOM_E_MINUS_PREFIX;
    domESpinBox->setPrefix( prefix );
}

void UHMM3SearchDialogImpl::sl_queryHmmFileToolButtonClicked() {
    LastOpenDirHelper helper( HMM_FILES_DIR_ID );
    helper.url = QFileDialog::getOpenFileName( this, tr( "Select query HMM profile" ), 
        helper, DialogUtils::prepareDocumentsFileFilterByObjType( UHMMObject::UHMM_OT, true ) );
    if( !helper.url.isEmpty() ) {
        queryHmmFileEdit->setText( helper.url );
    }
}

void UHMM3SearchDialogImpl::sl_domZCheckBoxChanged( int state ) {
    assert( Qt::PartiallyChecked != state );
    bool checked = Qt::Checked == state;
    domZDoubleSpinBox->setEnabled( checked );
}

} // GB2
