/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

//#include "vld.h"
#include "NeighborJoinAdapter.h"
#include <core_api/Counter.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/Task.h>
#include <core_api/Log.h>
#include <core_api/SubstMatrixRegistry.h>
#include <phyltree/CreatePhyTreeDialogController.h>
#include <util_algorithm/GAutoDeleteList.h>
#include <datatype/MAlignment.h>
#include <QPushButton>

#include <algorithm>
#include <QtCore/QVector>
#include "qstring.h"
#include <exception>
#include <iostream>
#include "DistanceMatrix.h"
#include <QVector>
#include <core_api/AppContext.h>
#include <memory>
#include "neighbor.h"
#include "dnadist.h"
#include "protdist.h"
#include "DistMatrixModelWidget.h"

namespace GB2 {

//static LogCategory log(ULOG_CAT_PHYLIP_NJOIN);
QMutex NeighborJoinAdapter::runLock;

void createPhyTreeFromPhylipTree(const MAlignment &ma, node *p, double m, boolean njoin, node *start, PhyNode* root)
{
    
    
    /* used in fitch & neighbor */
    long i=0, w=0;
    double x=0.0;
    naym* nayme = getNayme();
    static int counter = 0;

    PhyNode* current = NULL;

    if (p == start) {
        //printf("we got root!\n");
        current = root;
    } else {
        current = new PhyNode;
    }

    if (p->tip) {
        assert(p->index - 1 < ma.getNumRows());
        current->name = ma.getRow(p->index - 1).getName();
    } else {
        current->name = QString("node %1").arg(counter++);
        createPhyTreeFromPhylipTree(ma, p->next->back,  m, njoin, start, current);
        createPhyTreeFromPhylipTree(ma, p->next->next->back, m, njoin, start, current);
        if (p == start && njoin) {
            createPhyTreeFromPhylipTree(ma, p->back, m, njoin, start, current);
        }
    }
    x = p->v;
    if (x > 0.0) 
        w = (long)(m * log(x));
    else if (x == 0.0)
        w = 0;
    else
        w = (long)(m * log(-x)) + 1;
    if (w < 0)
        w = 0;

    if (p == start) {
        counter = 0;
    } else {
        PhyNode::addBranch(root, current, x);
        QString message = QString("Added branch from %1 to %2\n").arg(root->name).arg(current->name);
        //printf("%s\n", qPrintable(message));
    }
}  

void replacePhylipRestrictedSymbols(QByteArray& name) {
    static const char badSymbols[] = {',',':','[',']','(',')',';' };
    static int sz = sizeof (badSymbols) / sizeof(char);
    for (int i = 0; i < sz; ++i) {
        name.replace(badSymbols[i], ' ');
    }
}


PhyTree NeighborJoinAdapter::calculatePhyTree( const MAlignment& ma, const CreatePhyTreeSettings& s, TaskStateInfo& ti)
{
    QMutexLocker runLocker( &runLock );

    GCOUNTER(cvar,tvar, "PhylipNeigborJoin" );


    PhyTree phyTree(NULL);

    if (ma.getNumRows() < 3) {
        ti.setError("Neighbor-Joining runs must have at least 3 species");
        return phyTree;
    }
    
    // Exceptions are used to avoid phylip exit(-1) error handling and canceling task 
    try {   
        
        setTaskInfo(&ti);
        
        std::auto_ptr<DistanceMatrix> distanceMatrix(new DistanceMatrix);
        distanceMatrix->calculateOutOfAlignment(ma,s);

        if (!distanceMatrix->isValid()) {
            ti.setError("Calculated distance matrix is invalid");
            return phyTree;
        }

        int sz = distanceMatrix->rawMatrix.count();

        // Allocate memory resources
        neighbour_init(sz);

        // Fill data
        vector* m = getMtx();
        for (int i = 0; i < sz; ++i) {
            for (int j = 0; j < sz; ++j) {
                m[i][j] = distanceMatrix->rawMatrix[i][j];
            }
        }

        naym* nayme = getNayme();
        for (int i = 0; i < sz; ++i) {
            const MAlignmentRow& row = ma.getRow(i);
            QByteArray name = row.getName().toAscii();
            replacePhylipRestrictedSymbols(name);
            qstrncpy(nayme[i], name.constData(), sizeof(naym));
        }

        // Calculate tree
        const tree* curTree = neighbour_calc_tree();
        PhyNode* root = new PhyNode();
        bool njoin = true;

        ti.progress = 99;
        createPhyTreeFromPhylipTree(ma, curTree->start, 0.43429448222, njoin, curTree->start, root);

        neighbour_free_resources();

        PhyTreeData* data = new PhyTreeData();
        data->rootNode = root;

        phyTree = data;
    } catch (const char* message) {
        ti.setError(QString("Phylip error %1").arg(message));
    }
    return phyTree;

}


void NeighborJoinAdapter::setupCreatePhyTreeUI( CreatePhyTreeDialogController* c, const MAlignment& ma )
{
     CreatePhyTreeWidget* w = new DistMatrixModelWidget(c, ma);
     c->insertWidget(1,w);
     c->adjustSize();
}


} 
