/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SiteconWorkers.h"
#include "SiteconIOWorkers.h"
#include "SiteconSearchTask.h"

#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>

#include <datatype/DNASequence.h>
#include <core_api/DNATranslation.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <util_tasks/MultiTask.h>
#include <util_tasks/FailTask.h>
#include <util_tasks/TaskSignalMapper.h>

#include <QtGui/QApplication>
/* TRANSLATOR GB2::LocalWorkflow::SiteconSearchWorker */

namespace GB2 {
namespace LocalWorkflow {

static const QString MODEL_PORT("model");
static const QString SEQ_PORT("seq");

static const QString NAME_ATTR("a_name");
static const QString STRAND_ATTR("b_strand");
static const QString SCORE_ATTR("c_score");
static const QString E1_ATTR("d_err1");
static const QString E2_ATTR("e_err2");

const QString SiteconSearchWorker::ACTOR_ID("sitecon.search");

static LogCategory log(ULOG_CAT_WD);

void SiteconSearchWorker::registerProto() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    {
        Descriptor md(MODEL_PORT, SiteconSearchWorker::tr("Sitecon Model"), SiteconSearchWorker::tr("Profile data to search with."));
        Descriptor sd(SEQ_PORT, SiteconSearchWorker::tr("Sequence"), SiteconSearchWorker::tr("Input nucleotide sequence to search in."));
        Descriptor od(CoreLibConstants::OUT_PORT_ID, SiteconSearchWorker::tr("SITECON annotations"), SiteconSearchWorker::tr("Annotations marking found TFBS sites."));
        p << new PortDescriptor(md, SiteconWorkerFactory::SITECON_MODEL_TYPE(), true /*input*/, false, BusPort::BLIND_INPUT);
        p << new PortDescriptor(sd, BioDataTypes::DNA_SEQUENCE_TYPE(), true /*input*/);
        p << new PortDescriptor(od, BioDataTypes::ANNOTATION_TABLE_TYPE(), false /*input*/, true /*multi*/);
    }
    {
        Descriptor nd(NAME_ATTR, SiteconSearchWorker::tr("Result annotation"), SiteconSearchWorker::tr("Annotation name for marking found regions"));
        Descriptor snd(STRAND_ATTR, SiteconSearchWorker::tr("Search in"), SiteconSearchWorker::tr("Which strands should be searched: direct, complement or both."));
        Descriptor scd(SCORE_ATTR, SiteconSearchWorker::tr("Min score"), QApplication::translate("SiteconSearchDialog", "min_err_tip", 0, QApplication::UnicodeUTF8));
        Descriptor e1d(E1_ATTR, SiteconSearchWorker::tr("Min Err1"), SiteconSearchWorker::tr("Alternative setting for filtering results, minimal value of Error type I."
            "<br>Note that all thresholds (by score, by err1 and by err2) are applied when filtering results."));
        Descriptor e2d(E2_ATTR, SiteconSearchWorker::tr("Max Err2"), SiteconSearchWorker::tr("Alternative setting for filtering results, max value of Error type II."
            "<br>Note that all thresholds (by score, by err1 and by err2) are applied when filtering results."));
        a << new Attribute(nd, CoreDataTypes::STRING_TYPE(), true, "misc_feature");
        a << new Attribute(snd, CoreDataTypes::NUM_TYPE(), false, 0);
        a << new Attribute(scd, CoreDataTypes::NUM_TYPE(), false, 85);
        a << new Attribute(e1d, CoreDataTypes::NUM_TYPE(), false, 0.);
        a << new Attribute(e2d, CoreDataTypes::NUM_TYPE(), false, 1000);
    }

    Descriptor desc(ACTOR_ID, tr("Search TFBS"), 
        tr("Searches each input sequence for transcription factor binding sites significantly similar to specified SITECON profiles."
        " In case several profiles were supplied, searches with all profiles one by one and outputs merged set of annotations for each sequence.")
        );
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;

    {
        QVariantMap m; m["minimum"] = 60; m["maximum"] = 100; m["suffix"] = "%";
        delegates[SCORE_ATTR] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = 0.; m["maximum"] = 1.; m["singleStep"] = 0.1;
        delegates[E1_ATTR] = new DoubleSpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = 10; m["maximum"] = INT_MAX; m["prefix"] = "1/"; m["singleStep"] = 500;
        delegates[E2_ATTR] = new SpinBoxDelegate(m);
    }

    QVariantMap strandMap; 
    strandMap[tr("both strands")] = 0;
    strandMap[tr("direct strand")] = 1;
    strandMap[tr("complement strand")] = -1;
    delegates[STRAND_ATTR] = new ComboBoxDelegate(strandMap);
    
    proto->setPrompter(new SiteconSearchPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":sitecon/images/sitecon.png");
    WorkflowEnv::getProtoRegistry()->registerProto(SiteconWorkerFactory::SITECON_CATEGORY(), proto);
}

QString SiteconSearchPrompter::composeRichDoc() {
    Actor* modelProducer = qobject_cast<BusPort*>(target->getPort(MODEL_PORT))->getProducer(MODEL_PORT);
    Actor* seqProducer = qobject_cast<BusPort*>(target->getPort(SEQ_PORT))->getProducer(SEQ_PORT);

    QString seqName = seqProducer ? tr("For each sequence from <u>%1</u>,").arg(seqProducer->getLabel()) : "";
    QString modelName = modelProducer ? tr("with all profiles provided by <u>%1</u>,").arg(modelProducer->getLabel()) : "";

    QString resultName = getRequiredParam(NAME_ATTR);

    QString strandName;
    switch (getParameter(STRAND_ATTR).toInt()) {
    case 0: strandName = SiteconSearchWorker::tr("both strands"); break;
    case 1: strandName = SiteconSearchWorker::tr("direct strand"); break;
    case -1: strandName = SiteconSearchWorker::tr("complement strand"); break;
    }

    QString doc = tr("%1 search transcription factor binding sites (TFBS) %2."
        "<br>Recognize sites with <u>similarity %3%</u>, process <u>%4</u>."
        "<br>Output the list of found regions annotated as <u>%5</u>.")
        .arg(seqName)
        .arg(modelName)
        .arg(getParameter(SCORE_ATTR).toInt())
        .arg(strandName)
        .arg(resultName);

    return doc;
}

void SiteconSearchWorker::init() {
    modelPort = ports.value(MODEL_PORT);
    dataPort = ports.value(SEQ_PORT);
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
    dataPort->addComplement(output);
    output->addComplement(dataPort);

    strand = actor->getParameter(STRAND_ATTR)->getAttributeValue<int>();
    cfg.minPSUM = actor->getParameter(SCORE_ATTR)->getAttributeValue<int>();
    cfg.minE1 = actor->getParameter(E1_ATTR)->getAttributeValue<double>();
    cfg.maxE2 = 1./actor->getParameter(E2_ATTR)->getAttributeValue<int>();
    resultName = actor->getParameter(NAME_ATTR)->getAttributeValue<QString>();
}

bool SiteconSearchWorker::isReady() {
    return ((!models.isEmpty() && modelPort->isEnded()) && dataPort->hasMessage()) || modelPort->hasMessage();
}

Task* SiteconSearchWorker::tick() {
    while (modelPort->hasMessage()) {
        models << modelPort->get().getData().value<SiteconModel>();
    }
    if (models.isEmpty() || !modelPort->isEnded() || !dataPort->hasMessage()) {
        return NULL;
    }
    DNASequence seq = dataPort->get().getData().value<DNASequence>();

    if (!seq.isNull() && seq.alphabet->getType() == DNAAlphabet_NUCL) {
        SiteconSearchCfg config(cfg);
        config.complOnly = (strand < 0);
        if (strand <= 0) {
            QList<DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
                lookupTranslation(seq.alphabet, DNATranslationType_NUCL_2_COMPLNUCL);
            if (!compTTs.isEmpty()) {
                config.complTT = compTTs.first();
            }
        }
        QList<Task*> subtasks;
        foreach(SiteconModel model, models) {
            subtasks << new SiteconSearchTask(model, seq.constData(), seq.length(), config, 0);
        }
        Task* t = new MultiTask(tr("Search TFBS in %1").arg(seq.getName()), subtasks);
        connect(new TaskSignalMapper(t), SIGNAL(si_taskFinished(Task*)), SLOT(sl_taskFinished(Task*)));
        return t;
    }
    QString err = tr("Bad sequence supplied to SiteconSearch: %1").arg(seq.getName());
    if (failFast) {
        return new FailTask(err);
    } else {
        log.error(err);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), QVariant()));
        if (dataPort->isEnded()) {
            output->setEnded();
        }
        return NULL;
    }
}

void SiteconSearchWorker::sl_taskFinished(Task* t) {
    QList<SharedAnnotationData> res;
    foreach(Task* sub, t->getSubtasks()) {
        SiteconSearchTask* sst = qobject_cast<SiteconSearchTask*>(sub);
        res += SiteconSearchResult::toTable(sst->takeResults(), resultName);
    }
    QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(res);
    output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
    if (dataPort->isEnded()) {
        output->setEnded();
    }
    log.info(tr("Found %1 TFBS").arg(res.size())); //TODO set task description for report
}

bool SiteconSearchWorker::isDone() {
    return dataPort->isEnded();
}

} //namespace LocalWorkflow
} //namespace GB2
