/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/AppContext.h>
#include <distributed_computing/LocalTaskManager.h>

#include "LocalTaskStarter.h"

#include <QtCore/QEventLoop>

namespace GB2 {

LocalTaskStarter::LocalTaskStarter(
        const QStringList &_taskFileNames, const QString &_tasksDirectory):
    Task(tr("LocalTaskStarter"), TaskFlags_FOSCOE),
    taskFileNames(_taskFileNames),
    tasksDirectory(_tasksDirectory)
{
}

void LocalTaskStarter::run() {
    foreach(const QString& taskFileName, taskFileNames) {
        QVariantList list;
        {
            RemoteTaskError error(false, "unknown error");
            QVariant result;
            {
                QFile file(taskFileName);
                if(file.open(QIODevice::ReadOnly))
                {
                    QVariant data;
                    {
                        QDataStream stream(&file);
                        stream >> data;
                    }
                    error = runLocalTask(data, &result);
                    file.remove();
                }
                else
                {
                    error = RemoteTaskError(false, "can't open task file");
                }
            }
            list << error.serialize();
            list << result;
        }
        {
            QFile file(taskFileName + ".response");
            if(file.open(QIODevice::WriteOnly | QIODevice::Truncate))
            {
                QDataStream stream(&file);
                stream << list;
            }
        }
    }
    while(!taskIds.isEmpty()) {
        foreach(qint64 taskId, taskIds) {
            {
                QFile file(tasksDirectory.absoluteFilePath(QString::number(taskId) + ".cancel"));
                if(file.exists()) {
                    file.remove();
                    AppContext::getLocalTaskManager()->cancelTask(taskId);
                }
            }
            {
                QFile file(tasksDirectory.absoluteFilePath(QString::number(taskId) + ".delete"));
                if(file.exists()) {
                    file.remove();
                    AppContext::getLocalTaskManager()->deleteTask(taskId);
                }
            }
            if(AppContext::getLocalTaskManager()->getTasks().contains(taskId)) {
                QFile file(tasksDirectory.absoluteFilePath(QString::number(taskId)));
                if(file.open(QIODevice::WriteOnly | QIODevice::Truncate))
                {
                    QVariantList list;
                    {
                        QVariant result;
                        RemoteTaskError error = getTaskInfo(taskId, &result);
                        list << error.serialize();
                        list << result;
                    }
                    QDataStream stream(&file);
                    stream << list;
                }
            } else {
                tasksDirectory.remove(tasksDirectory.absoluteFilePath(QString::number(taskId)));
                taskIds.remove(taskId);
            }
        }
        QEventLoop loop;
        QTimer::singleShot(UPDATE_TIME, &loop, SLOT(quit()));
        loop.exec();
    }
}

RemoteTaskError LocalTaskStarter::runLocalTask( const QVariant &data, QVariant *result) {
    assert(NULL != result);
    if(!data.canConvert(QVariant::List)) {
        return RemoteTaskError(false, "invalid request");
    }
    QVariantList list = data.toList();
    if (2 != list.size()) {
        return RemoteTaskError(false, "invalid request");
    }
    if(!list[0].canConvert(QVariant::String)) {
        return RemoteTaskError(false, "invalid request");
    }
    qint64 taskId = -1;
    {
        RemoteTaskError error = AppContext::getLocalTaskManager()->
                                runTask(list[0].toString(), list[1], &taskId);
        if(!error.getOk()) {
            return error;
        }
    }
    taskIds << taskId;
    *result = taskId;
    return RemoteTaskError(true);
}

RemoteTaskError LocalTaskStarter::getTaskInfo(qint64 taskId, QVariant *result)
{
    assert(NULL != result);
    QVariantMap map;
    {
        bool cancelFlag = false;
        RemoteTaskError error =
                AppContext::getLocalTaskManager()->getTaskCancelFlag(taskId, &cancelFlag);
        if(!error.getOk()) {
            return error;
        }
        map.insert("cancelFlag", cancelFlag);
    }
    Task::State state = Task::State_Finished;
    {
        RemoteTaskError error =
                AppContext::getLocalTaskManager()->getTaskState(taskId, &state);
        if(!error.getOk())
        {
            return error;
        }
        map.insert("state", state);
    }
    {
        int progress = 0;
        RemoteTaskError error =
                AppContext::getLocalTaskManager()->getTaskProgress(taskId, &progress);
        if(!error.getOk())
        {
            return error;
        }
        map.insert("progress", progress);
    }
    {
        QVariant taskResult;
        if(Task::State_Finished == state)
        {
            RemoteTaskError error =
                    AppContext::getLocalTaskManager()->getTaskResult(taskId, &taskResult);
            if(!error.getOk())
            {
                return error;
            }
        }
        map.insert("result", taskResult);
    }
    {
        QString errorMessage;
        RemoteTaskError error =
                AppContext::getLocalTaskManager()->getTaskError(taskId, &errorMessage);
        if(!error.getOk())
        {
            return error;
        }
        map.insert("error", errorMessage);
    }
    *result = map;
    return RemoteTaskError(true);
}

} // namespace GB2
