/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_services/AppContextImpl.h>
#include <core_services/SettingsImpl.h>
#include <core_services/DocumentFormatRegistryImpl.h>
#include <core_services/IOAdapterRegistryImpl.h>

#include <plugin_support/PluginSupportImpl.h>
#include <plugin_support/ServiceRegistryImpl.h>
#include <task_scheduler/TaskSchedulerImpl.h>
#include <core_services/AppSettingsImpl.h>

#include <core_api/AppGlobals.h>
#include <core_api/Log.h>
#include <core_api/Timer.h>
#include <core_api/DNATranslation.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/ResourceTracker.h>
#include <core_api/DocumentFormatConfigurators.h>
#include <core_api/DBXRefRegistry.h>
#include <core_api/UserApplicationsSettings.h>
#include <core_api/CudaGpuRegistry.h>
#include <core_api/AtiStreamGpuRegistry.h>
#include <core_api/SubstMatrixRegistry.h>
#include <core_api/SWResultFilterRegistry.h>
#include <core_api/SecStructPredictAlgRegistry.h>
#include <core_api/SmithWatermanTaskFactoryRegistry.h>
#include <core_api/VirtualFileSystem.h>
#include <core_api/DnaAssemblyAlgRegistry.h>
#include <core_api/DataBaseRegistry.h>
#include <util_msa_consensus/MSAConsensusAlgorithmRegistry.h>
#include <util_weight_matrix/PWMConversionAlgorithmRegistry.h>
#include <core_api/RepeatFinderTaskFactoryRegistry.h>


#include <cmdline/CMDLineRegistry.h>
#include <cmdline/CMDLineUtils.h>
#include <cmdline/CMDLineCoreOptions.h>

#include <molecular_geometry/MolecularSurfaceFactoryRegistry.h>

#include <document_format/DNAAlphabetRegistryImpl.h>
#include <gobjects/AnnotationSettings.h>
#include <gobjects/GObjectTypes.h>
#include <test_framework/GTestFrameworkComponents.h>
#include <test_framework/TestRunnerTask.h>
#include <util_gui/BaseDocumentFormatConfigurators.h>
#include <util_ov_msaedit/MSAColorScheme.h>

#include <workflow_support/WorkflowEnvImpl.h>
#include <workflow_support/WorkflowRunTask.h>
#include <workflow_library/BioActorLibrary.h>

#include <util_tasks/TaskStarter.h>
#include <util_index_support/UIndexSupport.h>

#include <distributed_computing/DistributedComputingUtil.h>

#include <phyltree/PhyTreeGeneratorRegistry.h>

#include "LocalTaskStarter.h"
#include "ForeverTask.h"
#include "MachineInfoWriter.h"

#include "LogDriver.h"
#include "ShutdownTask.h"
#include "TestStarter.h"
#include "TaskStatusBar.h"

#include <QtCore/QCoreApplication>

static GB2::LogCategory logCat(ULOG_CAT_USER_INTERFACE);
#define TR_SETTINGS_ROOT QString("test_runner/")

/* TRANSLATOR GB2::AppContextImpl */

using namespace GB2;

static void registerCoreServices() {
    ServiceRegistry* sr = AppContext::getServiceRegistry();
    TaskScheduler* ts = AppContext::getTaskScheduler();
    Q_UNUSED(sr); Q_UNUSED(ts);
    // unlike ugene's UI Main.cpp we don't create PluginViewerImpl, ProjectViewImpl
//    ts->registerTopLevelTask(sr->registerServiceTask(new ScriptRegistryService()));
}

// we will run task that don't die
static void setCongeneStayAlive() {
    AppContext::getTaskScheduler()->registerTopLevelTask( new ForeverTask() );
}

static bool openDocs() {
    bool ret = false;
    QStringList suiteUrls = CMDLineRegistryUtils::getParameterValuesByWords( CMDLineCoreOptions::SUITE_URLS );
    if( suiteUrls.size() > 0 ) {
        TestStarter* ts=new TestStarter( suiteUrls );
        
        GTestEnvironment* envs=ts->getEnv();
        envs->setVar(TIME_OUT_VAR, AppContext::getSettings()->getValue(TR_SETTINGS_ROOT + TIME_OUT_VAR,QString("0")).toString());
        envs->setVar(NUM_THREADS_VAR, AppContext::getSettings()->getValue(TR_SETTINGS_ROOT + NUM_THREADS_VAR,QString("5")).toString());
        
        QObject::connect(AppContext::getPluginSupport(), SIGNAL(si_allStartUpPluginsLoaded()), new TaskStarter(ts), SLOT(registerTask()));
        ret = true;
    }
    QStringList tasks = CMDLineRegistryUtils::getParameterValuesByWords( CMDLineCoreOptions::LOCAL_TASK_FILES );
    if( tasks.size() > 0 ) {
        QObject::connect(AppContext::getPluginSupport(), SIGNAL(si_allStartUpPluginsLoaded()),
            new TaskStarter(new LocalTaskStarter( tasks, QDir::homePath())), SLOT(registerTask()));
        ret = true;
    }
    
    return ret;
}


static void updateStaticTranslations() {
    GObjectTypes::initTypeTranslations();
}


static void setScriptsSearchPath() {
    QStringList scriptsSearchPath;
    const static char * RELATIVE_SCRIPTS_DIR = "/scripts";
    const static char * RELATIVE_DEV_SCRIPTS_DIR = "/../../scripts";

    QString appDirPath = QCoreApplication::applicationDirPath();
    if( QDir(appDirPath+RELATIVE_SCRIPTS_DIR).exists() ) {
        scriptsSearchPath.push_back( appDirPath+RELATIVE_SCRIPTS_DIR );
    } else if( QDir(appDirPath+RELATIVE_DEV_SCRIPTS_DIR).exists() ) {
        scriptsSearchPath.push_back( appDirPath+RELATIVE_DEV_SCRIPTS_DIR );
    }
    if( scriptsSearchPath.empty() ) {
        scriptsSearchPath.push_back("/");
    }
    QDir::setSearchPaths( PATH_PREFIX_SCRIPTS, scriptsSearchPath );
}

static void setDataSearchPaths() {
    //set search paths for data files
    QStringList dataSearchPaths;
#if (defined(Q_OS_LINUX) || defined(Q_OS_UNIX)) && defined( UGENE_DATA_DIR )
    //using directory which is set during installation process on linux
    QString ugene_data_dir( UGENE_DATA_DIR );
    if( QDir(ugene_data_dir).exists() ) {
        dataSearchPaths.push_back( QString(UGENE_DATA_DIR) );
    }
#endif
    const static char * RELATIVE_DATA_DIR = "/data";
    const static char * RELATIVE_DEV_DATA_DIR = "/../../data";
    //on windows data is normally located in the application directory
    QString appDirPath = QCoreApplication::applicationDirPath();
    
    if( QDir(appDirPath+RELATIVE_DATA_DIR).exists() ) {
        dataSearchPaths.push_back( appDirPath+RELATIVE_DATA_DIR );
    } else if( QDir(appDirPath+RELATIVE_DEV_DATA_DIR).exists() ) {          //data location for developers
        dataSearchPaths.push_back( appDirPath+RELATIVE_DEV_DATA_DIR );
    }
    if( dataSearchPaths.empty() ) {
        dataSearchPaths.push_back("/");
    }
    QDir::setSearchPaths( PATH_PREFIX_DATA, dataSearchPaths );
    //now data files may be opened using QFile( "data:some_data_file" )
}

static void setSearchPaths() {
    setDataSearchPaths();
    setScriptsSearchPath();
}

int main(int argc, char **argv) 
{
    const char* build = QT_VERSION_STR, *runtime = qVersion();
    if (strcmp(build, runtime) > 0){
        printf("Installed Qt version must be %s or greater \r\n", QT_VERSION_STR);
        return -1;
    }

    if (argc == 1) {
        fprintf(stderr, "Console interface for Unipro UGENE v%d.%d.%d\n"
#ifdef Q_OS_WIN
            "Hint: Use 'ugeneui.exe' command to run GUI version of UGENE\n"
            "Hint: Use 'ugene.exe --help' or 'ugene.exe --help=<sectionName>' for more information\n"
#else
            "Hint: Use 'ugene -ui' command to run GUI version of UGENE\n"
            "Hint: Use 'ugene --help' or 'ugene --help=<sectionName>' for more information\n"
#endif
            "\nError: missing operands!\n",
             UGENE_VERSION_1, UGENE_VERSION_2, UGENE_VERSION_3
            );
        return -1;   
    }

    GTIMER(c1, t1, "main()->QApp::exec");

    QCoreApplication app(argc, argv);
    QCoreApplication::addLibraryPath(QCoreApplication::applicationDirPath());
    QString devPluginsPath = QDir(QCoreApplication::applicationDirPath()+"/../../installer/windows").absolutePath();
    QCoreApplication::addLibraryPath(devPluginsPath); //dev version

    setSearchPaths();

    AppContextImpl* appContext = AppContextImpl::getApplicationContext();
    
    // parse all cmdline arguments
    CMDLineRegistry* cmdLineRegistry = new CMDLineRegistry(app.arguments()); 
    appContext->setCMDLineRegistry(cmdLineRegistry);
    
    //1 create settings
    SettingsImpl* globalSettings = new SettingsImpl(QSettings::SystemScope);
    appContext->setGlobalSettings(globalSettings);

    QString pathToSettings = cmdLineRegistry->getParameterValue( CMDLineCoreOptions::LOAD_SETTINGS_FILE );
    bool settingsFromFile = QFileInfo( pathToSettings ).exists();
    SettingsImpl * settings = settingsFromFile ? new SettingsImpl( pathToSettings ) : new SettingsImpl( QSettings::UserScope );
    settings->setValue( SETTINGS_FROM_FILE, settingsFromFile );
    settings->setValue(GLOBAL_SETTINGS + "drawIcons", false);
    appContext->setSettings( settings );
    
    AppSettings* appSettings = new AppSettingsImpl();
    appContext->setAppSettings(appSettings);

    UserAppsSettings* userAppSettings = AppContext::getAppSettings()->getUserAppsSettings();

    // set translations
    QTranslator translator;
    QString transFile[] = {
        userAppSettings->getTranslationFile(),
        "transl_" + QLocale::system().name(),
        "transl_en"
    };
    bool trOK = false;
    for (int i = transFile[0].isEmpty() ? 1 : 0; i < 3; ++i) {
        if (!translator.load(transFile[i], QCoreApplication::applicationDirPath())) {
            fprintf(stderr, "Translation not found: %s\n", transFile[i].toAscii().constData());
        } else {
            trOK = true;
            break;
        }
    }
    if (!trOK) {
        fprintf(stderr, "No translations found, exiting\n");
        return 1;   
    }
    
    app.installTranslator(&translator);
    updateStaticTranslations();
    
    // 2 create functional components of congene
    LogDriver logs;
    logCat.details(AppContextImpl::tr("UGENE initialization started"));

    ResourceTracker* resTrack = new ResourceTracker();
    appContext->setResourceTracker(resTrack);

    TaskSchedulerImpl* ts = new TaskSchedulerImpl(appSettings->getAppResourcePool());
    appContext->setTaskScheduler(ts);

    AnnotationSettingsRegistry* asr = new AnnotationSettingsRegistry();
    appContext->setAnnotationSettingsRegistry(asr);

    TestFramework* tf = new TestFramework();
    appContext->setTestFramework(tf);
    
    RepeatFinderTaskFactoryRegistry* rfr = new RepeatFinderTaskFactoryRegistry();
    appContext->setRepeatFinderTaskFactoryRegistry(rfr);

    CMDLineUtils::init();
    
    PhyTreeGeneratorRegistry* phyreg = new PhyTreeGeneratorRegistry();
    appContext->setPhyTreeGeneratorRegistry(phyreg);
    
    // unlike ugene's main.cpp we don't create MainWindowImpl, AppSettingsGUI and GObjectViewFactoryRegistry
    
    PluginSupportImpl* psp = new PluginSupportImpl();
    appContext->setPluginSupport(psp);
    
    ServiceRegistryImpl* sreg = new ServiceRegistryImpl() ;
    appContext->setServiceRegistry(sreg);

    DocumentFormatRegistryImpl* dfr = new DocumentFormatRegistryImpl();
    appContext->setDocumentFormatRegistry(dfr);

    DocumentFormatConfigurators* dfc = new DocumentFormatConfigurators();
    appContext->setDocumentFormatConfigurators(dfc);
    BaseDocumentFormatConfigurators::initBuiltInConfigurators();

    IOAdapterRegistryImpl* io = new IOAdapterRegistryImpl();
    appContext->setIOAdapterRegistry(io);

    DNATranslationRegistry* dtr = new DNATranslationRegistry();
    appContext->setDNATranslationRegistry(dtr);

    DNAAlphabetRegistry* dal = new DNAAlphabetRegistryImpl(dtr);
    appContext->setDNAAlphabetRegistry(dal);

    // unlike ugene's main.cpp we don't create ScriptManagerView, DBXRefRegistry and MSAColorSchemeRegistry
    
    MSAConsensusAlgorithmRegistry* msaConsReg = new MSAConsensusAlgorithmRegistry();
    appContext->setMSAConsensusAlgorithmRegistry(msaConsReg);

    PWMConversionAlgorithmRegistry* pwmConvReg = new PWMConversionAlgorithmRegistry();
    appContext->setPWMConversionAlgorithmRegistry(pwmConvReg);

    SubstMatrixRegistry* smr = new SubstMatrixRegistry();
    appContext->setSubstMatrixRegistry(smr);

    SmithWatermanTaskFactoryRegistry* swar = new SmithWatermanTaskFactoryRegistry();
    appContext->setSmithWatermanTaskFactoryRegistry(swar);

    MolecularSurfaceFactoryRegistry* msfr = new MolecularSurfaceFactoryRegistry();
    appContext->setMolecularSurfaceFactoryRegistry(msfr);

    SWResultFilterRegistry* swrfr = new SWResultFilterRegistry();
    appContext->setSWResultFilterRegistry(swrfr);

    SecStructPredictAlgRegistry* sspar = new SecStructPredictAlgRegistry();
    appContext->setSecStructPedictAlgRegistry(sspar);

    CudaGpuRegistry * cgr = new CudaGpuRegistry();
    appContext->setCudaGpuRegistry( cgr ); 

    AtiStreamGpuRegistry * asgr = new AtiStreamGpuRegistry ();
    appContext->setAtiStreamGpuRegistry( asgr ); 

    DistributedComputingUtil * distrUtil = new DistributedComputingUtil();
    
    VirtualFileSystemRegistry * vfsReg = new VirtualFileSystemRegistry();
    appContext->setVirtualFileSystemRegistry( vfsReg );
    
    Workflow::WorkflowEnv::init(new Workflow::WorkflowEnvImpl());
    Workflow::BioActorLibrary::init();

    DnaAssemblyAlgRegistry* assemblyReg = new DnaAssemblyAlgRegistry();
    appContext->setDnaAssemblyAlgRegistry(assemblyReg);

    DataBaseRegistry* dbr = new DataBaseRegistry();
    appContext->setDataBaseRegistry(dbr);
    
    MachineInfoWriter machineInfoWriter;
    
    TaskStatusBarCon* tsbc=new TaskStatusBarCon();
    
    //work with congene long life
    if( cmdLineRegistry->hasParameter( CMDLineCoreOptions::STAY_ALIVE ) ) {
        setCongeneStayAlive();
    }
    
    // show help if need
    bool showHelp = cmdLineRegistry->hasParameter( CMDLineCoreOptions::HELP );
    if( showHelp ) {
        QObject::connect( psp, SIGNAL( si_allStartUpPluginsLoaded() ), cmdLineRegistry, SLOT( sl_dumpHelp() ) );
    }
    
    openDocs();
    registerCoreServices();
    
    //3 run QT 
    t1.stop();
    logCat.info(AppContextImpl::tr("UGENE started"));
    ShutdownTask watchQuit(&app);
    int rc = app.exec();

    //4 deallocate resources
    Workflow::WorkflowEnv::shutdown();
    
    delete tsbc;
    
    delete distrUtil;
    
    appContext->setVirtualFileSystemRegistry( NULL );
    delete vfsReg;
    
    delete swrfr;
    appContext->setSWResultFilterRegistry(NULL);

    delete swar;
    appContext->setSmithWatermanTaskFactoryRegistry(NULL);

    delete msfr;
    appContext->setMolecularSurfaceFactoryRegistry(NULL);

    delete smr;
    appContext->setSubstMatrixRegistry(NULL);

    delete sreg;
    appContext->setServiceRegistry(NULL);

    delete psp;
    appContext->setPluginSupport(NULL);

    delete tf;
    appContext->setTestFramework(0);

    delete dal;
    appContext->setDNAAlphabetRegistry(NULL);

    delete dtr;
    appContext->setDNATranslationRegistry(NULL);

    delete io;
    appContext->setIOAdapterRegistry(NULL);

    delete dfc;
    appContext->setDocumentFormatConfigurators(NULL);

    delete dfr;
    appContext->setDocumentFormatRegistry(NULL);

    delete ts;
    appContext->setTaskScheduler(NULL);

    delete asr;
    appContext->setAnnotationSettingsRegistry(NULL);

    delete resTrack;
    appContext->setResourceTracker(NULL);

    delete cgr;
    appContext->setCudaGpuRegistry(NULL);

    delete asgr;
    appContext->setAtiStreamGpuRegistry(NULL);

    delete appSettings;
    appContext->setAppSettings(NULL);

    delete settings;
    appContext->setSettings(NULL);

    delete globalSettings;
    appContext->setGlobalSettings(NULL);

    delete cmdLineRegistry;
    appContext->setCMDLineRegistry(NULL);
    
    delete sspar;
    appContext->setSecStructPedictAlgRegistry(NULL);

    delete msaConsReg;
    appContext->setMSAConsensusAlgorithmRegistry(NULL);

    appContext->setPWMConversionAlgorithmRegistry(NULL);
    delete pwmConvReg;

    delete dbr;
    appContext->setDataBaseRegistry(NULL);

    return rc;
}
