/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QSysInfo>
#include <QtCore/QDate>
#include <QtCore/QBuffer>

#include <QtGui/QMessageBox>

#include <core_api/AppGlobals.h>
#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <core_api/Log.h>
#include <core_api/NetworkConfiguration.h>
#include <core_api/AppSettings.h>
#include <core_api/Counter.h>
#include <core_api/MainWindow.h>

#include <util_algorithm/SynchHttp.h>

#include <core_api/UserApplicationsSettings.h>
#include "Shtirlitz.h"

const static char * SHTIRLITZ_LOG = "Plugin Shtirlitz";

const static char * SETTINGS_NOT_FIRST_LAUNCH     = "shtirlitz/not_first_launch";
const static char * SETTINGS_PREVIOUS_REPORT_DATE = "shtirlitz/previous_report_date";
const static char * SETTINGS_COUNTERS             = "shtirlitz/counters";
const static char * SETTINGS_UGENE_UID            = "shtirlitz/uid";

const static int DAYS_BETWEEN_REPORTS = 7;

//This file stores the actual location of reports-receiver script.
const static char * DESTINATION_URL_KEEPER_SRV = "http://ugene.unipro.ru";
const static char * DESTINATION_URL_KEEPER_PAGE = "/reports_dest.html";

namespace GB2 {

static LogCategory log(SHTIRLITZ_LOG);

//////////////////////////////////////////////////////////////////////////
// Shtirlitz itself

QUuid Shtirlitz::getUniqueUgeneId() {
    static QUuid uniqueUgeneId;
    if( uniqueUgeneId.isNull() ) {
        //try to load it from settings
        QVariant uuidQvar = AppContext::getSettings()->getValue( SETTINGS_UGENE_UID );
        if( uuidQvar.isNull() ) {
            uniqueUgeneId = QUuid::createUuid();
            AppContext::getSettings()->setValue( SETTINGS_UGENE_UID, uniqueUgeneId.toString() );
        } else {
            uniqueUgeneId = QUuid( uuidQvar.toString() );
        }
    }
    return uniqueUgeneId;
}

const static char * firstTimeNotice =
"Dear UGENE user!\n"
"UGENE has the possibility to collect statistical information about its usage and report it to our team. "
"Such information helps us a lot in making our software better and more functional than it is.\n"
"\n"
"Kinds of information we collect:\n"
"1. System info: UGENE version, OS name, Qt version and so on.\n"
"2. Counters info: number of launches of certain tasks (e. g. HMM search, MUSCLE align)\n"
"\n"
"We DO NOT collect any personal data.\n"
"If you want this feature enabled, press 'Yes'. Otherwise, press 'No'."
;

//Report about system is sent on the first launch of UGENE.
//Statistical reports are sent once per DAYS_BETWEEN_REPORTS.
void Shtirlitz::wakeup() {
    Settings * s = AppContext::getSettings();
    bool firztLaunch = firstLaunch();
    //1. Check if UGENE is launched for the first time
    if( firztLaunch ) {
        s->setValue( SETTINGS_NOT_FIRST_LAUNCH, QVariant(true) );
        QMessageBox::StandardButton answ = QMessageBox::question(0, tr("Statistical reports"), tr(firstTimeNotice), QMessageBox::Yes | QMessageBox::No );
        if( QMessageBox::Yes != answ ) {
            AppContext::getAppSettings()->getUserAppsSettings()->setEnableCollectingStatistics( false );
            return;
        }
        AppContext::getAppSettings()->getUserAppsSettings()->setEnableCollectingStatistics( true );
        log.details( tr("Shtirlitz is sending the first-time report") );
        sendSystemReport();
        //Leave a mark that the first-time report was sent
    } 

    //2. Do nothing if Shtirlitz was disabled
    if( !enabled() ) {
        return;
    }

    //3. Check if previous report was sent more than a week ago
    if( !firztLaunch  ) {
        QVariant prevDateQvar = AppContext::getSettings()->getValue( SETTINGS_PREVIOUS_REPORT_DATE );
        QDate prevDate = prevDateQvar.toDate();
        int daysPassed = prevDate.isValid() ? prevDate.daysTo(QDate::currentDate()) : 0;

        if( !prevDate.isValid() || daysPassed > DAYS_BETWEEN_REPORTS ) {
            log.details( tr("%1 days passed passed since previous Shtirlitz's report. Shtirlitz is sending the new one.") );
            sendCountersReport();
            //and save the new date
            s->setValue( SETTINGS_PREVIOUS_REPORT_DATE, QDate::currentDate() );
        }
    }
}

void Shtirlitz::sendCustomReport( const QString & customReport ) {
    Task * t = new ShtirlitzTask(customReport);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void Shtirlitz::saveGatheredInfo() {
    if( !enabled() ) {
        return;
    }
    //1. Save counters
    Settings * s = AppContext::getSettings();
    QList<GCounter*> appCounters = GCounter::allCounters();
    foreach( GCounter * ctr, appCounters ) {
        if( qobject_cast<GReportableCounter*>(ctr) ) {
            QString ctrKey = ctr->name;
            double ctrVal = ctr->scaledTotal();

            QString curKey = QString(SETTINGS_COUNTERS) + "/" + ctrKey;
            QVariant lastValQvar = s->getValue( curKey, QVariant() );
            double lastVal = lastValQvar.canConvert<double>() ? lastValQvar.toDouble() : 0.;
            double newVal = (lastVal + ctrVal > lastVal) ? lastVal + ctrVal : lastVal; //overflow detection
            s->setValue( curKey, newVal );
        }
    }
}

bool Shtirlitz::enabled() {
    //1. Check environment variable for developers
    if( QProcess::systemEnvironment().contains(ENV_UGENE_DEV) ) {
        return false;
    }
    return AppContext::getAppSettings()->getUserAppsSettings()->enableCollectingStatistics();
}

void Shtirlitz::sendCountersReport() {
    QString countersReport = formCountersReport();
    sendCustomReport(countersReport);
}

void Shtirlitz::sendSystemReport() {
    QString systemReport = formSystemReport();
    sendCustomReport(systemReport);
}

//Tries to load saved counters from settings.
//Adds loaded counters to report, sets saved values to zero
QString Shtirlitz::formCountersReport() {
    Settings * s = AppContext::getSettings();
    QString countersReport;
    countersReport += "COUNTERS REPORT:\n";
    countersReport += "ID: " + getUniqueUgeneId() + "\n";
    QStringList savedCounters = s->getAllKeys( SETTINGS_COUNTERS );
    if( savedCounters.empty() ){
        countersReport += "NO INFO\n";
    } else {
        foreach( QString savedCtrName, savedCounters ) {
            QVariant savedCtrQvar = s->getValue( QString(SETTINGS_COUNTERS) + "/" + savedCtrName );
            double savedCtrVal = savedCtrQvar.canConvert<double>() ? savedCtrQvar.toDouble() : 0.;
            countersReport += savedCtrName + " : " + QString::number(savedCtrVal) + "\n";
            s->setValue( QString(SETTINGS_COUNTERS) + "/" + savedCtrName, 0. );
        }
    }
    countersReport += "ENDOF COUNTERS REPORT.\n";
    return countersReport;
}

QString Shtirlitz::formSystemReport() {
    QString dateAndTime = QDateTime::currentDateTime().toString(Qt::ISODate);
    QString ugeneVer = QString::number(UGENE_VERSION_1) + "." + QString::number(UGENE_VERSION_2) + "." + QString::number(UGENE_VERSION_3);
    QString qtVersion = qVersion();
    QString osName;
    QString osVersion;
    getOsNameAndVersion( osName, osVersion );
    
    QString systemReport;
    systemReport += "SYSTEM REPORT:\n";
    systemReport += "ID: " + getUniqueUgeneId() + "\n";
    systemReport += "Date and time: " + dateAndTime + "\n";
    systemReport += "Qt Version: " + qtVersion + "\n";
    systemReport += "UGENE version: " + ugeneVer + "\n";
    systemReport += "Word size: " + QString::number(QSysInfo::WordSize) + "\n";
    systemReport += "OS name: " + osName + "\n";
    systemReport += "OS version: " + osVersion + "\n";
    systemReport += "ENDOF SYSTEM REPORT.\n";

    return systemReport;
}

void Shtirlitz::getOsNameAndVersion( QString & name, QString & version ) {
#if defined(Q_OS_WIN)
    name = "Windows";
    version = QString::number(QSysInfo::WindowsVersion);
#elif defined(Q_OS_MAC) 
    name = "Mac";
    version = QString::number(QSysInfo::MacintoshVersion);
#elif defined(Q_OS_LINUX)
    name = "Linux";
    Q_UNUSED(version);//no version is available :(
#else
    name = "Other";
#endif
}

bool Shtirlitz::firstLaunch() {
    QVariant launchedQvar = AppContext::getSettings()->getValue( SETTINGS_NOT_FIRST_LAUNCH );
    return ( !launchedQvar.isValid() || launchedQvar.isNull() );

}

QString Shtirlitz::tr( const char * str ) {
    return QObject::tr(str);
}

//////////////////////////////////////////////////////////////////////////
// Shtirlitz Task

ShtirlitzTask::ShtirlitzTask( const QString & _report ) :
Task("Shtirlitz task", TaskFlag_None), report(_report) {
}

void ShtirlitzTask::run() {
    stateInfo.setStateDesc( tr("Connecting to remote server") );

    //Creating QHttp object and enabling proxy if needed.
    SyncHTTP http( QUrl(DESTINATION_URL_KEEPER_SRV).host() );
    NetworkConfiguration * nc = AppContext::getAppSettings()->getNetworkConfiguration();
    bool isProxy = nc->isProxyUsed( QNetworkProxy::HttpProxy );
    bool isException = nc->getExceptionsList().contains( QUrl(DESTINATION_URL_KEEPER_SRV).host() );
    
    if (isProxy && !isException) {
        http.setProxy(nc->getProxy(QNetworkProxy::HttpProxy));
    }

    QByteArray preparedReport("data=");
    preparedReport += QUrl::toPercentEncoding(report);
    QBuffer reportBuf(&preparedReport);

    // Get actual location of the reports receiver
    //FIXME: error handling
    QString reportsPath = http.syncGet( DESTINATION_URL_KEEPER_PAGE );
    if( reportsPath.isEmpty() ) {
        stateInfo.setError( tr("Cannot resolve destination path for statistical reports") );
        return;
    } 
    if( QHttp::NoError != http.error() ) {
        stateInfo.setError( tr("Network error while resolving destination URL: ") + http.errorString() );
        return;
    }

    //Checking proxy again, for the new url
    SyncHTTP http2( QUrl(reportsPath).host() );
    isException = nc->getExceptionsList().contains( QUrl(reportsPath).host() );
    if( isProxy && !isException ) {
        http2.setProxy( nc->getProxy(QNetworkProxy::HttpProxy) );
    }
    QString fullPath = reportsPath + "?" + preparedReport;
    QString res = http2.syncGet(fullPath); //TODO: consider using POST method?
    if( QHttp::NoError != http.error() ) {
        stateInfo.setError( tr("Network error while sending report: ") + http2.errorString() );
        return;
    }
}

}//GB2
