/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */


#ifndef UG_CATEGORY_H
#define UG_CATEGORY_H

#include <ug_dataset.h>
#include <ug_plugin.h>

#ifdef __cplusplus
extern "C" {
#endif


typedef struct	UgCategory_			UgCategory;
typedef enum	UgCategoryHints_	UgCategoryHints;
typedef struct	UgRelation_			UgRelation;

typedef gboolean (*UgWatchFunc) (gpointer bin, UgDataset* dataset, UgMessage* message, gpointer user_data);
typedef void     (*UgCategoryAddFunc) (gpointer category, UgDataset* dataset);
typedef GList*   (*UgCategoryGetList) (gpointer category);

extern const	UgDataClass*		UgCategoryClass;
extern const	UgDataClass*		UgRelationClass;

// ----------------------------------------------------------------------------
// UgCategory: abstract class
//
// UgData
// |
// `- UgCategory  (abstract class)
//
#define	UG_CATEGORY_CLASS_NAME		"category"		// UgDataClass.name

#define	UG_CATEGORY_DATA_ENTRY		\
	{"name",			G_STRUCT_OFFSET (UgCategory, name),				UG_DATA_TYPE_STRING,	NULL,	NULL},	\
	{"ActiveLimit",		G_STRUCT_OFFSET (UgCategory, limit.active),		UG_DATA_TYPE_UINT,		NULL,	NULL},	\
	{"FinishedLimit",	G_STRUCT_OFFSET (UgCategory, limit.finished),	UG_DATA_TYPE_UINT,		NULL,	NULL},	\
	{"RecycledLimit",	G_STRUCT_OFFSET (UgCategory, limit.recycled),	UG_DATA_TYPE_UINT,		NULL,	NULL},	\
	{"DownloadDefault",	G_STRUCT_OFFSET (UgCategory, defaults),			UG_DATA_TYPE_INSTANCE,	NULL,	NULL},	\
	{"DownloadIndices",	G_STRUCT_OFFSET (UgCategory, indices),			UG_DATA_TYPE_CUSTOM,	(UgInMarkupFunc)ug_int_list_in_markup,	(UgToMarkupFunc)ug_int_list_to_markup}

#define UG_CATEGORY_MEMBERS				\
	UG_DATA_MEMBERS;					\
	gchar*			name;				\
	struct {							\
		guint		active;				\
		guint		finished;			\
		guint		recycled;			\
	} limit;							\
	GList*			indices;			\
	UgDataset*		defaults;			\
	struct {							\
		UgWatchFunc	func;				\
		gpointer	data;				\
	} watch

struct UgCategory_
{
	UG_CATEGORY_MEMBERS;
//	const UgDataClass*	data_class;			// UG_DATA_MEMBERS
//	gchar*				name;
//	struct {
//		guint			active;
//		guint			finished;			// finished: completed and paused
//		guint			recycled;
//	} limit;
//	GList*				indices;			// used when program save/load category
//	UgDataset*			defaults;			// default setting of UgDataset
//	struct {
//		UgWatchFunc		func;				// watch (global)
//		gpointer		data;
//	} watch;
};

// ------------------------------------
// UgCategoryHints
//
enum	UgCategoryHints_
{
//	UG_HINT_QUEUING				= 1 << 0,
	UG_HINT_PAUSED				= 1 << 1,
	UG_HINT_ACTIVE				= 1 << 2,
	UG_HINT_ERROR				= 1 << 3,
	UG_HINT_COMPLETED			= 1 << 4,	// Download completed only
//	UG_HINT_RESERVE				= 1 << 5,

	UG_HINT_FINISHED			= 1 << 6,	// Download completed, Uget will not use it in future.
	UG_HINT_RECYCLED			= 1 << 7,	// Download will be deleted.

//	UG_HINT_SEEDING				= UG_HINT_ACTIVE | UG_HINT_COMPLETED,
	UG_HINT_UNRUNNABLE			= UG_HINT_PAUSED | UG_HINT_ERROR | UG_HINT_FINISHED | UG_HINT_RECYCLED,

	UG_HINT_RESUMABLE			= 1 << 8,
	UG_HINT_UNRESUMABLE			= 1 << 9,
	UG_HINT_RESUMED_FIELD		= UG_HINT_RESUMABLE | UG_HINT_UNRESUMABLE,
};


void	ug_category_init (UgCategory* category);
void	ug_category_finalize (UgCategory* category);
void	ug_category_assign (UgCategory* category, UgCategory* src);

// refresh state, dispatch messages, and call watch functions.
gboolean	ug_category_refresh (UgCategory* category, UgDataset* dataset);
// start plug-in and set UG_HINT_ACTIVE to UgRelation.hints
gboolean	ug_category_start   (UgCategory* category, UgDataset* dataset);
// stop  plug-in and set UG_HINT_PAUSED to UgRelation.hints
void		ug_category_stop    (UgCategory* category, UgDataset* dataset);

// UgCategory.indices input/output for UgMarkup
void	ug_int_list_in_markup (GList** list, GMarkupParseContext* context);
void	ug_int_list_to_markup (GList** list, UgMarkup* markup);

// ----------------------------------------------------------------------------
// CategoryList
//
// Before calling ug_category_list_load(), user must register data class of UgCategory.
// Before calling ug_category_list_save(), user must call ug_download_list_save() to save all download in all category.
GList*		ug_category_list_load (const gchar* filename);
gboolean	ug_category_list_save (GList* category_list, const gchar* filename, UgCategoryGetList get_list);
void		ug_category_list_link (GList* category_list, GList* download_list,  UgCategoryAddFunc add_func);

// ----------------------------------------------------------------------------
// DownloadList
// ug_download_list_load() load file and return list of newly-created UgDataset.
// To free the return value, use:
//	g_list_foreach (list, (GFunc) ug_dataset_unref, NULL);
//	g_list_free (list);
// Before calling ug_download_list_load(), user must register data class of UgRelation.
GList*		ug_download_list_load (const gchar* filename);
gboolean	ug_download_list_save (GList* download_list, const gchar* filename);
// Below utility functions can be used by g_list_foreach()
gboolean	ug_download_attachment_create (UgDataset* dataset);
gboolean	ug_download_attachment_set (UgDataset* dataset, UgDataset* src);
void		ug_download_data_complete (UgDataset* dataset);
void		ug_download_temp_delete (UgDataset* dataset);


// ----------------------------------------------------------------------------
// UgRelation : relation of UgCategory, UgDataset, and UgPlugin. (abstract class)
//
// UgData
// |
// `- UgDataList
//    |
//    `- UgRelation  (abstract class)
//
#define	UG_RELATION_CLASS_NAME		"relation"		// UgDataClass.name

#define	UG_RELATION_DATA_ENTRY		\
	{"hints",		G_STRUCT_OFFSET (UgRelation, hints),	UG_DATA_TYPE_UINT,	NULL,	NULL}

#define	UG_RELATION_MEMBERS(UgRelationChild)	\
	UG_DATA_LIST_MEMBERS(UgRelationChild);		\
	guint				index;					\
	UgCategoryHints		hints;					\
	UgPlugin*			plugin;					\
	struct {									\
		UgWatchFunc		func;					\
		gpointer		data;					\
	} watch

struct UgRelation_
{
	UG_RELATION_MEMBERS (UgRelation);
//	const UgDataClass*	data_class;		// UG_DATA_LIST_MEMBERS (UgRelation)
//	UgRelation*			next;
//	UgRelation*			prev;
//	guint				index;			// use index when program save/load file.
//	UgCategoryHints		hints;
//	UgPlugin*			plugin;			// plug-in
//	struct {
//		UgWatchFunc		func;			// watch
//		gpointer		data;
//	} watch;
};


#ifdef __cplusplus
}
#endif

#endif  // UG_CATEGORY_H

