/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <stdlib.h>
#include <string.h>
#include <uglib.h>
#include <ug_data_download.h>
#include <ug_category-gtk.h>
#include <glib/gi18n.h>

// static functions
static void		ug_category_gtk_init     (UgCategoryGtk* cgtk);
static void		ug_category_gtk_finalize (UgCategoryGtk* cgtk);
static gboolean	ug_category_gtk_watch (UgCategoryGtk* ccmd, UgDataset* dataset, UgMessage* message, gpointer user_data);
// Function used by GtkTreeModelSort.
static gint		ug_download_model_compare (GtkTreeModel* model, GtkTreeIter* a, GtkTreeIter* b, gpointer user_data);
// Function used by GtkTreeModelFilter.
static gboolean	ug_download_model_filter_category (GtkTreeModel *model, GtkTreeIter *iter, UgCategoryGtk* category);
static gboolean	ug_download_model_filter_queuing  (GtkTreeModel *model, GtkTreeIter *iter, gpointer data);
static gboolean	ug_download_model_filter_finished (GtkTreeModel *model, GtkTreeIter *iter, gpointer data);
static gboolean	ug_download_model_filter_recycled (GtkTreeModel *model, GtkTreeIter *iter, gpointer data);

// define UgDataEntry for UgDataClass
static UgDataEntry	category_gtk_data_entry[] =
{
	UG_CATEGORY_DATA_ENTRY,
	{NULL},			// null-terminated
};

static UgDataClass	category_gtk_data_class =
{
	UG_CATEGORY_CLASS_NAME,		// name
	NULL,						// reserve
	sizeof (UgCategoryGtk),		// instance_size
	category_gtk_data_entry,	// entry

	(UgInitFunc)     ug_category_gtk_init,
	(UgFinalizeFunc) ug_category_gtk_finalize,
	(UgAssignFunc)   ug_category_assign,
};
// extern
const	UgDataClass*	UgCategoryGtkClass = &category_gtk_data_class;

static void ug_category_gtk_init (UgCategoryGtk* cgtk)
{
	GtkTreeView*		view;
	GtkTreeViewColumn*	column;

	// UgCategory
	ug_category_init ((UgCategory*) cgtk);
	cgtk->watch.func = (UgWatchFunc) ug_category_gtk_watch;
	// UgCategoryGtk
	g_queue_init (&cgtk->active);
	ug_download_widget_init (&cgtk->all,      NULL);
	ug_download_widget_init (&cgtk->queuing,  NULL);
	ug_download_widget_init (&cgtk->finished, NULL);
	ug_download_widget_init (&cgtk->recycled, NULL);

	// All
	view = cgtk->all.view;
	ug_download_view_use_all_icon (view, TRUE);
	// Finished column
	view = cgtk->finished.view;
	column = gtk_tree_view_get_column (view, UG_DOWNLOAD_COLUMN_COMPLETE);
	gtk_tree_view_column_set_visible (column, FALSE);
	column = gtk_tree_view_get_column (view, UG_DOWNLOAD_COLUMN_PERCENT);
	gtk_tree_view_column_set_visible (column, FALSE);
	column = gtk_tree_view_get_column (view, UG_DOWNLOAD_COLUMN_LEFT);
	gtk_tree_view_column_set_visible (column, FALSE);
	column = gtk_tree_view_get_column (view, UG_DOWNLOAD_COLUMN_SPEED);
	gtk_tree_view_column_set_visible (column, FALSE);
	// Recycled column
	view = cgtk->recycled.view;
	column = gtk_tree_view_get_column (view, UG_DOWNLOAD_COLUMN_ELAPSED);
	gtk_tree_view_column_set_visible (column, FALSE);
	column = gtk_tree_view_get_column (view, UG_DOWNLOAD_COLUMN_LEFT);
	gtk_tree_view_column_set_visible (column, FALSE);
	column = gtk_tree_view_get_column (view, UG_DOWNLOAD_COLUMN_SPEED);
	gtk_tree_view_column_set_visible (column, FALSE);
}

static void ug_category_gtk_finalize (UgCategoryGtk* cgtk)
{
	UgDataset*		dataset;
	GtkTreeModel*	model;
	GtkTreeIter		iter;

	// UgCategory
	ug_category_finalize ((UgCategory*) cgtk);
	// free UgDownloadWidget
	ug_download_widget_finalize (&cgtk->all);
	ug_download_widget_finalize (&cgtk->queuing);
	ug_download_widget_finalize (&cgtk->finished);
	ug_download_widget_finalize (&cgtk->recycled);
	// free all jobs
	model = cgtk->filter;
	while (gtk_tree_model_get_iter_first (model, &iter)) {
		gtk_tree_model_get (model, &iter, 0, &dataset, -1);
		ug_category_gtk_remove (cgtk, dataset);
	}
	// free others
	if (cgtk->store)	// category->primary == NULL
		g_object_unref (cgtk->store);
	else if (cgtk->filter)
		g_object_unref (cgtk->filter);
	g_queue_clear (&cgtk->active);
}

UgCategoryGtk*	ug_category_gtk_new (UgCategoryGtk* primary)
{
	UgCategoryGtk*		category;

	category = ug_data_new (&category_gtk_data_class);
	ug_category_gtk_set_primary (category, primary);
	return category;
}

void	ug_category_gtk_free (UgCategoryGtk* category)
{
	ug_data_free (category);
}

void	ug_category_gtk_set_primary (UgCategoryGtk* category, UgCategoryGtk* primary)
{
	// clear
	if (category->store)	// category->primary == NULL
		g_object_unref (category->store);
	else if (category->filter)
		g_object_unref (category->filter);

	category->primary = primary;
	if (primary == NULL) {
		category->store = gtk_list_store_new (1, G_TYPE_POINTER);
		category->filter = GTK_TREE_MODEL (category->store);
	}
	else {	// primary == NULL
		category->store = NULL;
		category->filter = gtk_tree_model_filter_new (
				GTK_TREE_MODEL (primary->store), NULL);
		gtk_tree_model_filter_set_visible_func (GTK_TREE_MODEL_FILTER (category->filter),
				(GtkTreeModelFilterVisibleFunc) ug_download_model_filter_category,
				category, NULL);
	}

	ug_download_widget_set_sortable (&category->all, category->filter,
			ug_download_model_compare, NULL);
	ug_download_widget_set_filter (&category->queuing, category->filter,
			ug_download_model_filter_queuing, NULL);
	ug_download_widget_set_filter (&category->finished, category->filter,
			ug_download_model_filter_finished, NULL);
	ug_download_widget_set_filter (&category->recycled, category->filter,
			ug_download_model_filter_recycled, NULL);
}

void	ug_category_gtk_add (UgCategoryGtk* category, UgDataset* dataset)
{
	UgCategoryGtk*		primary;
	UgRelationGtk*		relation;
	GtkTreeModel*		model;
	GtkTreePath*		path;

	// add and set UgRelationGtkClass to dataset
	UG_DATASET_RELATION_CLASS (dataset) = UgRelationGtkClass;
	relation = UG_DATASET_RELATION_GTK (dataset);
	if (relation == NULL)
		relation = ug_dataset_alloc_front (dataset, UgRelationGtkClass);

	primary = (category->primary) ? category->primary : category;
	if (relation->category == NULL  &&  category != primary) {
		relation->category = category;
		if (relation->primary) {
			model = GTK_TREE_MODEL (relation->primary->store);
			path = gtk_tree_model_get_path (model, &relation->iter);
			gtk_tree_model_row_changed (model, path, &relation->iter);
			gtk_tree_path_free (path);
		}
	}
	if (relation->primary == NULL  &&  primary) {
		relation->primary = primary;
		relation->hints &= ~UG_HINT_ACTIVE;
		gtk_list_store_append (primary->store, &relation->iter);
		gtk_list_store_set (primary->store, &relation->iter, 0, dataset, -1);
		ug_dataset_ref (dataset);
	}
}

void	ug_category_gtk_remove (UgCategoryGtk* category, UgDataset* dataset)
{
	UgRelationGtk*		relation;

	relation = UG_DATASET_RELATION_GTK (dataset);
	if (relation->category)
		ug_category_gtk_stop (relation->category, dataset);
	if (relation->primary)
		gtk_list_store_remove (relation->primary->store, &relation->iter);
	relation->primary = NULL;
	relation->category = NULL;
	// delete data and files
	ug_download_temp_delete (dataset);
	ug_dataset_unref (dataset);
}

void	ug_category_gtk_switch (UgCategoryGtk* category, UgDataset* dataset, UgCategoryHints hint)
{
	UgRelationGtk*		relation;
	GtkListStore*		store;
	GtkTreeModel*		model;
	GtkTreePath*		path;

	relation = UG_DATASET_RELATION_GTK (dataset);
	store = relation->primary->store;

	if (hint & UG_HINT_RECYCLED) {
		gtk_list_store_move_after (store, &relation->iter, NULL);
		relation->hints &= ~UG_HINT_FINISHED;
		relation->hints |=  UG_HINT_RECYCLED;
	}
	else if (hint & UG_HINT_FINISHED) {
		gtk_list_store_move_after (store, &relation->iter, NULL);
		relation->hints &= ~UG_HINT_RECYCLED;
		relation->hints |=  UG_HINT_FINISHED;
	}
	else {
		gtk_list_store_move_before (store, &relation->iter, NULL);
		relation->hints &= ~(UG_HINT_FINISHED | UG_HINT_RECYCLED);
	}

	model = GTK_TREE_MODEL (store);
	path = gtk_tree_model_get_path (model, &relation->iter);
	gtk_tree_model_row_changed (model, path, &relation->iter);
	gtk_tree_path_free (path);
}

void	ug_category_gtk_clear (UgCategoryGtk* category, UgCategoryHints hint, guint from_nth)
{
	UgDataset*		dataset;
	GtkTreeModel*	model;
	GtkTreeIter		iter;
	GList*			list;
	GList*			link;
	gboolean		valid;

	if (hint & UG_HINT_RECYCLED)
		model = category->recycled.model;
	else if (hint & UG_HINT_FINISHED)
		model = category->finished.model;
	else
		model = category->queuing.model;

	// get jobs to clear
	list = NULL;
	valid = gtk_tree_model_iter_nth_child (model, &iter, NULL, from_nth);
	while (valid) {
		gtk_tree_model_get (model, &iter, 0, &dataset, -1);
		list = g_list_prepend (list, dataset);
		valid = gtk_tree_model_iter_next (model, &iter);
	}
	// remove jobs
	for (link = list;  link;  link = link->next)
		ug_category_gtk_remove (category, link->data);
	g_list_free (list);
}

void	ug_category_gtk_move_to (UgCategoryGtk* category, UgDataset* dataset, UgCategoryGtk* category_dest)
{
	UgRelationGtk*		relation;

	relation = UG_DATASET_RELATION_GTK (dataset);
	if (relation->category == category_dest)
		return;

	// stop active job in secondary category
	ug_category_gtk_stop (category, dataset);
	// move
	relation->category = category_dest;
	ug_category_gtk_switch (category_dest, dataset, relation->hints);
}

GList*	ug_category_gtk_get_list (UgCategoryGtk* category)
{
	UgDataset*		dataset;
	GtkTreeModel*	model;
	GtkTreeIter		iter;
	GList*			list;
	gboolean		valid;

	list = NULL;
	model = category->filter;
	valid = gtk_tree_model_get_iter_first (model, &iter);
	while (valid) {
		gtk_tree_model_get (model, &iter, 0, &dataset, -1);
		list = g_list_prepend (list, dataset);
		valid = gtk_tree_model_iter_next (model, &iter);
	}
	return g_list_reverse (list);
}

gboolean	ug_category_gtk_move_selected_up (UgCategoryGtk* category, UgDownloadWidget* dwidget)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	UgRelationGtk*	relation_prev;
	GtkTreeIter		iter;
	GtkTreePath*	path;
	GList*			list;
	GList*			link;
	gint			index;
	gint			index_prev;
	gboolean		result = FALSE;

	index_prev = -1;
	list = ug_download_widget_get_selected_indices (dwidget);
	// scroll to first item
	if (list) {
		index = GPOINTER_TO_INT (list->data);
		if (index > 0)
			index--;
		path = gtk_tree_path_new_from_indices (index, -1);
		gtk_tree_view_scroll_to_cell (dwidget->view, path, NULL, FALSE, 0, 0);
		gtk_tree_path_free (path);
	}
	// move
	for (link = list;  link;  link = link->next) {
		index = GPOINTER_TO_INT (link->data);
		if (index == index_prev+1) {
			index_prev++;
			continue;
		}
		index_prev = index - 1;
		// get previous
		gtk_tree_model_iter_nth_child (dwidget->model, &iter, NULL, index_prev);
		gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
		relation_prev = UG_DATASET_RELATION_GTK (dataset);
		// get current
		gtk_tree_model_iter_next (dwidget->model, &iter);
		gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
		relation = UG_DATASET_RELATION_GTK (dataset);
		// swap
		gtk_list_store_swap (relation->primary->store,
				&relation->iter, &relation_prev->iter);
		result = TRUE;
	}

	g_list_free (list);
	return result;
}

gboolean	ug_category_gtk_move_selected_down (UgCategoryGtk* category, UgDownloadWidget* dwidget)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	UgRelationGtk*	relation_next;
	GtkTreeIter		iter;
	GtkTreePath*	path;
	GList*			list;
	GList*			link;
	gint			index;
	gint			index_next;
	gboolean		result = FALSE;

	index_next = gtk_tree_model_iter_n_children (dwidget->model, NULL);
	list = ug_download_widget_get_selected_indices (dwidget);
	link = g_list_last (list);
	// scroll to last item
	if (link) {
		index = GPOINTER_TO_INT (link->data);
		if (index < index_next -1)
			index++;
		path = gtk_tree_path_new_from_indices (index, -1);
		gtk_tree_view_scroll_to_cell (dwidget->view, path, NULL, FALSE, 0, 0);
		gtk_tree_path_free (path);
	}
	// move
	for (;  link;  link = link->prev) {
		index = GPOINTER_TO_INT (link->data);
		if (index == index_next-1) {
			index_next--;
			continue;
		}
		index_next = index + 1;
		// get current
		gtk_tree_model_iter_nth_child (dwidget->model, &iter, NULL, index);
		gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
		relation = UG_DATASET_RELATION_GTK (dataset);
		// get next
		gtk_tree_model_iter_next (dwidget->model, &iter);
		gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
		relation_next = UG_DATASET_RELATION_GTK (dataset);
		// swap
		gtk_list_store_swap (relation->primary->store,
				&relation->iter, &relation_next->iter);
		result = TRUE;
	}

	g_list_free (list);
	return result;
}

gboolean	ug_category_gtk_move_selected_to_top (UgCategoryGtk* category, UgDownloadWidget* dwidget)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	UgRelationGtk*	relation_top;
	GtkTreeIter		iter;
	GList*			list;
	GList*			link;
	gint			index;
	gint			index_top;

	// get movable jobs
	relation_top = NULL;
	index_top = 0;
	list = ug_download_widget_get_selected_indices (dwidget);
	for (link=list;  link;  link=link->next, index_top++) {
		index = GPOINTER_TO_INT (link->data);
		if (index == index_top) {
			link->data = NULL;
			continue;
		}
		else {
			gtk_tree_model_iter_nth_child (dwidget->model, &iter, NULL, index);
			gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
			link->data = UG_DATASET_RELATION_GTK (dataset);
		}
		if (relation_top == NULL) {
			gtk_tree_model_iter_nth_child (dwidget->model, &iter, NULL, index_top);
			gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
			relation_top = UG_DATASET_RELATION_GTK (dataset);
		}
	}
	list = g_list_remove_all (list, NULL);
	if (list == NULL)
		return FALSE;

	// move to top
	for (link = list;  link;  link = link->next) {
		relation = link->data;
		gtk_list_store_move_before (relation->primary->store,
				&relation->iter, &relation_top->iter);
	}
	g_list_free (list);

	// scroll to top
	gtk_tree_view_scroll_to_point (dwidget->view, -1, 0);
	return TRUE;
}

gboolean	ug_category_gtk_move_selected_to_bottom (UgCategoryGtk* category, UgDownloadWidget* dwidget)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	UgRelationGtk*	relation_bottom;
	GtkTreeIter		iter;
	GtkTreePath*	path;
	GList*			list;
	GList*			link;
	gint			index;
	gint			index_bottom;

	// get movable jobs
	relation_bottom = NULL;
	index_bottom = gtk_tree_model_iter_n_children (dwidget->model, NULL) - 1;
	list = ug_download_widget_get_selected_indices (dwidget);
	list = g_list_reverse (list);
	for (link=list;  link;  link=link->next, index_bottom--) {
		index = GPOINTER_TO_INT (link->data);
		if (index == index_bottom) {
			link->data = NULL;
			continue;
		}
		else {
			gtk_tree_model_iter_nth_child (dwidget->model, &iter, NULL, index);
			gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
			link->data = UG_DATASET_RELATION_GTK (dataset);
		}
		if (relation_bottom == NULL) {
			gtk_tree_model_iter_nth_child (dwidget->model, &iter, NULL, index_bottom);
			gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
			relation_bottom = UG_DATASET_RELATION_GTK (dataset);
		}
	}
	list = g_list_remove_all (list, NULL);
	if (list == NULL)
		return FALSE;

	// move to bottom
	for (link = list;  link;  link = link->next) {
		relation = link->data;
		gtk_list_store_move_after (relation->primary->store,
				&relation->iter, &relation_bottom->iter);
	}
	g_list_free (list);

	// scroll to bottom
	index = gtk_tree_model_iter_n_children (dwidget->model, NULL) -1;
	path = gtk_tree_path_new_from_indices (index, -1);
	gtk_tree_view_scroll_to_cell (dwidget->view, path, NULL, FALSE, 0, 0);
	gtk_tree_path_free (path);
	return TRUE;
}

gboolean	ug_category_gtk_start (UgCategoryGtk* category, UgDataset* dataset)
{
	if (ug_category_start ((UgCategory*) category, dataset)) {
		if (g_queue_find (&category->active, dataset) == NULL)
			g_queue_push_tail (&category->active, dataset);
		return TRUE;
	}
	return FALSE;
}

void	ug_category_gtk_stop (UgCategoryGtk* category, UgDataset* dataset)
{
	ug_category_stop ((UgCategory*) category, dataset);
	g_queue_remove (&category->active, dataset);
}

// returned value
// FALSE: status no changed.
// TRUE:  status changed. (start, stop, move, remove)
gboolean	ug_category_gtk_refresh (UgCategoryGtk* cgtk, GList** completed)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	GList*			link;
	gboolean		result = FALSE;

	for (link = cgtk->active.head;  link;  link = link->next) {
		dataset = link->data;
		// refresh status & data
		if (ug_category_refresh ((UgCategory*) cgtk, dataset))
			continue;
		// stop & clear
		ug_category_stop ((UgCategory*) cgtk, dataset);
		link->data = NULL;
		// relation
		relation = UG_DATASET_RELATION_GTK (dataset);
		result = TRUE;
		if (relation->hints & UG_HINT_FINISHED) {
			ug_category_gtk_switch (cgtk, dataset, UG_HINT_FINISHED);
			if (completed)
				*completed = g_list_prepend (*completed, dataset);
		}
	}
	g_queue_remove_all (&cgtk->active, NULL);

	return result;
}

gboolean	ug_category_gtk_activate (UgCategoryGtk* cgtk)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	GtkTreeIter		iter;
	gboolean		valid;
	gboolean		result = FALSE;

	valid = gtk_tree_model_get_iter_first (cgtk->queuing.model, &iter);
	while (valid  &&  cgtk->active.length < cgtk->limit.active) {
		gtk_tree_model_get (cgtk->queuing.model, &iter, 0, &dataset, -1);
		valid = gtk_tree_model_iter_next (cgtk->queuing.model, &iter);
		// check & start
		relation = UG_DATASET_RELATION_GTK (dataset);
		if (relation->hints & (UG_HINT_UNRUNNABLE | UG_HINT_ACTIVE))
			continue;
		if (ug_category_gtk_start (cgtk, dataset) == FALSE) {
			g_free (relation->message);
			relation->message_type = UG_MESSAGE_ERROR;
			relation->message = g_strdup (_("Unsupported scheme (protocol)."));
		}
		result = TRUE;
	}
	return result;
}

gboolean	ug_category_gtk_clear_excess (UgCategoryGtk* cgtk)
{
	guint		n;
	gboolean	result = FALSE;

	n = gtk_tree_model_iter_n_children (cgtk->finished.model, NULL);
	if (n > cgtk->limit.finished)
		result = TRUE;
	n = gtk_tree_model_iter_n_children (cgtk->recycled.model, NULL);
	if (n > cgtk->limit.recycled)
		result = TRUE;

	ug_category_gtk_clear (cgtk, UG_HINT_FINISHED, cgtk->limit.finished);
	ug_category_gtk_clear (cgtk, UG_HINT_RECYCLED, cgtk->limit.recycled);

	return result;
}

// ----------------------------------------------
// static function
static gboolean ug_category_gtk_watch (UgCategoryGtk* ccmd, UgDataset* dataset, UgMessage* message, gpointer user_data)
{
	UgRelationGtk*	relation;

	relation = UG_DATASET_RELATION_GTK (dataset);

	switch (message->type) {
	case UG_MESSAGE_ERROR:
	case UG_MESSAGE_WARNING:
		g_free (relation->message);
		relation->message = g_strdup (message->string);
		relation->message_type = message->type;
		break;

	default:
		break;
	}
	return FALSE;
}

// Function used by GtkTreeModelSort.
static gint	ug_download_model_compare (GtkTreeModel* model, GtkTreeIter* a, GtkTreeIter* b, gpointer user_data)
{
	UgDataset*		dataset;
	UgDataCommon*	common;
	gchar*			name1;
	gchar*			name2;

	gtk_tree_model_get (model, a, 0, &dataset, -1);
	if (dataset == NULL)
		return 0;
	common = UG_DATASET_COMMON (dataset);
	if (common->name)
		name1 = common->name;
	else if (common->file)
		name1 = common->file;
	else
		name1 = "unnamed";

	gtk_tree_model_get (model, b, 0, &dataset, -1);
	if (dataset == NULL)
		return 0;
	common = UG_DATASET_COMMON (dataset);
	if (common->name)
		name2 = common->name;
	else if (common->file)
		name2 = common->file;
	else
		name2 = "unnamed";
	return strcmp (name1, name2);
}

// Function used by GtkTreeModelFilter.
static gboolean	ug_download_model_filter_category (GtkTreeModel *model, GtkTreeIter *iter, UgCategoryGtk* category)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	if (dataset) {
		relation = UG_DATASET_RELATION_GTK (dataset);
		if (relation->category == category)
			return TRUE;
	}
	return FALSE;
}

static gboolean	ug_download_model_filter_queuing  (GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	UgDataset*		dataset;
	UgRelation*		relation;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	if (dataset) {
		relation = UG_DATASET_RELATION (dataset);
		if (relation->hints & (UG_HINT_FINISHED | UG_HINT_RECYCLED))
			return FALSE;
	}
	return TRUE;
}

// Function used by GtkTreeModelFilter.
static gboolean	ug_download_model_filter_finished (GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	UgDataset*		dataset;
	UgRelation*		relation;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	if (dataset) {
		relation = UG_DATASET_RELATION (dataset);
		if (relation->hints & UG_HINT_FINISHED)
			return TRUE;
	}
	return FALSE;
}

// Function used by GtkTreeModelFilter.
static gboolean	ug_download_model_filter_recycled (GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	UgDataset*		dataset;
	UgRelation*		relation;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	if (dataset) {
		relation = UG_DATASET_RELATION (dataset);
		if (relation->hints & UG_HINT_RECYCLED)
			return TRUE;
	}
	return FALSE;
}


// ----------------------------------------------------------------------------
// UgRelationGtk
//
static void	ug_relation_gtk_assign (UgRelation* relation, UgRelation* src);

static UgDataEntry	relation_gtk_data_entry[] =
{
	UG_RELATION_DATA_ENTRY,
	{NULL},			// null-terminated
};

static UgDataClass	relation_gtk_data_class =
{
	UG_RELATION_CLASS_NAME,		// name
	NULL,						// reserve
	sizeof (UgRelationGtk),		// instance_size
	relation_gtk_data_entry,	// entry

	(UgInitFunc)     NULL,
	(UgFinalizeFunc) NULL,
	(UgAssignFunc)   ug_relation_gtk_assign,
};
// extern
const	UgDataClass*	UgRelationGtkClass = &relation_gtk_data_class;

static void	ug_relation_gtk_assign (UgRelation* relation, UgRelation* src)
{
	if (src->hints & UG_HINT_PAUSED) {
		relation->hints |=  UG_HINT_PAUSED;
		relation->hints &= ~UG_HINT_ACTIVE;
	}
	else
		relation->hints &= ~UG_HINT_PAUSED;
}

