;;; property.scm: Handles properties of input method conveniently
;;;
;;; Copyright (c) 2003,2004 uim Project http://uim.freedesktop.org/
;;;
;;; All rights reserved.
;;;
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above copyright
;;;    notice, this list of conditions and the following disclaimer in the
;;;    documentation and/or other materials provided with the distribution.
;;; 3. Neither the name of authors nor the names of its contributors
;;;    may be used to endorse or promote products derived from this software
;;;    without specific prior written permission.
;;;
;;; THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
;;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
;;; FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
;;; OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
;;; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
;;; LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
;;; OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
;;; SUCH DAMAGE.
;;;;

;; This file is not yet used by current uim implementation, and not
;; yet tested. This is maybe used in next next release.
;;
;; This preliminary property handling framework will eliminates all of
;; property related handlings such as im-update-prop-list or
;; im-update-prop-label from each input method. Input method developer
;; can forget about the property (and legacy 'mode') feature except
;; for register-property, register-im and init-handler.
;;
;; Input method developer must not use all procedures or records in
;; this file except for the 3 procedures described below. This is
;; required to ensure isolation of the property feature from uim
;; core. It is important for future extensibility and wide-range
;; platform support. Please satisfy following conditions.

;; - Use only following 3 procedures.
;;   * register-property
;;   * context-prop-activate-handler
;;   * context-mode-handler
;;
;; - Don't insert (require "property.scm") in your input method
;;   file. All necessary procedures described above are appropriately
;;   prepared in im.scm.
;;
;; - Name the property used for choose input mode as
;;   "prop_*_input_mode" (replace "*" with your input method name such
;;   as "your_im"). This naming convention is used to display only the
;;   primary input state for user by some helper applets, or support a
;;   legacy 'mode' API.
;;
;; -- 2004-10-08 YamaKen

(require "util.scm")

(define property-definition-list ())

(define property-definition-rec-spec
  (list
   (list 'id          #f)
   (list 'preprocess  list)
   (list 'postprocess list)
   (list 'states      ())))
(define-record 'property-definition property-definition-rec-spec)

(define register-property
  (lambda prop-def
    (let* ((id (property-definition-id prop-def))
	   (prev-def (assq id property-definition-list)))
      (if prev-def
	  (set-cdr! prev-def (cdr prop-def))
	  (set! property-definition-list
		(cons prop-def property-definition-list))))))

(define-record 'prop-state
  (list
   (list 'id               #f) ;; must be first member
   (list 'iconic-label     "")
   (list 'label            "")
   (list 'short-desc       "")
   (list 'predicate        list)
   (list 'activate-handler list)))

(define-record 'property
  (append
   property-definition-rec-spec
   (list
    (list 'current-state #f)
    (list 'owner         #f))))
(define property-new-internal property-new)

;; API for uim developers
(define property-new
  (lambda (owner prop-id)
    (let ((prop-def (assq prop-id property-definition-list)))
      (if prop-def
	  (let* ((states (property-definition-states prop-def))
		 (default-state-sym (symbolconc 'default- prop-id))
		 (initial-state (if (symbol-bound? default-state-sym)
				    (assq (symbol-value default-state-sym)
					  states)
				    (car states)))
		 (init-list (append prop-def
				    (list initial-state owner)))
		 (prop (apply property-new-internal init-list)))
	    (property-activate-state! prop initial-state)
	    prop)
	  #f))))

(define property-next-state
  (lambda (prop)
    (let* ((owner (property-owner prop))
	   (old-state-id (prop-state-id (property-current-state prop)))
	   (next-state? (lambda (state)
			  (let ((here? (prop-state-predicate state)))
			    (here? owner old-state-id))))
	   (candidates (filter next-state?
			       (property-states prop))))
      (if (= (length candidates)
	     1)
	  (car candidates)
	  (if (>= (verbose)
		  5)
	      (print (property-debug-message prop
					     "property-next-state"
					     (if (null? candidates)
						 "no possible next state found"
						 "ambiguous next state")))
	      (property-current-state prop))))))

(define property-will-state-transit?
  (lambda (prop)
    (not (eq? (prop-state-id (property-current-state prop))
	      (prop-state-id (property-next-state prop))))))

;; API for uim developers
(define property-transit-state!
  (lambda (prop)
    (and (property-will-state-transit? prop)
	 (begin
	   (property-set-current-state! prop (property-next-state prop))
	   #t))))

;; API for uim developers
;; state transition is not yet occurred at end of this procedure
(define property-activate-state!
  (lambda (prop state)
    (and state
	 (let* ((activate-handler (prop-state-activate-handler state))
		(preprocess (property-preprocess prop))
		(postprocess (property-postprocess prop))
		(owner (property-owner prop))
		(old-state-id (prop-state-id (property-current-state prop))))
	   (and preprocess
		(preprocess owner old-state-id))
	   (and activate-handler
		(activate-handler owner old-state-id))
	   (and postprocess
		(postprocess owner old-state-id))
	   #t))))

(define property-debug-message
  (lambda (prop location defect)
    (let* ((prop-id (property-id prop))
	   (prop-id-str (symbol->string prop-id)))
      (string-append
       defect " in " location ". debug " prop-id-str "."))))


;;
;; property message handlings
;;

;; See doc/HELPER-PROTOCOL for the protocol specification

(define prop-state-compose-label
  (lambda (state)
    (string-append (prop-state-iconic-label state) "\t"
		   (prop-state-label state) "\n")))

(define prop-state-compose-branch
  (lambda (state)
    (string-append "branch\t"
		   (prop-state-compose-label state))))

(define prop-state-compose-leaf
  (lambda (state active?)
    (string-append "leaf\t"
		   (prop-state-iconic-label state) "\t"
		   (prop-state-label state) "\t"
		   (prop-state-short-desc state) "\t"
		   (symbol->string (prop-state-id state)) "\t"
		   (if active?
		       "*\n"
		       "\n"))))

(define property-compose-live-branch
  (lambda (prop)
    (let* ((current-state (property-current-state prop))
	   (branch (prop-state-compose-branch current-state))
	   (leaves (map (lambda (state)
			  (let ((active? (eq? (prop-state-id state)
					      (prop-state-id current-state))))
			    (prop-state-compose-leaf state active?)))
			(property-states prop))))
      (apply string-append (cons branch leaves)))))


;; following code change will be applied to im.scm
;;
;; - prop-handler will be renamed to prop-activate-handler
;;
;;(if (and (symbol-bound? disable-property)
;;	 disable-property)
;;    (begin
;;      (define do-nothing
;;	(lambda args
;;	  #f))
;;      (define register-property do-nothing)
;;      (define context-init-prop-list do-nothing)
;;      (define property-transit-state! do-nothing)
;;      (define context-propagate-prop-state do-nothing))
;;      (define context-prop-activate-handler do-nothing)
;;      (define context-mode-handler do-nothing))
;;    (require "property.scm"))
;;
;;(define context-rec-spec
;;  '((id        #f)  ;; must be first member
;;    (im        #f)
;;    (prop-list ()))
;;
;; create-context:
;;       (let* ((handler (im-init-handler im))
;;              (c (handler id im arg)))
;;+        (context-init-prop-list c (context-prop-list c))
;;         (register-context c)))))
;;
;; (define invoke-handler
;;   (lambda args
;;     (let* ((handler-reader (car args))
;;            (id (cadr args))
;;            (c (find-context id))
;;            (handler-args (cons c (cddr args)))
;;            (im (and c (context-im c)))
;;-           (handler (and im (handler-reader im))))
;;-      (if handler
;;-          (apply handler handler-args)))))
;;+           (handler (and im (handler-reader im)))
;;+           (result (and handler
;;+                        (apply handler handler-args)))
;;+           (transition-occurred? (find property-transit-state!
;;+                                       (context-prop-list c))))
;;+      (if transition-occurred?
;;+          (context-propagate-prop-state c))
;;+      result)))

;; API for uim developers
(define context-init-prop-list
  (lambda (context prop-id-list)
    (let ((prop-list (map (lambda (prop-id)
			    (property-new prop-id context))
			  prop-id-list)))
      (context-set-prop-list! context prop-list)
      (context-propagate-prop-configuration context))))

(define context-propagate-prop-label-update
  (lambda (context)
    (let* ((prop-list (context-prop-list context))
	   (active-label (lambda (prop)
			   (let ((state (property-current-state prop)))
			     (prop-state-compose-label state))))
	   (labels (map active-label
			prop-list))
	   (active-state-labels (apply string-append labels)))
      (im-update-prop-label context
			    active-state-labels))))

(define context-propagate-prop-list-update
  (lambda (context)
    (let* ((prop-list (context-prop-list context))
	   (branches (map property-compose-live-branch
			  prop-list))
	   (prop-config-tree (apply string-append branches)))
      (im-update-prop-list context
			   prop-config-tree))))

;; API for uim developers
(define context-propagate-prop-state
  (lambda (context)
    (context-propagate-prop-label-update context)
    ;; Sending prop_list every time costs all uim participant
    ;; processes slightly heavy resource consumptions. Although it is
    ;; not a problem for the rich desktop environment today, we should
    ;; also consider resource sensitive embedded environments if it is
    ;; not hard.
    ;;
    ;; We should adopt another message to send lightweight status
    ;; update (such as prop-id and state-id pair), and revise
    ;; prop_list as initialization message (i.e. remove the flag
    ;; field) -- 2004-10-08 YamaKen
    (context-propagate-prop-list-update context)
    (context-update-mode context)))

;; API for uim developers
(define context-propagate-prop-configuration
  (lambda (context)
    (context-propagate-prop-list-update context)
    (context-update-mode-list context)))

;; API for input method developers
;; ready to use for register-im
(define context-prop-activate-handler
  (lambda (context message)
    (let* ((prop-list (context-property-list context))
	   (destination-id (string->symbol message))
	   (activate! (lambda (prop)
			(let* ((states (property-states prop))
			       (dest (assq destination-id states)))
			  (property-activate-state! prop dest)))))
      (find activate! prop-list))))


;;
;; legacy 'mode' handlings for backward compatibility
;;

;; find the property that has "_input_mode" suffix
(define prop-list-find-mode-prop
  (lambda (prop-list)
    (let* ((prop-ids (map property-id prop-list))
	   (extract-suffix (lambda (prop-id)
			     (let* ((as-str (symbol->string prop-id))
				    (rev-words (reverse
						(string-split as-str "_"))))
			       (list (nth 1 rev-words)
				     (nth 0 rev-words)))))
	   (mode-prop? (lambda (suffix)
			 (equal? suffix
				 '("input" "mode")))))
      (or (find mode-prop? prop-ids)
	  ()))))

(define property-prop-state-id->mode-value
  (lambda (mode-prop state-id)
    (let ((index (lambda (val lst)
		   (- (length (memq val
				    (reverse lst)))
		      1)))
	  (state-ids (map prop-state-id
			  (property-states mode-prop))))
      (index state-id state-ids))))

(define property-mode-value->prop-state-id
  (lambda (mode-prop mode)
    (let ((state-ids (map prop-state-id
			  (property-states mode-prop))))
      (nth mode state-ids))))

;; don't invoke directly. use context-propagate-prop-state instead
(define context-update-mode
  (lambda (context)
    (let* ((prop-list (context-property-list context))
	   (mode-prop (property-find-mode-prop prop-list))
	   (current-state (property-current-state mode-prop))
	   (current-state-id (prop-state-id current-state))
	   (mode (property-prop-state-id->mode-value current-state-id)))
      (im-update-mode context mode))))

;; don't invoke directly. use context-propagate-prop-configuration instead
(define context-update-mode-list
  (lambda (context)
    (let* ((prop-list (context-property-list context))
	   (mode-prop (property-find-mode-prop prop-list))
	   (current-state-id (property-current-state mode-prop))
	   (mode (property-prop-state-id->mode-value current-state-id)))
      (im-clear-mode-list context)
      (for-each (lambda (state)
		  (im-pushback-mode-list context (prop-state-label state)))
		(property-states mode-prop))
      (im-update-mode-list context)
      (im-update-mode context mode))))

;; API for input method developers
;; ready to use for register-im
(define context-mode-handler
  (lambda (context mode)
    (let* ((prop-list (context-property-list context))
	   (mode-prop (property-find-mode-prop prop-list))
	   (state-id (property-mode-value->prop-state-id mode)))
      (property-activate! mode-prop state-id))))

;;
;; property definitions for anthy (will be moved into anthy.scm later)
;;

(require "anthy.scm")

;;; user configs

;; controls:
;; - what properties will be shown for user
;; - shown in what order
(define anthy-property-id-list '(prop_anthy_input_mode
				 prop_anthy_kana_input_method))

;; default value for each property
(define default-prop_anthy_input_mode 'prop_anthy_hiragana)
(define default-prop_anthy_kana_input_method 'prop_anthy_roma)

;;; property definitions
(register-property
 ;; property id
 'prop_anthy_input_mode
 ;; preprocess at prop-activate
 (lambda (ac old-state-id)
   (anthy-flush ac)
   (anthy-update-preedit ac))
 ;; postprocess at prop-activate
 (lambda (ac old-state-id)
   #f)
 ;; state definitions
 (list
  (prop-state-new 'prop_anthy_hiragana
		  ""
		  "Ҥ餬"
		  "Ҥ餬ϥ⡼"
		  ;; state predicate
		  (lambda (ac old-state-id)
		    (and (anthy-context-on ac)
			 (= (anthy-context-kana-mode ac)
			    anthy-type-hiragana)))
		  ;; prop-activate-handler
		  (lambda (ac old-state-id)
		    (anthy-context-set-on! ac #t)
		    (anthy-context-set-kana-mode! ac anthy-type-hiragana)))
  (prop-state-new 'prop_anthy_katakana
		  ""
		  ""
		  "ϥ⡼"
		  (lambda (ac old-state-id)
		    (and (anthy-context-on ac)
			 (= (anthy-context-kana-mode ac)
			    anthy-type-katakana)))
		  (lambda (ac old-state-id)
		    (anthy-context-set-on! ac #t)
		    (anthy-context-set-kana-mode! ac anthy-type-katakana)))
  (prop-state-new 'prop_anthy_hankana
		  ""
		  "Ⱦѥ"
		  "Ⱦѥϥ⡼"
		  (lambda (ac old-state-id)
		    (and (anthy-context-on ac)
			 (= (anthy-context-kana-mode ac)
			    anthy-type-hankana)))
		  (lambda (ac old-state-id)
		    (anthy-context-set-on! ac #t)
		    (anthy-context-set-kana-mode! ac anthy-type-hankana)))
  (prop-state-new 'prop_anthy_direct
		  "A"
		  "ľ"
		  "ľ(̵Ѵ)ϥ⡼"
		  (lambda (ac old-state-id)
		    (and (not (anthy-context-on ac))
			 (not (anthy-context-wide-latin ac))))
		  (lambda (ac old-state-id)
		    (anthy-context-set-on! ac #f)
		    (anthy-context-set-wide-latin! ac #f)))
  (prop-state-new 'prop_anthy_zenkaku
		  ""
		  "ѱѿ"
		  "ѱѿϥ⡼"
		  (lambda (ac old-state-id)
		    (and (not (anthy-context-on ac))
			 (anthy-context-wide-latin ac)))
		  (lambda (ac old-state-id)
		    (anthy-context-set-on! ac #f)
		    (anthy-context-set-wide-latin! ac #t)))))

(register-property
 'prop_anthy_kana_input_method
 (lambda (ac old-state-id)
   (anthy-flush ac)
   (anthy-update-preedit ac))
 (lambda (ac old-state-id)
   #f)
 (list
  (prop-state-new 'prop_anthy_roma
		  ""
		  "޻"
		  "޻ϥ⡼"
		  (lambda (ac old-state-id)
		    (= (anthy-context-input-rule ac)
		       anthy-input-rule-roma))
		  (lambda (ac old-state-id)
		    (rk-context-set-rule! (anthy-context-rkc ac)
					  ja-rk-rule)
		    (anthy-context-set-input-rule! ac anthy-input-rule-roma)))
  (prop-state-new 'prop_anthy_kana
		  ""
		  ""
		  "ϥ⡼"
		  (lambda (ac old-state-id)
		    (= (anthy-context-input-rule ac)
		       anthy-input-rule-kana))
		  (lambda (ac old-state-id)
		    (rk-context-set-rule! (anthy-context-rkc ac)
					  ja-kana-rule)
		    (anthy-context-set-input-rule! ac anthy-input-rule-kana)
		    ;;(define-key anthy-kana-toggle-key? "")
		    ;;(define-key anthy-latin-key? generic-on-key?)
		    ;;(define-key anthy-wide-latin-key? "")
		    ))))
