/*
 * unity-webapps-base-js-object.c
 * Copyright (C) Canonical LTD 2012
 * 
 * Author: Alexandre Abreu <alexandre.abreu@canonical.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <string.h>
#include <memory.h> // TODO remove

#include <glib.h>

#include "npapi-headers/headers/npfunctions.h"
#include "npapi-headers/headers/nptypes.h"
#include "npapi-headers/headers/npruntime.h"

#include "unity-webapps-base-js-object.h"
#include "unity-npapi-common.h"
#include "unity-npapi-debug.h"


static bool NPClass_HasMethod (NPObject *npobj, NPIdentifier name);

static NPObject * NPClass_Allocate (NPP npp, NPClass *aClass);

static bool NPClass_Invoke (NPObject *npobj
			    , NPIdentifier name
			    , const NPVariant *args
			    , uint32_t argCount
			    , NPVariant *result);

static bool NPClass_InvokeDefault (NPObject *npobj
				   , const NPVariant *args
				   , uint32_t argCount
				   , NPVariant *result);

static bool NPClass_HasProperty (NPObject * npobj
				 , NPIdentifier name);

static bool NPClass_GetProperty (NPObject *npobj
				 , NPIdentifier name
				 , NPVariant *result);

static bool NPClass_SetProperty (NPObject *npobj
				 , NPIdentifier name
				 , const NPVariant *value);

static bool NPClass_RemoveProperty (NPObject *npobj
				    , NPIdentifier name);

static bool NPClass_Enumerate (NPObject *npobj
			       , NPIdentifier **identifier
			       , uint32_t *count);

static bool NPClass_Construct (NPObject *npobj
			       , const NPVariant *args
			       , uint32_t argCount
			       , NPVariant *result);

static void NPClass_Deallocate (NPObject *npobj);

static void NPClass_Invalidate (NPObject *npobj);

static NPClass js_object_class = {
  .structVersion = NP_CLASS_STRUCT_VERSION,
  .allocate = NPClass_Allocate,
  .deallocate = NPClass_Deallocate,
  .invalidate = NPClass_Invalidate,
  .hasMethod = NPClass_HasMethod,
  .invoke = NPClass_Invoke,
  .invokeDefault = NPClass_InvokeDefault,
  .hasProperty = NPClass_HasProperty,
  .getProperty = NPClass_GetProperty,
  .setProperty = NPClass_SetProperty,
  .removeProperty = NPClass_RemoveProperty,
  .enumerate = NPClass_Enumerate,
  .construct = NPClass_Construct
};


static NPVariant
to_string_method (NPP instance
		  , NPObject * object
		  , const NPVariant args[]
		  , uint32_t argcnt);


///////////////////////////////////////////////////////
//
//	public
//
///////////////////////////////////////////////////////

NPObject *
unity_webapps_npapi_create_js_object (NPP instance, const char * const name)
{
  NPObject *
    pObject = NPN_CreateObject (instance, &js_object_class);
  g_return_val_if_fail (pObject != NULL, NULL);
  
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) pObject;
  
  pWrapper->methods = g_hash_table_new (g_str_hash, g_str_equal);
  pWrapper->properties = g_hash_table_new (g_str_hash, g_str_equal);
  
  pWrapper->name = g_strdup (name);
  pWrapper->instance = instance;
  
  unity_webapps_npapi_add_method (pWrapper, "toString", to_string_method);
  
  return pObject;
}


void
unity_webapps_npapi_add_array_support (UnityWebappsJavascriptObjectWrapper * pWrapper
				       , UnityWebAppsNpapiIndexedGetPtr access)
{
  g_return_if_fail (NULL != pWrapper);
  
  pWrapper->array_access = access;
}


void
unity_webapps_npapi_set_deallocator (UnityWebappsJavascriptObjectWrapper * pWrapper
				     , UnityWebappsDeallocatorPtr deallocator)
{
  g_return_if_fail (NULL != pWrapper);
  
  pWrapper->deallocator = deallocator;
}


// retains name param
void
unity_webapps_npapi_add_method (UnityWebappsJavascriptObjectWrapper * pWrapper
				, const char * const name
				, UnityWebAppsNpapiMethodPtr pMethod)
{
  g_return_if_fail (NULL != pWrapper);
  g_return_if_fail (NULL != name || 0 == strlen(name));
  g_return_if_fail (NULL != pMethod);
  g_return_if_fail (NULL != pWrapper->methods);
  
  g_hash_table_insert (pWrapper->methods, g_strdup (name), pMethod);
}


// retains name param
void
unity_webapps_npapi_add_property (UnityWebappsJavascriptObjectWrapper * pWrapper
				  , const char * const name
				  , UnityWebAppsNpapiPropertyGetPtr get
				  , UnityWebAppsNpapiPropertySetPtr set)
{
  g_return_if_fail (NULL != pWrapper);
  g_return_if_fail (NULL != name || 0 == strlen(name));
  g_return_if_fail (NULL != get || NULL != set);
  g_return_if_fail (NULL != pWrapper->properties);
  
  UnityWebAppsNpapiPropertyDescr
    * property = g_malloc0 (sizeof (UnityWebAppsNpapiPropertyDescr));
  g_return_if_fail (NULL != property);
  
  property->get = get;
  property->set = set;
  
  g_hash_table_insert (pWrapper->properties, g_strdup (name), property);
}




///////////////////////////////////////////////////////
//
//	private
//
///////////////////////////////////////////////////////


static void
free_property (UnityWebAppsNpapiPropertyDescr * property)
{
  g_return_if_fail (NULL != property);
  g_free (property);
}


static NPVariant
to_string_method (NPP instance
		  , NPObject * object
		  , const NPVariant args[]
		  , uint32_t argcnt)
{
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) object;
  
  // watch out w/ UTF8
  NPVariant result;
  NULL_TO_NPVARIANT (result);
  
  g_return_val_if_fail (NULL != pWrapper, result);
  
  static const char * const format = "[Object - %s]";
  
  char * object_descr = pWrapper->name != NULL ? pWrapper->name : "Unknown";
  
  gsize size = strlen(format) + 1 + strlen(object_descr) + 1;
  char * name = g_malloc0 (size);
  
  g_snprintf (name, size, format, object_descr);
  
  char * returnValue = NPN_MemAlloc (size);
  memset (returnValue, 0, size);
  strncpy (returnValue, name, strlen(name));
  
  STRINGZ_TO_NPVARIANT(returnValue, result);
  
  g_free (name);
  
  return result;
}


static NPObject *
NPClass_Allocate (NPP npp, NPClass *aClass)
{
  g_return_val_if_fail (NULL != npp, NULL);
  
  NPObject *
    pObject = g_malloc0 (sizeof (UnityWebappsJavascriptObjectWrapper));
  
  return pObject;
}


static void
NPClass_Deallocate (NPObject *npobj)
{
  g_return_if_fail (NULL != npobj);
  
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) npobj;
  
  // call custom deallocator if any
  if (NULL != pWrapper->deallocator)
    {
      pWrapper->deallocator (pWrapper->instance, npobj);
    }
  
  void free_one_method (gpointer key,
			gpointer value,
			gpointer user_data)
  {
    g_free (key);
  }
  g_hash_table_foreach (pWrapper->methods, free_one_method, NULL);
  g_hash_table_destroy (pWrapper->methods);
  
  void free_one_property (gpointer key,
			  gpointer value,
			  gpointer user_data)
  {
    g_free (key);
    free_property (value);
  }
  g_hash_table_foreach (pWrapper->properties, free_one_property, NULL);
  g_hash_table_destroy (pWrapper->properties);
  
  g_free (pWrapper->name);
  g_free (pWrapper);
}


static void
NPClass_Invalidate (NPObject *npobj)
{
  // TODO
}


static bool
NPClass_HasMethod (NPObject *npobj, NPIdentifier name)
{
  NPAPI_LOG (INFO, "%s", __func__);
  
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) npobj;
  
  g_return_val_if_fail (NULL != pWrapper, false);
  
  if ( ! NPN_IdentifierIsString (name))
    {
      return NULL != pWrapper->array_access;
    }
  
  char * methodname = NPN_UTF8FromIdentifier (name);
  
  bool hasMethod = (NULL != g_hash_table_lookup (pWrapper->methods, methodname));
  
  NPAPI_LOG (INFO, "%s %s %s", __func__, methodname, hasMethod ? "true" : "false");
  
  NPN_MemFree (methodname);
  
  return hasMethod;
}


static bool
NPClass_Invoke (NPObject *npobj
		, NPIdentifier name
		, const NPVariant *args
		, uint32_t argCount
		, NPVariant *result)
{
  NPAPI_LOG (INFO, "%s", __func__);
  
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) npobj;
  
  // 1. if array access
  if ( ! NPN_IdentifierIsString (name))
    {
      if (NULL == pWrapper->array_access)
	{
	  return false;
	}
      
      *result = pWrapper->array_access (pWrapper->instance
					, npobj
					, NPN_IntFromIdentifier (name));
      
      return true;
    }
  
  // 2. regular method call
  if ( ! NPClass_HasMethod (npobj, name))
    {
      return false;
    }
  
  char * methodname = NPN_UTF8FromIdentifier (name);
  
  NPAPI_LOG (INFO, "%s %s", __func__, methodname);
  
  if (NULL == pWrapper->methods)
    {
      NPN_MemFree (methodname);
      return false;
    }
  
  UnityWebAppsNpapiMethodPtr
    method = g_hash_table_lookup (pWrapper->methods, methodname);
  
  if (NULL == method)
    {
      NPN_MemFree (methodname);
      return false;
    }
  
  *result = method (pWrapper->instance, npobj, (NPVariant*) args, argCount);
  
  NPN_MemFree (methodname);
  
  return true;
}


static bool
NPClass_InvokeDefault (NPObject *npobj
		       , const NPVariant *args
		       , uint32_t argCount
		       , NPVariant *result)
{
  NPAPI_LOG (INFO, "%s", __func__);
  return false;
}


static bool
NPClass_HasProperty (NPObject * npobj, NPIdentifier name)
{
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) npobj;
  
  g_return_val_if_fail (NULL != pWrapper, false);
  
  if ( ! NPN_IdentifierIsString (name))
    {
      return NULL != pWrapper->array_access; 
    }
  
  char * propertyname = NPN_UTF8FromIdentifier (name);
  
  bool hasproperty = (NULL != g_hash_table_lookup (pWrapper->properties, propertyname));
  
  NPN_MemFree (propertyname);
  
  return hasproperty;
}


static bool
NPClass_GetProperty (NPObject *npobj, NPIdentifier name, NPVariant *result)
{
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) npobj;
  
  g_return_val_if_fail (NULL != pWrapper, false);
  
  if ( ! NPN_IdentifierIsString (name))
    {
      if (NULL == pWrapper->array_access)
	{
	  return false;
	}
      
      *result = pWrapper->array_access (pWrapper->instance
					, npobj
					, NPN_IntFromIdentifier (name));
      
      return true;
    }
  
  if ( ! NPClass_HasProperty (npobj, name))
    {
      return false;
    }
  
  char * propertyname = NPN_UTF8FromIdentifier (name);
  
  NPAPI_LOG (INFO, "%s %s", __func__, propertyname);
  
  if (NULL == pWrapper->properties)
    {
      NPN_MemFree (propertyname);
      return false;
    }
  
  UnityWebAppsNpapiPropertyDescr *
    property = g_hash_table_lookup (pWrapper->properties, propertyname);
  
  if (NULL == property || NULL == property->get)
    {
      NPN_MemFree (propertyname);
      return false;
    }
  
  *result = property->get (npobj);
  
  NPN_MemFree (propertyname);
  
  return true;
}


static bool NPClass_SetProperty (NPObject *npobj, NPIdentifier name, const NPVariant *value)
{
  if ( ! NPClass_HasProperty (npobj, name))
    {
      return false;
    }
  
  char * propertyname = NPN_UTF8FromIdentifier (name);
  
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) npobj;
  
  if (NULL == pWrapper->properties)
    {
      NPN_MemFree (propertyname);
      return false;
    }
  
  UnityWebAppsNpapiPropertyDescr *
    property = g_hash_table_lookup (pWrapper->properties, propertyname);
  
  if (NULL == property || NULL == property->set)
    {
      NPN_MemFree (propertyname);
      return false;
    }
  
  bool result = property->set (npobj, value);
  
  NPN_MemFree (propertyname);
  
  return result;
}


static bool NPClass_RemoveProperty (NPObject *npobj, NPIdentifier name)
{
  if ( ! NPClass_HasProperty (npobj, name))
    {
      return false;
    }
  
  char * propertyname = NPN_UTF8FromIdentifier (name);
  
  UnityWebappsJavascriptObjectWrapper *
    pWrapper = (UnityWebappsJavascriptObjectWrapper *) npobj;
  
  if (NULL == pWrapper->properties)
    {
      NPN_MemFree (propertyname);
      return false;
    }
  
  UnityWebAppsNpapiPropertyDescr *
    property = g_hash_table_lookup (pWrapper->properties, propertyname);
  
  if (NULL == property)
    {
      NPN_MemFree (propertyname);
      return false;
    }
  
  free_property (property);
  
  bool result = g_hash_table_remove (pWrapper->properties, property);
  
  NPN_MemFree (propertyname);
  
  return result;
}


static bool NPClass_Enumerate (NPObject *npobj, NPIdentifier **identifier, uint32_t *count)
{
  // no properties/methods
  return false;
}


static bool NPClass_Construct (NPObject *npobj, const NPVariant *args, uint32_t argCount, NPVariant *result)
{
  // no need for specific construct
  return false;
}


