/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#include "SmartScopesClient.h"
#include <QJsonDocument>
#include <QNetworkRequest>
#include <QNetworkAccessManager>
#include <QtNetwork/QNetworkReply>
#include <QEventLoop>
#include <QThread>
#include <QUrlQuery>
#include <QStringList>
#include <QProcessEnvironment>
#include <QDebug>
#include <deelistmodel.h>
#include "Utils.h"
#include "NetworkRequestThread.h"

const QString SmartScopesClient::SERVER = "https://productsearch.ubuntu.com";
const QString SmartScopesClient::SEARCH_URI = "/smartscopes/v1/search";
const QString SmartScopesClient::PREVIEW_URI = "/smartscopes/v1/preview";

SmartScopesClient::SmartScopesClient(const QString &locale) :
    m_locale(locale)
{
    m_server = QProcessEnvironment::systemEnvironment().value("SMART_SCOPES_SERVER", QString::null);
    if (m_server == QString::null)
        m_server = SERVER;
}

SmartScopesClient::~SmartScopesClient()
{
}

void SmartScopesClient::search(const QString& query, const QString& sessionId, const QStringList& scopes,
                               const QString &origin,
                               const ResultHandler& resultHandler,
                               const RecommendationsHandler& recommendHandler,
                               UnityCancellable *cancellable)
{
    auto url = buildSearchUrl(query, scopes, sessionId, origin);

    auto thread = new NetworkRequestThread(url, cancellable);
    QEventLoop loop;
    QObject::connect(thread, SIGNAL(finished()), &loop, SLOT(quit()));
    thread->start();
    loop.exec();
    thread->wait();

    qDebug() << "Search request finished" << url;

    QNetworkReply* reply = thread->getReply();
    Q_ASSERT(reply != nullptr);

    if (reply->error() == QNetworkReply::NoError)
    {
        if (cancellable != nullptr && unity_cancellable_is_cancelled(cancellable)) {
            return;
        }
        while (reply->canReadLine())
        {
            auto line = reply->readLine();
            m_parser.parseLine(line, resultHandler, recommendHandler);
        }
    }
    else
    {
        qWarning() << "Error processing search request:" << reply->errorString();
    }
    thread->deleteLater();
}

UnityAbstractPreview* SmartScopesClient::preview(const QString& serverSid, const QString& sessionId, const QString &resultId, const UnityScopeResult& result)
{
    auto url = buildPreviewUrl(sessionId, resultId);
    auto encodedResult = resultToJson(result).replace("\r", "").replace("\n","");

    HttpHeadersList headers;
    headers.append(HttpHeader("X-SERVER-SID", serverSid.toUtf8()));
    headers.append(HttpHeader("X-PREVIOUS-RESULT", encodedResult));
    
    auto thread = new NetworkRequestThread(url, nullptr, headers);
    QEventLoop loop;
    QObject::connect(thread, SIGNAL(finished()), &loop, SLOT(quit()));
    thread->start();
    loop.exec();
    thread->wait();

    QNetworkReply* reply = thread->getReply();
    Q_ASSERT(reply != nullptr);

    UnityPreview *preview = nullptr;
    if (reply->error() == QNetworkReply::NoError)
    {
        auto data = reply->readAll();
        qDebug() << "Got preview:" << data;
        preview = m_previewParser.parse(data);        
    }
    else
    {
        qWarning() << "Error processing preview request:" << reply->errorString();
    }    
    thread->deleteLater();

    return UNITY_ABSTRACT_PREVIEW(preview);
}

QUrl SmartScopesClient::buildSearchUrl(const QString &searchString, const QStringList& scopes, const QString& sessionId,
                                       const QString &origin)
{
    QUrlQuery query;
    query.addQueryItem("q", searchString);
    query.addQueryItem("session_id", sessionId);
    query.addQueryItem("locale", m_locale);
    if (scopes.size() > 0) {
        query.addQueryItem("scopes", scopes.join(","));
    }
    if (!origin.isEmpty()) {
        query.addQueryItem("origin", origin);
    }

    QUrl url(QString("%1%2").arg(m_server).arg(SEARCH_URI));
    url.setQuery(query);
    return url;
}

QUrl SmartScopesClient::buildPreviewUrl(const QString& sessionId, const QString& resultId)
{
    QUrlQuery query;
    query.addQueryItem("session_id", sessionId);
    query.addQueryItem("result_id", resultId);
    query.addQueryItem("locale", m_locale);

    QUrl url(QString("%1%2").arg(m_server).arg(PREVIEW_URI));
    url.setQuery(query);
    return url;
}

QByteArray SmartScopesClient::resultToJson(const UnityScopeResult& result)
{
    QVariantMap vmap;
    vmap.insert("uri", result.uri);
    vmap.insert("title", result.title);
    vmap.insert("icon_hint", result.icon_hint);
    vmap.insert("comment", result.comment);
    vmap.insert("metadata", ghashtableToQVariantHash(result.metadata));

    const QJsonDocument doc = QJsonDocument::fromVariant(vmap);
    return doc.toJson();
}
