/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import QtTest 1.0
import Ubuntu.Components 0.1
import Unity.Test 0.1 as UT

TestCase {
    id: testCase
    TestUtil {id:util}

    // This is needed for waitForRendering calls to return
    // if the watched element already got rendered
    Rectangle {
        width: units.gu(1)
        height: width
        parent: testCase.parent
        border { width: units.dp(1); color: "black" }
        opacity: 0.6

        visible: testCase.running

        RotationAnimation on rotation {
            running: parent.visible
            from: 0
            to: 360
            loops: Animation.Infinite
            duration: 1000
        }
    }

    // Fake implementation to be provided to items under test
    property var fakeDateTime: new function() {
        this.currentTimeMs = 0
        this.getCurrentTimeMs = function() {return this.currentTimeMs}
    }

    // TODO This function can be removed altogether once we use Qt 5.5 which has the same feature
    function mouseClick(item, x, y, button, modifiers, delay) {
        if (button === undefined)
            button = Qt.LeftButton;
        if (modifiers === undefined)
            modifiers = Qt.NoModifier;
        if (delay === undefined)
            delay = -1;
        if (x === undefined)
            x = item.width / 2;
        if (y === undefined)
            y = item.height / 2;
        if (!qtest_events.mouseClick(item, x, y, button, modifiers, delay))
            qtest_fail("window not shown", 2);
    }

    // TODO This function can be removed altogether once we use Qt 5.5 which has the same feature
    function mouseDoubleClick(item, x, y, button, modifiers, delay) {
        if (button === undefined)
            button = Qt.LeftButton;
        if (modifiers === undefined)
            modifiers = Qt.NoModifier;
        if (delay === undefined)
            delay = -1;
        if (x === undefined)
            x = item.width / 2;
        if (y === undefined)
            y = item.height / 2;
        if (!qtest_events.mouseDoubleClick(item, x, y, button, modifiers, delay))
            qtest_fail("window not shown", 2)
    }

    // TODO This function can be removed altogether once we use Qt 5.5 which has the same feature
    function mousePress(item, x, y, button, modifiers, delay) {
        if (button === undefined)
            button = Qt.LeftButton;
        if (modifiers === undefined)
            modifiers = Qt.NoModifier;
        if (delay === undefined)
            delay = -1;
        if (x === undefined)
            x = item.width / 2;
        if (y === undefined)
            y = item.height / 2;
        if (!qtest_events.mousePress(item, x, y, button, modifiers, delay))
            qtest_fail("window not shown", 2)
    }

    // TODO This function can be removed altogether once we use Qt 5.5 which has the same feature
    function mouseRelease(item, x, y, button, modifiers, delay) {
        if (button === undefined)
            button = Qt.LeftButton;
        if (modifiers === undefined)
            modifiers = Qt.NoModifier;
        if (delay === undefined)
            delay = -1;
        if (x === undefined)
            x = item.width / 2;
        if (y === undefined)
            y = item.height / 2;
        if (!qtest_events.mouseRelease(item, x, y, button, modifiers, delay))
            qtest_fail("window not shown", 2)
    }


    // Flickable won't recognise a single mouse move as dragging the flickable.
    // Use 5 steps because it's what
    // Qt uses in QQuickViewTestUtil::flick
    // speed is in pixels/second
    function mouseFlick(item, x, y, toX, toY, pressMouse, releaseMouse,
                        speed, iterations) {
        pressMouse = ((pressMouse != null) ? pressMouse : true); // Default to true for pressMouse if not present
        releaseMouse = ((releaseMouse != null) ? releaseMouse : true); // Default to true for releaseMouse if not present

        // set a default speed if not specified
        speed = (speed != null) ? speed : units.gu(10);

        // set a default iterations if not specified
        iterations = (iterations !== undefined) ? iterations : 5

        var distance = Math.sqrt(Math.pow(toX - x, 2) + Math.pow(toY - y, 2))
        var totalTime = (distance / speed) * 1000 /* converting speed to pixels/ms */

        var timeStep = totalTime / iterations
        var diffX = (toX - x) / iterations
        var diffY = (toY - y) / iterations
        if (pressMouse) {
            fakeDateTime.currentTimeMs += timeStep
            mousePress(item, x, y)
        }
        for (var i = 0; i < iterations; ++i) {
            fakeDateTime.currentTimeMs += timeStep
            if (i === iterations - 1) {
                // Avoid any rounding errors by making the last move be at precisely
                // the point specified
                mouseMove(item, toX, toY, iterations / speed)
            } else {
                mouseMove(item, x + (i + 1) * diffX, y + (i + 1) * diffY, iterations / speed)
            }
        }
        if (releaseMouse) {
            fakeDateTime.currentTimeMs += timeStep
            mouseRelease(item, toX, toY)
        }
    }


    // Find an object with the given name in the children tree of "obj"
    function findChild(obj, objectName) {
        return findChildIn(obj, "children", objectName);
    }

    // Find an object with the given name in the children tree of "obj"
    // Including invisible children like animations, timers etc.
    // Note: you should use findChild if you're not sure you need this
    // as this tree is much bigger and might contain stuff that goes
    // away randomly.
    function findInvisibleChild(obj, objectName) {
        return findChildIn(obj, "data", objectName);
    }

    // Find a child in the named property
    function findChildIn(obj, prop, objectName) {
        var childs = new Array(0);
        childs.push(obj)
        while (childs.length > 0) {
            if (childs[0].objectName == objectName) {
                return childs[0]
            }
            for (var i in childs[0][prop]) {
                childs.push(childs[0][prop][i])
            }
            childs.splice(0, 1);
        }
        return null;
    }

    function findChildsByType(obj, typeName) {
        var res = new Array(0);
        for (var i in obj.children) {
            var c = obj.children[i];
            if (UT.Util.isInstanceOf(c, typeName)) {
                res.push(c)
            }
            res = res.concat(findChildsByType(c, typeName));
        }
        return res;
    }

    // Type a full string instead of keyClick letter by letter
    function typeString(str) {
        for (var i = 0; i < str.length; i++) {
            keyClick(str[i])
        }
    }

    // Keeps executing a given parameter-less function until it returns the given
    // expected result or the timemout is reached (in which case a test failure
    // is generated)
    function tryCompareFunction(func, expectedResult, timeout) {
        var timeSpent = 0
        if (timeout === undefined)
            timeout = 5000;
        var success = false
        var actualResult
        while (timeSpent < timeout && !success) {
            actualResult = func()
            success = qtest_compareInternal(actualResult, expectedResult)
            if (success === false) {
                wait(50)
                timeSpent += 50
            }
        }

        var act = qtest_results.stringify(actualResult)
        var exp = qtest_results.stringify(expectedResult)
        if (!qtest_results.compare(success,
                                   "function returned unexpected result",
                                   act, exp,
                                   util.callerFile(), util.callerLine())) {
            throw new Error("QtQuickTest::fail")
        }
    }

    function flickToYEnd(item) {
        var i = 0;
        var x = item.width / 2;
        var y = item.height - units.gu(1);
        var toY = units.gu(1);
        while (i < 5 && !item.atYEnd) {
            touchFlick(item, x, y, x, toY);
            tryCompare(item, "moving", false);
            ++i;
        }
        tryCompare(item, "atYEnd", true);
    }

    function touchEvent(item) {
        return UT.Util.touchEvent(item)
    }

    // speed is in pixels/second
    function touchFlick(item, x, y, toX, toY, beginTouch, endTouch, speed, iterations) {
        // Make sure the item is rendered
        waitForRendering(item);

        var root = fetchRootItem(item);
        var rootFrom = item.mapToItem(root, x, y);
        var rootTo = item.mapToItem(root, toX, toY);

        // Default to true for beginTouch if not present
        beginTouch = (beginTouch !== undefined) ? beginTouch : true

        // Default to true for endTouch if not present
        endTouch = (endTouch !== undefined) ? endTouch : true

        // Set a default speed if not specified
        speed = (speed !== undefined) ? speed : units.gu(10)

        // Set a default iterations if not specified
        var iterations = (iterations !== undefined) ? iterations : 5

        var distance = Math.sqrt(Math.pow(rootTo.x - rootFrom.x, 2) + Math.pow(rootTo.Y - rootFrom.y, 2))
        var totalTime = (distance / speed) * 1000 /* converting speed to pixels/ms */

        var timeStep = totalTime / iterations
        var diffX = (rootTo.x - rootFrom.x) / iterations
        var diffY = (rootTo.y - rootFrom.y) / iterations
        if (beginTouch) {
            fakeDateTime.currentTimeMs += timeStep

            var event = touchEvent(item)
            event.press(0 /* touchId */, rootFrom.x, rootFrom.y)
            event.commit()
        }
        for (var i = 0; i < iterations; ++i) {
            fakeDateTime.currentTimeMs += timeStep
            if (i === iterations - 1) {
                // Avoid any rounding errors by making the last move be at precisely
                // the point specified
                wait(iterations / speed)
                var event = touchEvent(item)
                event.move(0 /* touchId */, rootTo.x, rootTo.y)
                event.commit()
            } else {
                wait(iterations / speed)
                var event = touchEvent(item)
                event.move(0 /* touchId */, rootFrom.x + (i + 1) * diffX, rootFrom.y + (i + 1) * diffY)
                event.commit()
            }
        }
        if (endTouch) {
            fakeDateTime.currentTimeMs += timeStep
            var event = touchEvent(item)
            event.release(0 /* touchId */, rootTo.x, rootTo.y)
            event.commit()
        }
    }

    function touchPinch(item, x1Start, y1Start, x1End, y1End, x2Start, y2Start, x2End, y2End) {
        // Make sure the item is rendered
        waitForRendering(item);

        var event1 = touchEvent(item);
        // first finger
        event1.press(0, x1Start, y1Start);
        event1.commit();
        // second finger
        event1.move(0, x1Start, y1Start);
        event1.press(1, x2Start, y2Start);
        event1.commit();

        // pinch
        for (var i = 0.0; i < 1.0; i += 0.02) {
            event1.move(0, x1Start + (x1End - x1Start) * i, y1Start + (y1End - y1Start) * i);
            event1.move(1, x2Start + (x2End - x2Start) * i, y2Start + (y2End - y2Start) * i);
            event1.commit();
        }

        // release
        event1.release(0, x1End, y1End);
        event1.release(1, x2End, y2End);
        event1.commit();
    }

    function fetchRootItem(item) {
        if (item.parent)
            return fetchRootItem(item.parent)
        else
            return item
    }

    function touchPress(item, x, y) {
        var root = fetchRootItem(item)
        var rootPoint = item.mapToItem(root, x, y)

        var event = touchEvent(item)
        event.press(0 /* touchId */, rootPoint.x, rootPoint.y)
        event.commit()
    }

    function touchRelease(item, x, y) {
        var root = fetchRootItem(item)
        var rootPoint = item.mapToItem(root, x, y)

        var event = touchEvent(item)
        event.release(0 /* touchId */, rootPoint.x, rootPoint.y)
        event.commit()
    }

    /*! \brief Tap the item with a touch event.

      \param item The item to be tapped
      \param x The x coordinate of the tap, defaults to horizontal center
      \param y The y coordinate of the tap, defaults to vertical center
     */
    function tap(item, x, y) {
        if (typeof x !== "number") x = item.width / 2;
        if (typeof y !== "number") y = item.height / 2;

        var root = fetchRootItem(item)
        var rootPoint = item.mapToItem(root, x, y)

        var event = touchEvent(item)
        event.press(0 /* touchId */, rootPoint.x, rootPoint.y)
        event.commit()

        event = touchEvent(item)
        event.release(0 /* touchId */, rootPoint.x, rootPoint.y)
        event.commit()
    }

    Component.onCompleted: {
        UT.Util.ensureTouchRegistryInstalled();

        var rootItem = parent;
        while (rootItem.parent != undefined) {
            rootItem = rootItem.parent;
        }
        removeTimeConstraintsFromDirectionalDragAreas(rootItem);
    }

    /*
      In qmltests, sequences of touch events are sent all at once, unlike in "real life".
      Also qmltests might run really slowly, e.g. when run from inside virtual machines.
      Thus to remove a variable that qmltests cannot really control, namely time, this
      function removes all constraints from DirectionalDragAreas that are sensible to
      elapsed time.

      This effectively makes DirectionalDragAreas easier to fool.
     */
    function removeTimeConstraintsFromDirectionalDragAreas(item) {

        // use duck-typing to identify a DirectionalDragArea
        if (item.minSpeed != undefined
                && item.maxSilenceTime != undefined
                && item.compositionTime != undefined) {
            item.minSpeed = 0;
            item.maxSilenceTime = 60 * 60 * 1000;
            item.compositionTime = 0;
        } else {
            for (var i in item.children) {
                removeTimeConstraintsFromDirectionalDragAreas(item.children[i]);
            }
        }
    }

    // TODO This function can be removed altogether once we use Qt 5.5 which has the same feature
    function waitForRendering(item, timeout) {
        if (timeout === undefined)
            timeout = 5000;
        if (!item)
            qtest_fail("No item given to waitForRendering", 1);
        return qtest_results.waitForRendering(item, timeout);
    }

}
