/*
 * UnixCW - Unix CW (Morse code) training program
 * Copyright (C) 2001, 2002  Simon Baldwin (simonb@caldera.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * cwgen.c - Simple morse code practice character generator.
 *
 */

/* Include files. */
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>

#if defined(HAVE_STRING_H)
#	include <string.h>
#endif /* HAVE_STRING_H */
#if defined(HAVE_STRINGS_H)
#	include <strings.h>
#endif /* HAVE_STRINGS_H */

#if defined(HAVE_GETOPT_H)
#	include <getopt.h>			/* Linux */
#endif /* HAVE_GETOPT_H */

/* Definitions. */
#define	VERSION		"cwgen version 2.2"
#define	COPYRIGHT	"Copyright (C) 2001, 2002  Simon Baldwin\n"	\
"This program comes with ABSOLUTELY NO WARRANTY; for details\n"		\
"please see the file 'COPYING' supplied with the source code.\n"	\
"This is free software, and you are welcome to redistribute it\n"	\
"under certain conditions; again, see 'COPYING' for details.\n"		\
"This program is released under the GNU General Public License."

#define	ASC_NUL		'\0'		/* End of string */
#define	ASC_FNS		'/'		/* ASCII filename sep char */

#define	MIN_GROUPS	1		/* Lowest number of groups allowed */
#define	MAX_GROUPS	20000		/* Highest number of groups allowed */
#define	MIN_GROUPSIZE	1		/* Lowest group size allowed */
#define	MAX_GROUPSIZE	80		/* Highest group size allowed */
#define	DEFAULT_CHARSET	"ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
					/* Default character set */

#define	MAXARGS		128		/* Args to exec */
#define	MAXOPTSTR	1024		/* Longest _OPTIONS env variable */
#define	ARGS_WHITESPACE	" \t"		/* Argument separators */
#define	ENV_OPTIONS	"CWGEN_OPTIONS"	/* Env string holding options */

/* Initial defaults for parameters. */
#define	INITIAL_GROUPS		128
#define	INITIAL_GROUPSIZE	5


/* Global variables. */
static int groups	= INITIAL_GROUPS;
					/* Initially 128 groups */
static int groupsize	= INITIAL_GROUPSIZE;
					/* Initially 5-character groups */
static char *charset	= DEFAULT_CHARSET;
					/* Initially send alphanumerics */

/* Base name of the program, from argv[0]. */
static char *argv0	= NULL;


/*
 * print_usage()
 *
 * Print out a brief message directing the user to the help function.
 */
static void
print_usage ()
{
	assert (argv0 != NULL);

	fprintf (stderr,
#if defined(HAVE_GETOPT_LONG)
		"Try '%s --help' for more information.\n",
#else  /* not HAVE_GETOPT_LONG */
		"Try '%s -h' for more information.\n",
#endif /* not HAVE_GETOPT_LONG */
		argv0);
	exit (1);
}

/*
 * print_help()
 *
 * Print out a brief page of help information.
 */
static void
print_help ()
{
	assert (argv0 != NULL);

	printf (
#if defined(HAVE_GETOPT_LONG)
	"Usage: %s [options...]\n\n\
	-g, --groups=GROUPS	send GROUPS groups of chars [default %d]\n\
				valid GROUPS values are between %d and %d\n\
	-n, --groupsize=GS	make groups GS chars [default %d]\n\
				valid GS values are between %d and %d\n\
	-c, --charset=CHARSET	select chars to send from this set\n\
				[default %s]\n\
	-h, --help		print this message\n\
	-v, --version		output version information and exit\n\n",
#else  /* not HAVE_GETOPT_LONG */
	"Usage: %s [options...]\n\n\
	-g GROUPS	send GROUPS groups of chars [default %d]\n\
			valid GROUPS values are between %d and %d\n\
	-n GS		make groups GS chars [default %d]\n\
			valid GS values are between %d and %d\n\
	-c CHARSET	select chars to send from this set\n\
			[default %s]\n\
	-h 		print this message\n\
	-v 		output version information and exit\n\n",
#endif /* not HAVE_GETOPT_LONG */
	argv0,
	INITIAL_GROUPS,    MIN_GROUPS,    MAX_GROUPS,
	INITIAL_GROUPSIZE, MIN_GROUPSIZE, MAX_GROUPSIZE, DEFAULT_CHARSET);
}


/*
 * parse_cmdline
 *
 * Parse the command line options for initial values for the various
 * global and flag defintions.
 */
static void
parse_cmdline (int argc, char **argv)
{
	int	c;				/* Option character */
	int	argind;				/* Loop index */
	char	env_options[ MAXOPTSTR ];	/* Env options string */
	char	*sptr;				/* String pointer */
	char	*local_argv[ MAXARGS ];		/* Local argv array */
	int	local_argc = 0;			/* Local argc */
#if defined(HAVE_GETOPT_LONG)
	int	option_index;			/* Option index */
	static const struct option long_options[] = {	/* Options table */
		{ "groups",	1, 0, 'g' }, { "groupsize",	1, 0, 'n' },
		{ "charset",	1, 0, 'c' }, { "help",		0, 0, 'h' },
		{ "version",	0, 0, 'v' },
		{ 0, 0, 0, 0 }};
#endif /* HAVE_GETOPT_LONG */

	/* Set argv0 to be the basename part of the program name. */
	argv0 = argv[0] + strlen (argv[0]);
	while (*argv0 != ASC_FNS && argv0 > argv[0])
		argv0--;
	if (*argv0 == ASC_FNS)
		argv0++;

	/*
	 * Build a new view of argc and argv by first prepending
	 * the strings from ..._OPTIONS, if defined, then putting the
	 * command line args on (so they take precedence).
	 */
	local_argv[ local_argc++ ] = argv[0];
	if (getenv (ENV_OPTIONS) != NULL)
	    {
		strcpy (env_options, getenv (ENV_OPTIONS));
		sptr = env_options;
		while (local_argc < MAXARGS - 1)
		    {
			while (strchr (ARGS_WHITESPACE, *sptr) != NULL
					&& *sptr != ASC_NUL)
				sptr++;
			if (*sptr == ASC_NUL)
				break;
			else
			    {
				local_argv[ local_argc++ ] = sptr;
				while (strchr (ARGS_WHITESPACE, *sptr)
						== NULL && *sptr != ASC_NUL)
					sptr++;
				if (strchr (ARGS_WHITESPACE, *sptr)
						!= NULL && *sptr != ASC_NUL)
				    {
					*sptr = ASC_NUL;
					sptr++;
				    }
			    }
		    }
	    }
	for (argind = 1; argind < argc; argind++)
	    {
		local_argv[ local_argc++ ] = argv[ argind ];
	    }

	/* Process every option. */
	for (;;)
	    {
#if defined(HAVE_GETOPT_LONG)
		c = getopt_long (local_argc, local_argv, "g:n:c:hv",
					long_options, &option_index);
#else  /* not HAVE_GETOPT_LONG */
		c = getopt (local_argc, local_argv, "g:n:c:hv");
#endif /* not HAVE_GETOPT_LONG */
		if (c == -1)
			break;

		switch (c)
		    {
			case 'g':
				if (sscanf (optarg, "%d", &groups) != 1
					|| groups < MIN_GROUPS
					|| groups > MAX_GROUPS)
				    {
					fprintf (stderr,
						"%s: invalid groups value\n",
						argv0);
					exit (1);
				    }
				break;

			case 'n':
				if (sscanf (optarg, "%d", &groupsize) != 1
					|| groupsize < MIN_GROUPSIZE
					|| groupsize > MAX_GROUPSIZE)
				    {
					fprintf (stderr,
						"%s: invalid groupsize value\n",
						argv0);
					exit (1);
				    }
				break;

			case 'c':
				if (strlen (optarg) == 0)
				    {
					fprintf (stderr,
						"%s: charset cannot be empty\n",
						argv0);
					exit (1);
				    }
				charset = optarg;
				break;

			case 'h':
				print_help ();
				exit (0);

			case 'v':
				printf ("%s, ", VERSION);
				printf ("%s\n", COPYRIGHT);
				exit (0);

			case '?':
				print_usage ();

			default:
				fprintf (stderr,
					"%s: getopts returned %c\n",
								argv0, c);
				exit (1);
		    }
	    }
	if (optind != local_argc)
		print_usage ();
}


/*
 * generate_characters()
 *
 * Generate random characters on stdout, in groups as requested, and up
 * to the requested number of groups.  Characters are selected from the
 * set given at random.
 */
static void
generate_characters (int num_groups, int groupchars, char *set)
{
	int	gcount;			/* Group counter */
	int	ccount;			/* Chars in a group counter */
	int	random_index;		/* Random index into set */

	/* Generate groups up to the number requested. */
	for (gcount = 0; gcount < num_groups; gcount++)
	    {
		for (ccount = 0; ccount < groupchars; ccount++)
		    {

			/* Pick a 'random' character from the set. */
			random_index = rand () % strlen (set);

			/* Print that character, immediately (with flush). */
			printf ("%c", set[ random_index ]);
			fflush (stdout);
		}

		/* Print a group separator space. */
		printf (" ");
		fflush (stdout);
	    }
}


/*
 * main()
 *
 * Parse the command line options, then generate the characters requested.
 */
int
main (int argc, char **argv)
{
	struct timeval	timeofday;	/* Current time of day, for seed */

	/* Parse the command line parameters and arguments. */
	parse_cmdline (argc, argv);

	/*
	 * Seed the random number generator, using the usecs field of
	 * the current timeofday as a source.  This gives us a reasonable
	 * seed value.
	 */
	if (gettimeofday (&timeofday, NULL) == -1)
	    {
		perror ("timeofday");
		exit (1);
	    }
	srand (timeofday.tv_usec);

	/* Now generate the character groups, as requested. */
	generate_characters (groups, groupsize, charset);
	printf ("\n");

	/* All done. */
	return 0;
}
