# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unperish plugin for bzr operations."""


import os

import unperishlib


class IsBzrBranch(unperishlib.Operation):

    """Verify that the directory is a bzr branch.
    
    This operation exists just to verify that the directory we operate
    on is a bzr branch. This way other operations in this plugin do not
    need to embed the check themselves, but can just require the
    is_bzr_branch option this operation provides.
    
    """

    name = "is-bzr-branch"
    
    provided_options = ["is_bzr_branch"]
    
    def add_options(self, parser):
        parser.add_option("--is-bzr-branch", action="store_true",
                          help="Assume the source directory is a bzr branch.")

    def do_it(self, options):
        if os.path.isdir(os.path.join(options.directory, ".bzr")):
            options.is_bzr_branch = True


class BuildTarballFromBzr(unperishlib.Operation):

    """Create a release tarball from a bzr branch.
    
    The entire contents of the bzr branch are exported into a .tar.gz
    file in the build area. The full path to the upstream tarball
    (not just basename) is stored into the upstream_tarball option.
    
    """
    
    name = "build-tarball-from-bzr"

    required_options = ["is_bzr_branch", "build_area_exists", "project_name", 
                        "project_version"]

    provided_options = ["upstream_tarball"]

    def do_it(self, options):
        if options.upstream_tarball is None:
            base = ("%s-%s.tar.gz" % 
                        (options.project_name, options.project_version))
            options.upstream_tarball = os.path.join(options.build_area, base)
        if not os.path.exists(options.upstream_tarball):
            unperishlib.run(["bzr", "export", "--quiet", 
                             options.upstream_tarball],
                            cwd=options.directory)


class BzrChangelog(unperishlib.Operation):

    """Generate a projectname.changelog file to build area."""

    name = "bzr-changelog"
    
    required_options = ["is_bzr_branch", "build_area_exists", "project_name",
                        "directory"]

    provided_options = ["changelog"]
    
    def add_options(self, parser):
        parser.add_public_option("--changelog", metavar="FILE",
                                 help="""\
Use FILE as the changelog to generate and publish.""")

    def do_it(self, options):
        stdout = unperishlib.run(["bzr", "log", "-v"], cwd=options.directory)
        
        if not options.changelog:
            options.changelog = os.path.join(options.build_area,
                                       "%s.changelog" % options.project_name)
        f = file(options.changelog, "w")
        f.write(stdout)
        f.close()
