# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unit tests for Unperish plugin for .deb packages."""


import os
import shutil
import tempfile
import unittest

import unperishlib


def _write(filename, contents):
    f = file(filename, "w")
    f.write(contents)
    f.close()

def _create_debian_files(dirname):
    debdir = os.path.join(dirname, "debian")
    os.mkdir(debdir)
    _write(os.path.join(debdir, "control"), """\
Source: pink
Maintainer: Pink <pretty@example.com>
Section: misc
Priority: extra
Standards-Version: 3.7.2

Package: pink
Architecture: all
Description: pretty
 pink is pretty.
""")
    _write(os.path.join(debdir, "changelog"), """\
pink (1.2-3) unstable; urgency=low

  * Pretty!
  
 -- Pretty in Pink <pink@example.com>  Tue, 01 May 2007 14:46:48 +0300

""")


class IsDebianPackageTests(unittest.TestCase):

    def setUp(self):
        self.dirname = tempfile.mkdtemp()
        pm = unperishlib.PluginManager()
        pm.set_plugin_directories(["plugins"])
        pm.load_plugins()
        self.oper = pm.get_operation("is-debian-package")
        self.options = unperishlib.FakeOptions(directory=self.dirname)

    def tearDown(self):
        shutil.rmtree(self.dirname)

    def testRaisesExceptionForNonDebianDirectory(self):
        self.failUnlessRaises(unperishlib.UnperishException,
                              self.oper.do_it, self.options)

    def testRecognizesDebianProjectCorrectly(self):
        os.mkdir(os.path.join(self.dirname, "debian"))
        file(os.path.join(self.dirname, "debian", "control"), "w").close()
        self.failUnlessEqual(self.oper.do_it(self.options), None)


class DebianInfoTests(unittest.TestCase):

    def setUp(self):
        self.dirname = tempfile.mkdtemp()
        _create_debian_files(self.dirname)
        self.pm = unperishlib.PluginManager()
        self.pm.set_plugin_directories(["plugins"])
        self.pm.load_plugins()
        self.cli = unperishlib.CommandLineParser()
        self.oper = self.pm.get_operation("debian-info")
        self.oper.add_options(self.cli)
        self.cli.set_defaults(directory=self.dirname)
        
    def tearDown(self):
        shutil.rmtree(self.dirname)

    def testAddsDebianSourceNameOption(self):
        self.failUnless(self.cli.has_option("--debian-source-name"))

    def testAddsDebianVersionOption(self):
        self.failUnless(self.cli.has_option("--debian-version"))

    def testGetsDebianSourcePackageNameCorrectly(self):
        options, _ = self.cli.parse_args([])
        self.oper.do_it(options)
        self.failUnlessEqual(options.debian_source_name, "pink")

    def testGetsDebianVersionCorrectly(self):
        options, _ = self.cli.parse_args([])
        self.oper.do_it(options)
        self.failUnlessEqual(options.debian_version, "1.2-3")

    def testGetsDebianUpstreamVersionCorrectly(self):
        options, _ = self.cli.parse_args([])
        self.oper.do_it(options)
        self.failUnlessEqual(options.debian_upstream_version, "1.2")
        
    def testComputesUpstreamVersionForNonNativePackageCorrectly(self):
        self.failUnlessEqual(self.oper.compute_upstream_part("1.2-3"), "1.2")

    def testComputesUpstreamVersionForNativePackageCorrectly(self):
        self.failUnlessEqual(self.oper.compute_upstream_part("1.2"), "1.2")

    def testRaisesExceptionIfPackageAndDebianUpstreamVersionsDiffer(self):
        projinfo = self.pm.get_operation("project-info")
        projinfo.add_options(self.cli)
        options, _ = self.cli.parse_args(["--project-version=0.0"])
        self.failUnlessRaises(unperishlib.UnperishException,
                              self.oper.do_it, options)

    def testDoesNotRaiseOptionIfPackageAndDebianUpstreamVersionsMatch(self):
        projinfo = self.pm.get_operation("project-info")
        projinfo.add_options(self.cli)
        options, _ = self.cli.parse_args(["--project-version=1.2"])
        self.failUnlessEqual(self.oper.do_it(options), None)


class DebianOrigTarGz(unittest.TestCase):

    def setUp(self):
        self.tmp = tempfile.mkdtemp()
        self.build_area = os.path.join(self.tmp, "build-area")
        self.directory = os.path.join(self.tmp, "trunk")
        os.mkdir(self.build_area)
        os.mkdir(self.directory)
        _write(os.path.join(self.directory, "pink"), "pretty")
        _create_debian_files(self.directory)
        unperishlib.run(["bzr", "init", "--quiet"], cwd=self.directory)
        unperishlib.run(["bzr", "add", "--quiet"], cwd=self.directory)
        unperishlib.run(["bzr", "commit", "-m.", "--quiet"], 
                        cwd=self.directory)
    
        self.app = unperishlib.Application()

        pm = self.app.get_plugin_manager()
        pm.set_plugin_directories(["plugins"])
        self.app.prepare_plugins()

        self.parser = self.app.get_command_line_parser()
        self.parser.set_defaults(directory=self.directory,
                                 build_area=self.build_area,
                                 tarballs=self.build_area,
                                 project_name="pink",
                                 project_version="1.2")

        self.oper = pm.get_operation("debian-orig-tar-gz")

    def tearDown(self):
        shutil.rmtree(self.tmp)

    def testRequiresUpstreamTarballOption(self):
        self.failUnless("upstream_tarball" in 
                            self.oper.get_required_options())

    def path(self, relative):
        return os.path.join(self.build_area, relative)

    def testCreatesOrigTarGz(self):
        self.app.run(["debian-orig-tar-gz"])
        self.failUnless(os.path.exists(self.path("pink_1.2.orig.tar.gz")))

    def testCreatesTarGzForNativePackage(self):
        self.app.run(["--debian-version=1.2", "debian-orig-tar-gz"])
        self.failUnless(os.path.exists(self.path("pink_1.2.tar.gz")))


class DebianSourceTreeFromBzrTests(unittest.TestCase):

    def setUp(self):
        self.tmp = tempfile.mkdtemp()
        self.build_area = os.path.join(self.tmp, "build-area")
        self.directory = os.path.join(self.tmp, "trunk")
        os.mkdir(self.build_area)
        os.mkdir(self.directory)
        _write(os.path.join(self.directory, "pink"), "pretty")
        _create_debian_files(self.directory)
        unperishlib.run(["bzr", "init", "--quiet"], cwd=self.directory)
        unperishlib.run(["bzr", "add", "--quiet"], cwd=self.directory)
        unperishlib.run(["bzr", "commit", "-m.", "--quiet"], 
                        cwd=self.directory)
    
        self.app = unperishlib.Application()

        pm = self.app.get_plugin_manager()
        pm.set_plugin_directories(["plugins"])
        self.app.prepare_plugins()

        self.parser = self.app.get_command_line_parser()
        self.parser.set_defaults(directory=self.directory,
                                 build_area=self.build_area,
                                 project_name="pink",
                                 project_version="1.2")

    def testCreatesSourceTree(self):
        self.app.run(["debian-source-tree-from-bzr"])
        dirname = os.path.join(self.build_area, "pink-1.2")
        self.failUnless(os.path.isdir(dirname))


class DebianSourcePackageTests(unittest.TestCase):

    def setUp(self):
        self.tmp = tempfile.mkdtemp()
        self.build_area = os.path.join(self.tmp, "build-area")
        self.directory = os.path.join(self.tmp, "trunk")
        os.mkdir(self.build_area)
        os.mkdir(self.directory)
        _write(os.path.join(self.directory, "pink"), "pretty")
        _create_debian_files(self.directory)
        unperishlib.run(["bzr", "init", "--quiet"], cwd=self.directory)
        unperishlib.run(["bzr", "add", "--quiet"], cwd=self.directory)
        unperishlib.run(["bzr", "commit", "-m.", "--quiet"], 
                        cwd=self.directory)
    
        self.app = unperishlib.Application()

        pm = self.app.get_plugin_manager()
        pm.set_plugin_directories(["plugins"])
        self.app.prepare_plugins()

        self.parser = self.app.get_command_line_parser()
        self.parser.set_defaults(directory=self.directory,
                                 build_area=self.build_area,
                                 tarballs=self.build_area,
                                 project_name="pink",
                                 project_version="1.2")

        self.oper = pm.get_operation("debian-source-package")

    def tearDown(self):
        shutil.rmtree(self.tmp)

    def testRequiresUpstreamTarballOption(self):
        self.failUnless("upstream_tarball" in 
                            self.oper.get_required_options())

    def path(self, relative):
        return os.path.join(self.build_area, relative)

    def testCreatesSourceTree(self):
        self.app.run(["debian-source-package"])
        self.failUnless(os.path.isdir(self.path("pink-1.2")))

    def testCreatesDscAndDiffGz(self):
        self.app.run(["debian-source-package"])
        self.failUnless(os.path.exists(self.path("pink_1.2-3.dsc")))
        self.failUnless(os.path.exists(self.path("pink_1.2-3.diff.gz")))

    def testDoesNotCreateDiffGzForNativePackage(self):
        path = os.path.join(self.directory, "debian", "changelog")
        f = file(path, "w")
        f.write("""\
pink (1.2) unstable; urgency=low

  * Pretty!
  
 -- Pretty in Pink <pink@example.com>  Tue, 01 May 2007 14:46:48 +0300

""")
        f.close()
        unperishlib.run(["bzr", "commit", "-m.", "--quiet"], 
                        cwd=self.directory)
        self.app.run(["debian-source-package"])
        self.failUnless(os.path.exists(self.path("pink_1.2.dsc")))
        self.failIf(os.path.exists(self.path("pink_1.2.diff.gz")))


class DebianArchitectureTests(unittest.TestCase):

    def testReturnsNonEmptyString(self):
        app = unperishlib.Application(plugins=["plugins"])
        options, _ = app.parse_args([])
        oper = app.get_plugin_manager().get_operation("debian-architecture")
        oper.do_it(options)
        self.failUnless(type(options.debian_architecture) is str)
        self.failIfEqual(options.debian_architecture, "")


class CreateAptRepositoryTests(unittest.TestCase):

    def setUp(self):
        self.dirname = tempfile.mkdtemp()
        self.repodir = os.path.join(self.dirname, "apt")
        app = unperishlib.Application(plugins=["plugins"])
        self.options, _ = app.parse_args(["--debian-source-name=pink",
                                          "--apt-repository", self.repodir])
        self.oper = app.get_plugin_manager().get_operation(
            "create-apt-repository-for-reprepro")

    def tearDown(self):
        shutil.rmtree(self.dirname)

    def testCreatesDirectory(self):
        self.oper.do_it(self.options)
        self.failUnless(os.path.isdir(self.repodir))

    def testSetsAptRepositoryExists(self):
        self.oper.do_it(self.options)
        self.failUnless(self.options.apt_repository_exists)
