#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <error.h>
#include <argp.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "config.h"
#include "opts.h"
#include "xrealpath.h"
#define FULL_VERSION PACKAGE_NAME " " PACKAGE_VERSION
const char *argp_program_version = FULL_VERSION;
const char *argp_program_bug_address = "<" PACKAGE_BUGREPORT ">";
static char doc[] = "Extract files from rar archives.";
static char args_doc[] = "ARCHIVE [FILE...] [DESTINATION]";
struct arguments_t arguments;
const char force_option_explanation[]="Overwrite files when extracting";
const char extract_newer_option_explanation[]="Only extract newer files from the archive";
const char junk_paths_option_explanation[]="Don't create directories while extracting";
const char extract_option_explanation[]="Extract files from archive (default)";
const char list_option_explanation[]="List files in archive";
const char verbose_option_explanation[]="Verbosely list files extracted";
const char password_option_explanation[]="Decrypt archive using a password";
static struct argp_option options[] = 
{
	{"extract", CMD_EXTRACT, 0, 0, extract_option_explanation, 0},
	{"list", CMD_LIST, 0, 0, list_option_explanation, 1},
	{"force", OPT_FORCE, 0, 0, force_option_explanation, 2},
	{"extract-newer", OPT_EXTRACT_NEWER, 0, 0, extract_newer_option_explanation, 3},
	{"extract-no-paths", OPT_JUNK_PATHS, 0, 0, junk_paths_option_explanation, 3},
	
	{"password", OPT_PASSWORD, 0, 0, password_option_explanation, 3},
	{"verbose", OPT_VERBOSE, 0, OPTION_HIDDEN, verbose_option_explanation, 3},
	{ 0 }
};


void 
free_arguments()
{
	if (arguments.args)
		free(arguments.args);
	if (arguments.unrar.destination_dir)
		free (arguments.unrar.destination_dir);
	if (arguments.unrar.archive_filename)
		free (arguments.unrar.archive_filename);
	if (arguments.unrar.password)
		free (arguments.unrar.password);
	return;
}

static error_t 
parse_opt (int key, char *arg, struct argp_state *state) 
{
	/* Get the INPUT argument from `argp_parse', which we
		know is a pointer to our arguments structure. */

	struct arguments_t *arguments = state->input;
	char *pass;

	switch (key) 
	{ 
		case CMD_EXTRACT:
			arguments->unrar.mode = MODE_EXTRACT;
			break;
		case CMD_LIST:
			arguments->unrar.mode = MODE_LIST;
			break;
		case OPT_FORCE:
			arguments->unrar.force = 1;
			break;
		case OPT_JUNK_PATHS:
			arguments->unrar.junk_paths = 1; 
			break;
		case OPT_EXTRACT_NEWER:
			arguments->unrar.extract_newer = 1; 
			break;
		case OPT_PASSWORD:
			if ((pass = getpass("Password:")));
				arguments->unrar.password = strdup(pass); 
			break;
		case OPT_VERBOSE:
			arguments->unrar.verbose = 1; 
			break;
		case ARGP_KEY_INIT:
			free_arguments();
			arguments->arraylen = 0; 
			arguments->args = (char**) malloc(1 * sizeof(char**));
			break;
		case ARGP_KEY_ARG:
			arguments->arraylen++; 
			arguments->args =(char **)realloc(arguments->args, arguments->arraylen * sizeof(char*));
			arguments->args[state->arg_num] = arg;
			break;
		case ARGP_KEY_END:
			if (arguments->arraylen > 0)
			{
				char *file;
				struct stat statbuf;
				file = xrealpath(arguments->args[0]);
				if (lstat(file,&statbuf) == -1)
				{
					free(file);
					error(0,0,"invalid archive '%s': %m",arguments->args[0]);
					argp_usage(state);
				}
				else
				{
					if (!(S_ISDIR(statbuf.st_mode)))
						arguments->unrar.archive_filename = file;
					else
					{
						free(file);
						error(0,0,"invalid archive '%s': is a directory",arguments->args[0]);
						argp_usage(state);
					}
				}
			}

			if (arguments->arraylen > 1)
			{
				char *dir;
				struct stat statbuf;
				dir = xrealpath(arguments->args[arguments->arraylen-1]);
				if (lstat(dir,&statbuf) == -1)
					free(dir);
				else
				{
					if (S_ISDIR(statbuf.st_mode))
					{
						arguments->unrar.destination_dir = dir;
						arguments->args[arguments->arraylen-1] = NULL;
						arguments->arraylen--;
					}
					else
						free(dir);
				}
			}
			break;
		default:
			return ARGP_ERR_UNKNOWN;
		}
	return 0;
}

int 
set_default_arguments(struct unrar_arguments_t *unrar)
{
	if (unrar->mode == MODE_UNKNOWN)
		unrar->mode = MODE_EXTRACT;
	if (unrar->destination_dir == NULL)
		unrar->destination_dir = xrealpath("./");
	return 0;
}

struct argp argp = { options, parse_opt, args_doc, doc }; 

int
parse_opts(int argc, char **argv, struct arguments_t *arguments)
{
	int retval;
	setenv("ARGP_HELP_FMT", "no-dup-args-note", 1);
	atexit(free_arguments);
	retval = argp_parse (&argp,argc,argv,0,0,arguments); 
	if (retval < 0)
		argp_help(&argp, stdout, ARGP_HELP_EXIT_ERR|ARGP_HELP_SEE,PACKAGE_NAME);
	set_default_arguments(&arguments->unrar);

	if (arguments->arraylen == 0)
	{
		error(0, 0, "Archive not specified\n");
		argp_help(&argp, stdout, ARGP_HELP_EXIT_ERR|ARGP_HELP_SEE,PACKAGE_NAME);
		return -1;
	}

	return 0;
}

