/* $Id: login.c,v 1.15 2002/10/30 17:09:28 bjk Exp $ */
/*
    Copyright (C) 2001-2002  Ben Kibbey <bjk@arbornet.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include "common.h"
#include "login.h"

static void lastinfo(INFO * info, char *str)
{
    int i = 0;
    char *buf;

    if (!str) {
	strcpy(info->lastinfo->tty, UNKNOWN);
	strcpy(info->lastinfo->host, UNKNOWN);
	strcpy(info->lastinfo->time, UNKNOWN);
	return;
    }

    while ((buf = strsep(&str, ",")) != NULL) {
	switch (i++) {
	    case 0:
		strcpy(info->lastinfo->tty, buf);
		break;
	    case 1:
		strcpy(info->lastinfo->host, buf);
		break;
	    case 2:
		strcpy(info->lastinfo->time, buf);
		break;
	    default:
		break;
	}
    }

    return;
}

static char *lastlogin(uid_t uid)
{
    int count;
    long offset;
    static char buf[LINE_MAX];
    char tmp[TIMEBUFSIZE], htmp[UT_HOSTSIZE + 1];
    struct lastlog last;

    buf[0] = tmp[0] = htmp[0] = 0;

    if (lastlogfd < 0)
        return NULL;

    if (!lastlogfd) {
	if ((lastlogfd = open(_PATH_LASTLOG, O_RDONLY)) == -1) {
	    warn("%s", _PATH_LASTLOG);
	    return NULL;
	}
    }

    offset = (long) uid * sizeof(struct lastlog);

    if (lseek(lastlogfd, offset, SEEK_SET) == -1) {
	warn("%s", _PATH_LASTLOG);
	return NULL;
    }

    if ((count = read(lastlogfd, &last, sizeof(struct lastlog))) !=
	    sizeof(struct lastlog)) {
	if (count == -1)
	    warn("%s", _PATH_LASTLOG);

	return NULL;
    }

    if (last.ll_line[0] == 0)
	strcpy(buf, NONE);
    else
	strcpy(buf, last.ll_line);

    strcat(buf, ",");

    strncpy(htmp, last.ll_host, sizeof(htmp));
    htmp[UT_HOSTSIZE] = 0;

    if (htmp[0] == 0)
	strcat(buf, NONE);
    else
	strcat(buf, htmp);

    strcat(buf, ",");

    if (last.ll_time)
	strcat(buf, stamp(last.ll_time, tf));
    else
	strcat(buf, NONE);

    return buf;
}

/* This will return a utmp structure if a user is logged in, NULL otherwise.
 * This may not be accurate for a long running process as the utmp file is
 * read only once into a linked list.
 */
static void *get_utmp(const char *user)
{
    ssize_t count;
    struct utmps *utmpt;

    if (utmpfd < 0)
	return NULL;

    if (!utmpfd) {
	if ((utmpfd = open(_PATH_UTMP, O_RDONLY)) == -1) {
	    warn("%s", _PATH_UTMP);
	    return NULL;
	}

	users = Malloc(sizeof(struct utmps));
	utmpt = users;

	while ((count = read(utmpfd, &utmpt->utmp, sizeof(UTMP))) ==
	       sizeof(UTMP)) {
	    utmpt->next = Malloc(sizeof(struct utmps));
	    utmpt = utmpt->next;
	}

	utmpt = NULL;
    }

    utmpt = users;

    while (utmpt) {
	if (strcmp(user, utmpt->utmp.ut_name) == 0)
	    return utmpt;

	utmpt = utmpt->next;
    }

    return NULL;
}

static int msgstat(const char *tty)
{
    char filename[FILENAME_MAX];
    struct stat st;

    snprintf(filename, sizeof(filename), "%s%s", _PATH_DEV, tty);

    if (stat(filename, &st) == -1)
	return 0;

    return (st.st_mode & S_IWGRP || st.st_mode & S_IWOTH) ? 1 : 0;
}

static int idle(UTMP *utmpt)
{
    char filename[FILENAME_MAX];
    time_t now, t;
    struct stat st;

    snprintf(filename, sizeof(filename), "%s/%s", _PATH_DEV, utmpt->ut_line);

    if (stat(filename, &st) == -1)
	return -1;

    if (utmpt->ut_time > st.st_atime)
	return 0;

    time(&now);
    t = st.st_atime;

    if (t < utmpt->ut_time)
	t = utmpt->ut_time;

    return (now - t <= 0) ? 0 : now - t;
}

void logininfo(INFO * info, const char *arg)
{
    int i;
    time_t now;
    UTMP *utmpt;

    if (info->login == NULL)
	info->login = Malloc(sizeof(LOGIN));

    memset(info->login, 0, sizeof(LOGIN));

    utmpt = get_utmp(info->passwd->pw_name);

    for (i = 0; i < ARRAYCNT(optspec) && optspec[i]; i++) {
	switch (optspec[i]) {
	    case 'q':
		info->login->idle = (utmpt) ? idle(utmpt) : -1;
		break;
	    case 'n':
		lastinfo(info, lastlogin(info->passwd->pw_uid));
		break;
	    case 'x':
		strncpy(info->login->host, (utmpt && utmpt->ut_host[0]) ?
			utmpt->ut_host : UNKNOWN, UT_HOSTSIZE);
		info->login->host[UT_HOSTSIZE] = 0;
		break;
	    case 'y':
		strncpy(info->login->tty, (utmpt && utmpt->ut_line[0]) ?
			utmpt->ut_line : UNKNOWN, UT_LINESIZE);
		info->login->tty[UT_LINESIZE] = 0;
		break;
	    case 'e':
		info->login->mesgstat = (utmpt) ? msgstat(utmpt->ut_line) : -1;
		break;
	    case 't':
		strcpy(info->login->stamp, (utmpt) ?
			stamp(utmpt->ut_time, tf) : UNKNOWN);
		break;
	    case 'b':
		if (!utmpt) {
		    info->login->duration = -1;
		    break;
		}

		time(&now);

		if ((now - utmpt->ut_time) > 60)
		    info->login->duration =
			((now - utmpt->ut_time) / 60);
		else
		    info->login->duration = -2;
		break;
#if defined(HAVE_PROCFS) || defined(HAVE_LIBKVM)
	    case 'C':
		info->login->ppid = (utmpt) ? 
		    parentpid(info->passwd->pw_uid) : -1;
		break;
#endif
	    default:
		break;
	}
    }

    return;
}
