/* valafield.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valafield.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaFieldPrivate {
	ValaExpression* _initializer;
	gboolean _is_volatile;
	gboolean _no_array_length;
	char* cname;
	gboolean _instance;
	gboolean lock_used;
	ValaDataType* _data_type;
};
#define VALA_FIELD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FIELD, ValaFieldPrivate))
enum  {
	VALA_FIELD_DUMMY_PROPERTY,
	VALA_FIELD_TYPE_REFERENCE,
	VALA_FIELD_INITIALIZER,
	VALA_FIELD_INSTANCE,
	VALA_FIELD_IS_VOLATILE,
	VALA_FIELD_NO_ARRAY_LENGTH
};
static void vala_field_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_field_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_field_set_cname (ValaField* self, const char* cname);
static void vala_field_process_ccode_attribute (ValaField* self, ValaAttribute* a);
static gboolean vala_field_real_get_lock_used (ValaLockable* base);
static void vala_field_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_field_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_field_parent_class = NULL;
static ValaLockableIface* vala_field_vala_lockable_parent_iface = NULL;
static void vala_field_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


/**
 * Creates a new field.
 *
 * @param name   field name
 * @param type   field type
 * @param init   initializer expression
 * @param source reference to source code
 * @return       newly created field
 */
ValaField* vala_field_new (const char* name, ValaDataType* type_reference, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaField * self;
	g_return_val_if_fail (type_reference == NULL || VALA_IS_DATA_TYPE (type_reference), NULL);
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_FIELD, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_field_set_type_reference (self, type_reference);
	vala_field_set_initializer (self, initializer);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_field_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaField * self;
	self = VALA_FIELD (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_field (visitor, self);
}


static void vala_field_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaField * self;
	self = VALA_FIELD (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_field_get_type_reference (self)), visitor);
	if (vala_field_get_initializer (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_field_get_initializer (self)), visitor);
	}
}


/**
 * Returns the name of this field as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_field_get_cname (ValaField* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_field_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


static void vala_field_set_cname (ValaField* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_FIELD (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


/**
 * Returns the default name of this field as it is used in C code.
 *
 * @return the name to be used in C code by default
 */
char* vala_field_get_default_cname (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	if (!vala_field_get_instance (self)) {
		char* _tmp0;
		char* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = g_strconcat ((_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self)), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	} else {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
	}
}


static void vala_field_process_ccode_attribute (ValaField* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_FIELD (self));
	g_return_if_fail (a == NULL || VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_field_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
				{
					vala_member_add_cheader_filename (VALA_MEMBER (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_field_process_attributes (ValaField* self) {
	g_return_if_fail (VALA_IS_FIELD (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = a_it->data, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_field_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "NoArrayLength") == 0) {
						vala_field_set_no_array_length (self, TRUE);
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_field_real_get_lock_used (ValaLockable* base) {
	ValaField * self;
	self = VALA_FIELD (base);
	return self->priv->lock_used;
}


static void vala_field_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaField * self;
	self = VALA_FIELD (base);
	self->priv->lock_used = used;
}


static void vala_field_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaField * self;
	self = VALA_FIELD (base);
	g_return_if_fail (old_type == NULL || VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (new_type == NULL || VALA_IS_DATA_TYPE (new_type));
	if (vala_field_get_type_reference (self) == old_type) {
		vala_field_set_type_reference (self, new_type);
	}
}


ValaDataType* vala_field_get_type_reference (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	return self->priv->_data_type;
}


void vala_field_set_type_reference (ValaField* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_FIELD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


ValaExpression* vala_field_get_initializer (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	return self->priv->_initializer;
}


void vala_field_set_initializer (ValaField* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_FIELD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL))), _tmp2);
}


gboolean vala_field_get_instance (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), FALSE);
	return self->priv->_instance;
}


void vala_field_set_instance (ValaField* self, gboolean value) {
	g_return_if_fail (VALA_IS_FIELD (self));
	self->priv->_instance = value;
}


gboolean vala_field_get_is_volatile (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), FALSE);
	return self->priv->_is_volatile;
}


void vala_field_set_is_volatile (ValaField* self, gboolean value) {
	g_return_if_fail (VALA_IS_FIELD (self));
	self->priv->_is_volatile = value;
}


gboolean vala_field_get_no_array_length (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_field_set_no_array_length (ValaField* self, gboolean value) {
	g_return_if_fail (VALA_IS_FIELD (self));
	self->priv->_no_array_length = value;
}


static void vala_field_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaField * self;
	self = VALA_FIELD (object);
	switch (property_id) {
		case VALA_FIELD_TYPE_REFERENCE:
		g_value_set_object (value, vala_field_get_type_reference (self));
		break;
		case VALA_FIELD_INITIALIZER:
		g_value_set_object (value, vala_field_get_initializer (self));
		break;
		case VALA_FIELD_INSTANCE:
		g_value_set_boolean (value, vala_field_get_instance (self));
		break;
		case VALA_FIELD_IS_VOLATILE:
		g_value_set_boolean (value, vala_field_get_is_volatile (self));
		break;
		case VALA_FIELD_NO_ARRAY_LENGTH:
		g_value_set_boolean (value, vala_field_get_no_array_length (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_field_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaField * self;
	self = VALA_FIELD (object);
	switch (property_id) {
		case VALA_FIELD_TYPE_REFERENCE:
		vala_field_set_type_reference (self, g_value_get_object (value));
		break;
		case VALA_FIELD_INITIALIZER:
		vala_field_set_initializer (self, g_value_get_object (value));
		break;
		case VALA_FIELD_INSTANCE:
		vala_field_set_instance (self, g_value_get_boolean (value));
		break;
		case VALA_FIELD_IS_VOLATILE:
		vala_field_set_is_volatile (self, g_value_get_boolean (value));
		break;
		case VALA_FIELD_NO_ARRAY_LENGTH:
		vala_field_set_no_array_length (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_field_class_init (ValaFieldClass * klass) {
	vala_field_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaFieldPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_field_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_field_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_field_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_field_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_field_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_field_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_TYPE_REFERENCE, g_param_spec_object ("type-reference", "type-reference", "type-reference", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_INITIALIZER, g_param_spec_object ("initializer", "initializer", "initializer", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_INSTANCE, g_param_spec_boolean ("instance", "instance", "instance", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_IS_VOLATILE, g_param_spec_boolean ("is-volatile", "is-volatile", "is-volatile", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_NO_ARRAY_LENGTH, g_param_spec_boolean ("no-array-length", "no-array-length", "no-array-length", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_field_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_field_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_field_real_get_lock_used;
	iface->set_lock_used = vala_field_real_set_lock_used;
}


static void vala_field_init (ValaField * self) {
	self->priv = VALA_FIELD_GET_PRIVATE (self);
	self->priv->_instance = TRUE;
	self->priv->lock_used = FALSE;
}


static void vala_field_dispose (GObject * obj) {
	ValaField * self;
	self = VALA_FIELD (obj);
	(self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL)));
	G_OBJECT_CLASS (vala_field_parent_class)->dispose (obj);
}


GType vala_field_get_type (void) {
	static GType vala_field_type_id = 0;
	if (G_UNLIKELY (vala_field_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFieldClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_field_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaField), 0, (GInstanceInitFunc) vala_field_init };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_field_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_field_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaField", &g_define_type_info, 0);
		g_type_add_interface_static (vala_field_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_field_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




