/* valafield.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;
typedef struct _ValaMemberPrivate ValaMemberPrivate;

#define VALA_TYPE_LOCKABLE (vala_lockable_get_type ())
#define VALA_LOCKABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCKABLE, ValaLockable))
#define VALA_IS_LOCKABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCKABLE))
#define VALA_LOCKABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_LOCKABLE, ValaLockableIface))

typedef struct _ValaLockable ValaLockable;
typedef struct _ValaLockableIface ValaLockableIface;

#define VALA_TYPE_FIELD (vala_field_get_type ())
#define VALA_FIELD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FIELD, ValaField))
#define VALA_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FIELD, ValaFieldClass))
#define VALA_IS_FIELD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FIELD))
#define VALA_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FIELD))
#define VALA_FIELD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FIELD, ValaFieldClass))

typedef struct _ValaField ValaField;
typedef struct _ValaFieldClass ValaFieldClass;
typedef struct _ValaFieldPrivate ValaFieldPrivate;

#define TYPE_MEMBER_BINDING (member_binding_get_type ())
#define _g_free0(var) (var = (g_free (var), NULL))
#define _vala_code_node_unref0(var) ((var == NULL) ? NULL : (var = (vala_code_node_unref (var), NULL)))

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_COMMENT (vala_comment_get_type ())
#define VALA_COMMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_COMMENT, ValaComment))
#define VALA_COMMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_COMMENT, ValaCommentClass))
#define VALA_IS_COMMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_COMMENT))
#define VALA_IS_COMMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_COMMENT))
#define VALA_COMMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_COMMENT, ValaCommentClass))

typedef struct _ValaComment ValaComment;
typedef struct _ValaCommentClass ValaCommentClass;

#define VALA_TYPE_LITERAL (vala_literal_get_type ())
#define VALA_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LITERAL, ValaLiteral))
#define VALA_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LITERAL, ValaLiteralClass))
#define VALA_IS_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LITERAL))
#define VALA_IS_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LITERAL))
#define VALA_LITERAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LITERAL, ValaLiteralClass))

typedef struct _ValaLiteral ValaLiteral;
typedef struct _ValaLiteralClass ValaLiteralClass;

#define VALA_TYPE_STRING_LITERAL (vala_string_literal_get_type ())
#define VALA_STRING_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STRING_LITERAL, ValaStringLiteral))
#define VALA_STRING_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_STRING_LITERAL, ValaStringLiteralClass))
#define VALA_IS_STRING_LITERAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STRING_LITERAL))
#define VALA_IS_STRING_LITERAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_STRING_LITERAL))
#define VALA_STRING_LITERAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_STRING_LITERAL, ValaStringLiteralClass))

typedef struct _ValaStringLiteral ValaStringLiteral;
typedef struct _ValaStringLiteralClass ValaStringLiteralClass;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;
#define _vala_source_file_unref0(var) ((var == NULL) ? NULL : (var = (vala_source_file_unref (var), NULL)))

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_OBJECT_TYPE_SYMBOL (vala_object_type_symbol_get_type ())
#define VALA_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbol))
#define VALA_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))
#define VALA_IS_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_IS_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_OBJECT_TYPE_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))

typedef struct _ValaObjectTypeSymbol ValaObjectTypeSymbol;
typedef struct _ValaObjectTypeSymbolClass ValaObjectTypeSymbolClass;

#define VALA_TYPE_INTERFACE (vala_interface_get_type ())
#define VALA_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INTERFACE, ValaInterface))
#define VALA_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INTERFACE, ValaInterfaceClass))
#define VALA_IS_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INTERFACE))
#define VALA_IS_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INTERFACE))
#define VALA_INTERFACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INTERFACE, ValaInterfaceClass))

typedef struct _ValaInterface ValaInterface;
typedef struct _ValaInterfaceClass ValaInterfaceClass;

#define VALA_TYPE_CLASS (vala_class_get_type ())
#define VALA_CLASS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CLASS, ValaClass))
#define VALA_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CLASS, ValaClassClass))
#define VALA_IS_CLASS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CLASS))
#define VALA_IS_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CLASS))
#define VALA_CLASS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CLASS, ValaClassClass))

typedef struct _ValaClass ValaClass;
typedef struct _ValaClassClass ValaClassClass;

struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
	gboolean (*is_instance_member) (ValaSymbol* self);
	gboolean (*is_class_member) (ValaSymbol* self);
};

struct _ValaMember {
	ValaSymbol parent_instance;
	ValaMemberPrivate * priv;
};

struct _ValaMemberClass {
	ValaSymbolClass parent_class;
};

struct _ValaLockableIface {
	GTypeInterface parent_iface;
	gboolean (*get_lock_used) (ValaLockable* self);
	void (*set_lock_used) (ValaLockable* self, gboolean used);
};

struct _ValaField {
	ValaMember parent_instance;
	ValaFieldPrivate * priv;
};

struct _ValaFieldClass {
	ValaMemberClass parent_class;
};

typedef enum  {
	MEMBER_BINDING_INSTANCE,
	MEMBER_BINDING_CLASS,
	MEMBER_BINDING_STATIC
} MemberBinding;

struct _ValaFieldPrivate {
	MemberBinding _binding;
	gboolean _is_volatile;
	gboolean _no_array_length;
	gboolean _no_delegate_target;
	gboolean _array_null_terminated;
	char* array_length_cname;
	char* array_length_cexpr;
	char* cname;
	gboolean lock_used;
	ValaExpression* _initializer;
	ValaDataType* _data_type;
};


static gpointer vala_field_parent_class = NULL;
static ValaLockableIface* vala_field_vala_lockable_parent_iface = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_member_get_type (void);
GType vala_lockable_get_type (void);
GType vala_field_get_type (void);
GType member_binding_get_type (void);
#define VALA_FIELD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FIELD, ValaFieldPrivate))
enum  {
	VALA_FIELD_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
gpointer vala_comment_ref (gpointer instance);
void vala_comment_unref (gpointer instance);
GParamSpec* vala_param_spec_comment (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_comment (GValue* value, gpointer v_object);
gpointer vala_value_get_comment (const GValue* value);
GType vala_comment_get_type (void);
ValaMember* vala_member_construct (GType object_type, const char* name, ValaSourceReference* source_reference, ValaComment* comment);
void vala_field_set_field_type (ValaField* self, ValaDataType* value);
void vala_field_set_initializer (ValaField* self, ValaExpression* value);
ValaField* vala_field_new (const char* name, ValaDataType* field_type, ValaExpression* initializer, ValaSourceReference* source_reference, ValaComment* comment);
ValaField* vala_field_construct (GType object_type, const char* name, ValaDataType* field_type, ValaExpression* initializer, ValaSourceReference* source_reference, ValaComment* comment);
void vala_code_visitor_visit_member (ValaCodeVisitor* self, ValaMember* m);
void vala_code_visitor_visit_field (ValaCodeVisitor* self, ValaField* f);
static void vala_field_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaDataType* vala_field_get_field_type (ValaField* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaExpression* vala_field_get_initializer (ValaField* self);
static void vala_field_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
char* vala_field_get_default_cname (ValaField* self);
char* vala_field_get_cname (ValaField* self);
void vala_field_set_cname (ValaField* self, const char* cname);
MemberBinding vala_field_get_binding (ValaField* self);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
const char* vala_symbol_get_name (ValaSymbol* self);
char* vala_field_get_array_length_cname (ValaField* self);
void vala_field_set_array_length_cname (ValaField* self, const char* array_length_cname);
char* vala_field_get_array_length_cexpr (ValaField* self);
void vala_field_set_array_length_cexpr (ValaField* self, const char* array_length_cexpr);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
void vala_member_add_cheader_filename (ValaMember* self, const char* filename);
gboolean vala_attribute_get_bool (ValaAttribute* self, const char* name);
void vala_field_set_no_array_length (ValaField* self, gboolean value);
void vala_field_set_array_null_terminated (ValaField* self, gboolean value);
void vala_field_set_no_delegate_target (ValaField* self, gboolean value);
static void vala_field_process_ccode_attribute (ValaField* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_field_process_attributes (ValaField* self);
static gboolean vala_field_real_get_lock_used (ValaLockable* base);
static void vala_field_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_field_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static void vala_field_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
ValaAttribute* vala_code_node_get_attribute (ValaCodeNode* self, const char* name);
char* vala_field_get_ctype (ValaField* self);
ValaAttribute* vala_attribute_new (const char* name, ValaSourceReference* source_reference);
ValaAttribute* vala_attribute_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
void vala_attribute_add_argument (ValaAttribute* self, const char* key, ValaExpression* value);
ValaStringLiteral* vala_string_literal_new (const char* value, ValaSourceReference* source_reference);
ValaStringLiteral* vala_string_literal_construct (GType object_type, const char* value, ValaSourceReference* source_reference);
GType vala_literal_get_type (void);
GType vala_string_literal_get_type (void);
void vala_field_set_ctype (ValaField* self, const char* ctype);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
ValaSourceFile* vala_semantic_analyzer_get_current_source_file (ValaSemanticAnalyzer* self);
ValaSymbol* vala_semantic_analyzer_get_current_symbol (ValaSemanticAnalyzer* self);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
void vala_semantic_analyzer_set_current_source_file (ValaSemanticAnalyzer* self, ValaSourceFile* value);
void vala_semantic_analyzer_set_current_symbol (ValaSemanticAnalyzer* self, ValaSymbol* value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
gboolean vala_semantic_analyzer_is_type_accessible (ValaSemanticAnalyzer* self, ValaSymbol* sym, ValaDataType* type);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
void vala_report_error (ValaSourceReference* source, const char* message);
char* vala_code_node_to_string (ValaCodeNode* self);
char* vala_symbol_get_full_name (ValaSymbol* self);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type);
gboolean vala_symbol_get_external (ValaSymbol* self);
GType vala_typesymbol_get_type (void);
GType vala_object_type_symbol_get_type (void);
GType vala_interface_get_type (void);
gboolean vala_symbol_is_internal_symbol (ValaSymbol* self);
GType vala_class_get_type (void);
gboolean vala_class_get_is_compact (ValaClass* self);
gboolean vala_symbol_get_external_package (ValaSymbol* self);
gboolean vala_member_get_hides (ValaMember* self);
ValaSymbol* vala_member_get_hidden_member (ValaMember* self);
void vala_report_warning (ValaSourceReference* source, const char* message);
static gboolean vala_field_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
void vala_field_set_binding (ValaField* self, MemberBinding value);
gboolean vala_field_get_is_volatile (ValaField* self);
void vala_field_set_is_volatile (ValaField* self, gboolean value);
gboolean vala_field_get_no_array_length (ValaField* self);
gboolean vala_field_get_no_delegate_target (ValaField* self);
gboolean vala_field_get_array_null_terminated (ValaField* self);
gboolean vala_field_get_has_array_length_cname (ValaField* self);
gboolean vala_field_get_has_array_length_cexpr (ValaField* self);
static void vala_field_finalize (ValaCodeNode* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaField* vala_field_construct (GType object_type, const char* name, ValaDataType* field_type, ValaExpression* initializer, ValaSourceReference* source_reference, ValaComment* comment) {
	ValaField* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (field_type != NULL, NULL);
	self = (ValaField*) vala_member_construct (object_type, name, source_reference, comment);
	vala_field_set_field_type (self, field_type);
	vala_field_set_initializer (self, initializer);
	return self;
}


ValaField* vala_field_new (const char* name, ValaDataType* field_type, ValaExpression* initializer, ValaSourceReference* source_reference, ValaComment* comment) {
	return vala_field_construct (VALA_TYPE_FIELD, name, field_type, initializer, source_reference, comment);
}


static void vala_field_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaField * self;
	self = (ValaField*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_member (visitor, (ValaMember*) self);
	vala_code_visitor_visit_field (visitor, self);
}


static void vala_field_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaField * self;
	self = (ValaField*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_field_get_field_type (self), visitor);
	if (vala_field_get_initializer (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_field_get_initializer (self), visitor);
	}
}


char* vala_field_get_cname (ValaField* self) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0_;
		self->priv->cname = (_tmp0_ = vala_field_get_default_cname (self), _g_free0 (self->priv->cname), _tmp0_);
	}
	result = g_strdup (self->priv->cname);
	return result;
}


void vala_field_set_cname (ValaField* self, const char* cname) {
	char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	self->priv->cname = (_tmp0_ = g_strdup (cname), _g_free0 (self->priv->cname), _tmp0_);
}


char* vala_field_get_default_cname (ValaField* self) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_binding == MEMBER_BINDING_STATIC) {
		char* _tmp0_;
		char* _tmp1_;
		result = (_tmp1_ = g_strconcat (_tmp0_ = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self), NULL), _g_free0 (_tmp0_), _tmp1_);
		return result;
	} else {
		result = g_strdup (vala_symbol_get_name ((ValaSymbol*) self));
		return result;
	}
}


char* vala_field_get_array_length_cname (ValaField* self) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = g_strdup (self->priv->array_length_cname);
	return result;
}


void vala_field_set_array_length_cname (ValaField* self, const char* array_length_cname) {
	char* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->array_length_cname = (_tmp0_ = g_strdup (array_length_cname), _g_free0 (self->priv->array_length_cname), _tmp0_);
}


char* vala_field_get_array_length_cexpr (ValaField* self) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = g_strdup (self->priv->array_length_cexpr);
	return result;
}


void vala_field_set_array_length_cexpr (ValaField* self, const char* array_length_cexpr) {
	char* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->array_length_cexpr = (_tmp0_ = g_strdup (array_length_cexpr), _g_free0 (self->priv->array_length_cexpr), _tmp0_);
}


static void vala_field_process_ccode_attribute (ValaField* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0_;
		vala_field_set_cname (self, _tmp0_ = vala_attribute_get_string (a, "cname"));
		_g_free0 (_tmp0_);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp1_;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			filename_collection = _tmp1_ = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp1_);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp1_); filename_it = filename_it + 1) {
				char* filename;
				filename = g_strdup (filename_collection[filename_it]);
				{
					vala_member_add_cheader_filename ((ValaMember*) self, filename);
					_g_free0 (filename);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		_g_free0 (val);
	}
	if (vala_attribute_has_argument (a, "array_length")) {
		vala_field_set_no_array_length (self, !vala_attribute_get_bool (a, "array_length"));
	}
	if (vala_attribute_has_argument (a, "array_null_terminated")) {
		vala_field_set_array_null_terminated (self, vala_attribute_get_bool (a, "array_null_terminated"));
	}
	if (vala_attribute_has_argument (a, "array_length_cname")) {
		char* _tmp2_;
		vala_field_set_array_length_cname (self, _tmp2_ = vala_attribute_get_string (a, "array_length_cname"));
		_g_free0 (_tmp2_);
	}
	if (vala_attribute_has_argument (a, "array_length_cexpr")) {
		char* _tmp3_;
		vala_field_set_array_length_cexpr (self, _tmp3_ = vala_attribute_get_string (a, "array_length_cexpr"));
		_g_free0 (_tmp3_);
	}
	if (vala_attribute_has_argument (a, "delegate_target")) {
		vala_field_set_no_delegate_target (self, !vala_attribute_get_bool (a, "delegate_target"));
	}
}


static gpointer _vala_code_node_ref0 (gpointer self) {
	return self ? vala_code_node_ref (self) : NULL;
}


void vala_field_process_attributes (ValaField* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a;
			a = _vala_code_node_ref0 ((ValaAttribute*) a_it->data);
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_field_process_ccode_attribute (self, a);
				}
				_vala_code_node_unref0 (a);
			}
		}
	}
}


static gboolean vala_field_real_get_lock_used (ValaLockable* base) {
	ValaField * self;
	gboolean result;
	self = (ValaField*) base;
	result = self->priv->lock_used;
	return result;
}


static void vala_field_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaField * self;
	self = (ValaField*) base;
	self->priv->lock_used = used;
}


static void vala_field_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaField * self;
	self = (ValaField*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_field_get_initializer (self) == old_node) {
		vala_field_set_initializer (self, new_node);
	}
}


static void vala_field_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaField * self;
	self = (ValaField*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_field_get_field_type (self) == old_type) {
		vala_field_set_field_type (self, new_type);
	}
}


char* vala_field_get_ctype (ValaField* self) {
	char* result;
	ValaAttribute* attr;
	g_return_val_if_fail (self != NULL, NULL);
	attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
	if (attr == NULL) {
		result = NULL;
		_vala_code_node_unref0 (attr);
		return result;
	}
	result = vala_attribute_get_string (attr, "type");
	_vala_code_node_unref0 (attr);
	return result;
}


void vala_field_set_ctype (ValaField* self, const char* ctype) {
	ValaAttribute* attr;
	ValaStringLiteral* _tmp2_;
	char* _tmp1_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (ctype != NULL);
	attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
	if (attr == NULL) {
		ValaAttribute* _tmp0_;
		attr = (_tmp0_ = vala_attribute_new ("CCode", NULL), _vala_code_node_unref0 (attr), _tmp0_);
		((ValaCodeNode*) self)->attributes = g_list_append (((ValaCodeNode*) self)->attributes, _vala_code_node_ref0 (attr));
	}
	vala_attribute_add_argument (attr, "type", (ValaExpression*) (_tmp2_ = vala_string_literal_new (_tmp1_ = g_strdup_printf ("\"%s\"", ctype), NULL)));
	_vala_code_node_unref0 (_tmp2_);
	_g_free0 (_tmp1_);
	_vala_code_node_unref0 (attr);
}


static gpointer _vala_source_file_ref0 (gpointer self) {
	return self ? vala_source_file_ref (self) : NULL;
}


static gboolean vala_field_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaField * self;
	gboolean result;
	ValaSourceFile* old_source_file;
	ValaSymbol* old_symbol;
	gboolean _tmp6_;
	gboolean field_in_header;
	gboolean _tmp8_;
	gboolean _tmp9_;
	self = (ValaField*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		result = !vala_code_node_get_error ((ValaCodeNode*) self);
		return result;
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	old_source_file = _vala_source_file_ref0 (vala_semantic_analyzer_get_current_source_file (analyzer));
	old_symbol = _vala_code_node_ref0 (vala_semantic_analyzer_get_current_symbol (analyzer));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	vala_code_node_check ((ValaCodeNode*) vala_field_get_field_type (self), analyzer);
	if (!vala_semantic_analyzer_is_type_accessible (analyzer, (ValaSymbol*) self, vala_field_get_field_type (self))) {
		char* _tmp2_;
		char* _tmp1_;
		char* _tmp0_;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp2_ = g_strdup_printf ("field type `%s` is less accessible than field `%s`", _tmp0_ = vala_code_node_to_string ((ValaCodeNode*) vala_field_get_field_type (self)), _tmp1_ = vala_symbol_get_full_name ((ValaSymbol*) self)));
		_g_free0 (_tmp2_);
		_g_free0 (_tmp1_);
		_g_free0 (_tmp0_);
		result = FALSE;
		_vala_source_file_unref0 (old_source_file);
		_vala_code_node_unref0 (old_symbol);
		return result;
	}
	vala_field_process_attributes (self);
	if (vala_field_get_initializer (self) != NULL) {
		vala_expression_set_target_type (vala_field_get_initializer (self), vala_field_get_field_type (self));
		vala_code_node_check ((ValaCodeNode*) vala_field_get_initializer (self), analyzer);
		if (!vala_data_type_compatible (vala_expression_get_value_type (vala_field_get_initializer (self)), vala_field_get_field_type (self))) {
			char* _tmp5_;
			char* _tmp4_;
			char* _tmp3_;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp5_ = g_strdup_printf ("Cannot convert from `%s' to `%s'", _tmp3_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_field_get_initializer (self))), _tmp4_ = vala_code_node_to_string ((ValaCodeNode*) vala_field_get_field_type (self))));
			_g_free0 (_tmp5_);
			_g_free0 (_tmp4_);
			_g_free0 (_tmp3_);
			result = FALSE;
			_vala_source_file_unref0 (old_source_file);
			_vala_code_node_unref0 (old_symbol);
			return result;
		}
		if (vala_symbol_get_external ((ValaSymbol*) self)) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "External fields cannot use initializers");
		}
	}
	_tmp6_ = FALSE;
	if (self->priv->_binding == MEMBER_BINDING_INSTANCE) {
		_tmp6_ = VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) self));
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Interfaces may not have instance fields");
		result = FALSE;
		_vala_source_file_unref0 (old_source_file);
		_vala_code_node_unref0 (old_symbol);
		return result;
	}
	field_in_header = !vala_symbol_is_internal_symbol ((ValaSymbol*) self);
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
		ValaClass* cl;
		gboolean _tmp7_;
		cl = _vala_code_node_ref0 (VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)));
		_tmp7_ = FALSE;
		if (vala_class_get_is_compact (cl)) {
			_tmp7_ = !vala_symbol_is_internal_symbol ((ValaSymbol*) cl);
		} else {
			_tmp7_ = FALSE;
		}
		if (_tmp7_) {
			field_in_header = TRUE;
		}
		_vala_code_node_unref0 (cl);
	}
	_tmp8_ = FALSE;
	_tmp9_ = FALSE;
	if (!vala_symbol_get_external_package ((ValaSymbol*) self)) {
		_tmp9_ = !vala_member_get_hides ((ValaMember*) self);
	} else {
		_tmp9_ = FALSE;
	}
	if (_tmp9_) {
		ValaSymbol* _tmp10_;
		_tmp8_ = (_tmp10_ = vala_member_get_hidden_member ((ValaMember*) self)) != NULL;
		_vala_code_node_unref0 (_tmp10_);
	} else {
		_tmp8_ = FALSE;
	}
	if (_tmp8_) {
		char* _tmp14_;
		char* _tmp13_;
		ValaSymbol* _tmp12_;
		char* _tmp11_;
		vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp14_ = g_strdup_printf ("%s hides inherited field `%s'. Use the `new' keyword if hiding was intentional", _tmp11_ = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp13_ = vala_symbol_get_full_name (_tmp12_ = vala_member_get_hidden_member ((ValaMember*) self))));
		_g_free0 (_tmp14_);
		_g_free0 (_tmp13_);
		_vala_code_node_unref0 (_tmp12_);
		_g_free0 (_tmp11_);
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	result = !vala_code_node_get_error ((ValaCodeNode*) self);
	_vala_source_file_unref0 (old_source_file);
	_vala_code_node_unref0 (old_symbol);
	return result;
}


ValaDataType* vala_field_get_field_type (ValaField* self) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_data_type;
	return result;
}


void vala_field_set_field_type (ValaField* self, ValaDataType* value) {
	ValaDataType* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_data_type = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_data_type), _tmp0_);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_data_type, (ValaCodeNode*) self);
}


ValaExpression* vala_field_get_initializer (ValaField* self) {
	ValaExpression* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_initializer;
	return result;
}


void vala_field_set_initializer (ValaField* self, ValaExpression* value) {
	ValaExpression* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_initializer = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_initializer), _tmp0_);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_initializer, (ValaCodeNode*) self);
	}
}


MemberBinding vala_field_get_binding (ValaField* self) {
	MemberBinding result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_binding;
	return result;
}


void vala_field_set_binding (ValaField* self, MemberBinding value) {
	g_return_if_fail (self != NULL);
	self->priv->_binding = value;
}


gboolean vala_field_get_is_volatile (ValaField* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_is_volatile;
	return result;
}


void vala_field_set_is_volatile (ValaField* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_volatile = value;
}


gboolean vala_field_get_no_array_length (ValaField* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_no_array_length;
	return result;
}


void vala_field_set_no_array_length (ValaField* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_array_length = value;
}


gboolean vala_field_get_no_delegate_target (ValaField* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_no_delegate_target;
	return result;
}


void vala_field_set_no_delegate_target (ValaField* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_delegate_target = value;
}


gboolean vala_field_get_array_null_terminated (ValaField* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_array_null_terminated;
	return result;
}


void vala_field_set_array_null_terminated (ValaField* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_array_null_terminated = value;
}


gboolean vala_field_get_has_array_length_cname (ValaField* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->array_length_cname != NULL;
	return result;
}


gboolean vala_field_get_has_array_length_cexpr (ValaField* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->array_length_cexpr != NULL;
	return result;
}


static void vala_field_class_init (ValaFieldClass * klass) {
	vala_field_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_field_finalize;
	g_type_class_add_private (klass, sizeof (ValaFieldPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_field_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_field_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_field_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_field_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_field_real_check;
}


static void vala_field_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_field_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_field_real_get_lock_used;
	iface->set_lock_used = vala_field_real_set_lock_used;
}


static void vala_field_instance_init (ValaField * self) {
	self->priv = VALA_FIELD_GET_PRIVATE (self);
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
	self->priv->lock_used = FALSE;
}


static void vala_field_finalize (ValaCodeNode* obj) {
	ValaField * self;
	self = VALA_FIELD (obj);
	_g_free0 (self->priv->array_length_cname);
	_g_free0 (self->priv->array_length_cexpr);
	_g_free0 (self->priv->cname);
	_vala_code_node_unref0 (self->priv->_initializer);
	_vala_code_node_unref0 (self->priv->_data_type);
	VALA_CODE_NODE_CLASS (vala_field_parent_class)->finalize (obj);
}


GType vala_field_get_type (void) {
	static GType vala_field_type_id = 0;
	if (vala_field_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFieldClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_field_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaField), 0, (GInstanceInitFunc) vala_field_instance_init, NULL };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_field_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_field_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaField", &g_define_type_info, 0);
		g_type_add_interface_static (vala_field_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_field_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




