/* valalocalvariable.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;
typedef struct _ValaLocalVariablePrivate ValaLocalVariablePrivate;
#define _vala_code_node_unref0(var) ((var == NULL) ? NULL : (var = (vala_code_node_unref (var), NULL)))

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_MEMBER_ACCESS (vala_member_access_get_type ())
#define VALA_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccess))
#define VALA_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))
#define VALA_IS_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER_ACCESS))
#define VALA_IS_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER_ACCESS))
#define VALA_MEMBER_ACCESS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))

typedef struct _ValaMemberAccess ValaMemberAccess;
typedef struct _ValaMemberAccessClass ValaMemberAccessClass;

#define VALA_TYPE_LAMBDA_EXPRESSION (vala_lambda_expression_get_type ())
#define VALA_LAMBDA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LAMBDA_EXPRESSION, ValaLambdaExpression))
#define VALA_LAMBDA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LAMBDA_EXPRESSION, ValaLambdaExpressionClass))
#define VALA_IS_LAMBDA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LAMBDA_EXPRESSION))
#define VALA_IS_LAMBDA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LAMBDA_EXPRESSION))
#define VALA_LAMBDA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LAMBDA_EXPRESSION, ValaLambdaExpressionClass))

typedef struct _ValaLambdaExpression ValaLambdaExpression;
typedef struct _ValaLambdaExpressionClass ValaLambdaExpressionClass;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;

#define VALA_TYPE_DELEGATE_TYPE (vala_delegate_type_get_type ())
#define VALA_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateType))
#define VALA_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))
#define VALA_IS_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE_TYPE))
#define VALA_IS_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE_TYPE))
#define VALA_DELEGATE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))

typedef struct _ValaDelegateType ValaDelegateType;
typedef struct _ValaDelegateTypeClass ValaDelegateTypeClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_DELEGATE (vala_delegate_get_type ())
#define VALA_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE, ValaDelegate))
#define VALA_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE, ValaDelegateClass))
#define VALA_IS_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE))
#define VALA_IS_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE))
#define VALA_DELEGATE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE, ValaDelegateClass))

typedef struct _ValaDelegate ValaDelegate;
typedef struct _ValaDelegateClass ValaDelegateClass;
#define _g_free0(var) (var = (g_free (var), NULL))

#define VALA_TYPE_POINTER_TYPE (vala_pointer_type_get_type ())
#define VALA_POINTER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_POINTER_TYPE, ValaPointerType))
#define VALA_POINTER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_POINTER_TYPE, ValaPointerTypeClass))
#define VALA_IS_POINTER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_POINTER_TYPE))
#define VALA_IS_POINTER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_POINTER_TYPE))
#define VALA_POINTER_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_POINTER_TYPE, ValaPointerTypeClass))

typedef struct _ValaPointerType ValaPointerType;
typedef struct _ValaPointerTypeClass ValaPointerTypeClass;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_BLOCK (vala_block_get_type ())
#define VALA_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_BLOCK, ValaBlock))
#define VALA_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_BLOCK, ValaBlockClass))
#define VALA_IS_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_BLOCK))
#define VALA_IS_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_BLOCK))
#define VALA_BLOCK_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_BLOCK, ValaBlockClass))

typedef struct _ValaBlock ValaBlock;
typedef struct _ValaBlockClass ValaBlockClass;

struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
	gboolean (*is_instance_member) (ValaSymbol* self);
	gboolean (*is_class_member) (ValaSymbol* self);
};

struct _ValaLocalVariable {
	ValaSymbol parent_instance;
	ValaLocalVariablePrivate * priv;
};

struct _ValaLocalVariableClass {
	ValaSymbolClass parent_class;
};

struct _ValaLocalVariablePrivate {
	gboolean _is_result;
	gboolean _floating;
	gboolean _captured;
	gboolean _no_init;
	ValaExpression* _initializer;
	ValaDataType* _variable_type;
};


static gpointer vala_local_variable_parent_class = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
#define VALA_LOCAL_VARIABLE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariablePrivate))
enum  {
	VALA_LOCAL_VARIABLE_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
void vala_local_variable_set_variable_type (ValaLocalVariable* self, ValaDataType* value);
void vala_local_variable_set_initializer (ValaLocalVariable* self, ValaExpression* value);
ValaLocalVariable* vala_local_variable_new (ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference);
ValaLocalVariable* vala_local_variable_construct (GType object_type, ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference);
void vala_code_visitor_visit_local_variable (ValaCodeVisitor* self, ValaLocalVariable* local);
static void vala_local_variable_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaExpression* vala_local_variable_get_initializer (ValaLocalVariable* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
void vala_code_visitor_visit_end_full_expression (ValaCodeVisitor* self, ValaExpression* expr);
ValaDataType* vala_local_variable_get_variable_type (ValaLocalVariable* self);
static void vala_local_variable_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_local_variable_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static void vala_local_variable_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
ValaDataType* vala_data_type_copy (ValaDataType* self);
void vala_data_type_set_value_owned (ValaDataType* self, gboolean value);
void vala_data_type_set_floating_reference (ValaDataType* self, gboolean value);
GType vala_member_access_get_type (void);
GType vala_lambda_expression_get_type (void);
ValaSymbol* vala_expression_get_symbol_reference (ValaExpression* self);
GType vala_member_get_type (void);
GType vala_method_get_type (void);
GType vala_delegate_type_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_delegate_get_type (void);
ValaDelegate* vala_delegate_type_get_delegate_symbol (ValaDelegateType* self);
gboolean vala_delegate_matches_method (ValaDelegate* self, ValaMethod* m);
char* vala_symbol_get_full_name (ValaSymbol* self);
void vala_expression_set_value_type (ValaExpression* self, ValaDataType* value);
gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type);
char* vala_code_node_to_string (ValaCodeNode* self);
gboolean vala_data_type_is_disposable (ValaDataType* self);
GType vala_pointer_type_get_type (void);
gboolean vala_data_type_get_value_owned (ValaDataType* self);
ValaSymbol* vala_semantic_analyzer_get_current_symbol (ValaSemanticAnalyzer* self);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym);
const char* vala_symbol_get_name (ValaSymbol* self);
GType vala_block_get_type (void);
void vala_block_add_local_variable (ValaBlock* self, ValaLocalVariable* local);
void vala_symbol_set_active (ValaSymbol* self, gboolean value);
static gboolean vala_local_variable_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
gboolean vala_local_variable_get_is_result (ValaLocalVariable* self);
void vala_local_variable_set_is_result (ValaLocalVariable* self, gboolean value);
gboolean vala_local_variable_get_floating (ValaLocalVariable* self);
void vala_local_variable_set_floating (ValaLocalVariable* self, gboolean value);
gboolean vala_local_variable_get_captured (ValaLocalVariable* self);
void vala_local_variable_set_captured (ValaLocalVariable* self, gboolean value);
gboolean vala_local_variable_get_no_init (ValaLocalVariable* self);
void vala_local_variable_set_no_init (ValaLocalVariable* self, gboolean value);
static void vala_local_variable_finalize (ValaCodeNode* obj);



ValaLocalVariable* vala_local_variable_construct (GType object_type, ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaLocalVariable* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaLocalVariable*) vala_symbol_construct (object_type, name, source_reference);
	vala_local_variable_set_variable_type (self, variable_type);
	vala_local_variable_set_initializer (self, initializer);
	return self;
}


ValaLocalVariable* vala_local_variable_new (ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference) {
	return vala_local_variable_construct (VALA_TYPE_LOCAL_VARIABLE, variable_type, name, initializer, source_reference);
}


static void vala_local_variable_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLocalVariable * self;
	self = (ValaLocalVariable*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_local_variable (visitor, self);
}


static void vala_local_variable_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLocalVariable * self;
	self = (ValaLocalVariable*) base;
	g_return_if_fail (visitor != NULL);
	if (vala_local_variable_get_initializer (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_local_variable_get_initializer (self), visitor);
		vala_code_visitor_visit_end_full_expression (visitor, vala_local_variable_get_initializer (self));
	}
	if (vala_local_variable_get_variable_type (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_local_variable_get_variable_type (self), visitor);
	}
}


static void vala_local_variable_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaLocalVariable * self;
	self = (ValaLocalVariable*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_local_variable_get_initializer (self) == old_node) {
		vala_local_variable_set_initializer (self, new_node);
	}
}


static void vala_local_variable_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaLocalVariable * self;
	self = (ValaLocalVariable*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_local_variable_get_variable_type (self) == old_type) {
		vala_local_variable_set_variable_type (self, new_type);
	}
}


static gpointer _vala_code_node_ref0 (gpointer self) {
	return self ? vala_code_node_ref (self) : NULL;
}


static gboolean vala_local_variable_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaLocalVariable * self;
	gboolean result;
	ValaSymbol* _tmp10_;
	ValaBlock* block;
	self = (ValaLocalVariable*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		result = !vala_code_node_get_error ((ValaCodeNode*) self);
		return result;
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (vala_local_variable_get_variable_type (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_local_variable_get_variable_type (self), analyzer);
	}
	if (vala_local_variable_get_initializer (self) != NULL) {
		vala_expression_set_target_type (vala_local_variable_get_initializer (self), vala_local_variable_get_variable_type (self));
		vala_code_node_check ((ValaCodeNode*) vala_local_variable_get_initializer (self), analyzer);
	}
	if (vala_local_variable_get_variable_type (self) == NULL) {
		ValaDataType* _tmp0_;
		if (vala_local_variable_get_initializer (self) == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "var declaration not allowed without initializer");
			result = FALSE;
			return result;
		}
		if (vala_expression_get_value_type (vala_local_variable_get_initializer (self)) == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "var declaration not allowed with non-typed initializer");
			result = FALSE;
			return result;
		}
		vala_local_variable_set_variable_type (self, _tmp0_ = vala_data_type_copy (vala_expression_get_value_type (vala_local_variable_get_initializer (self))));
		_vala_code_node_unref0 (_tmp0_);
		vala_data_type_set_value_owned (vala_local_variable_get_variable_type (self), TRUE);
		vala_data_type_set_floating_reference (vala_local_variable_get_variable_type (self), FALSE);
		vala_expression_set_target_type (vala_local_variable_get_initializer (self), vala_local_variable_get_variable_type (self));
	}
	if (vala_local_variable_get_initializer (self) != NULL) {
		if (vala_expression_get_value_type (vala_local_variable_get_initializer (self)) == NULL) {
			gboolean _tmp1_;
			gboolean _tmp2_;
			_tmp1_ = FALSE;
			if (!VALA_IS_MEMBER_ACCESS (vala_local_variable_get_initializer (self))) {
				_tmp1_ = !VALA_IS_LAMBDA_EXPRESSION (vala_local_variable_get_initializer (self));
			} else {
				_tmp1_ = FALSE;
			}
			if (_tmp1_) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "expression type not allowed as initializer");
				result = FALSE;
				return result;
			}
			_tmp2_ = FALSE;
			if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_local_variable_get_initializer (self)))) {
				_tmp2_ = VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (self));
			} else {
				_tmp2_ = FALSE;
			}
			if (_tmp2_) {
				ValaMethod* m;
				ValaDelegateType* dt;
				ValaDelegate* cb;
				m = _vala_code_node_ref0 (VALA_METHOD (vala_expression_get_symbol_reference (vala_local_variable_get_initializer (self))));
				dt = _vala_code_node_ref0 (VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (self)));
				cb = _vala_code_node_ref0 (vala_delegate_type_get_delegate_symbol (dt));
				if (!vala_delegate_matches_method (cb, m)) {
					char* _tmp5_;
					char* _tmp4_;
					char* _tmp3_;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp5_ = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp3_ = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp4_ = vala_symbol_get_full_name ((ValaSymbol*) cb)));
					_g_free0 (_tmp5_);
					_g_free0 (_tmp4_);
					_g_free0 (_tmp3_);
					result = FALSE;
					_vala_code_node_unref0 (m);
					_vala_code_node_unref0 (dt);
					_vala_code_node_unref0 (cb);
					return result;
				}
				vala_expression_set_value_type (vala_local_variable_get_initializer (self), vala_local_variable_get_variable_type (self));
				_vala_code_node_unref0 (m);
				_vala_code_node_unref0 (dt);
				_vala_code_node_unref0 (cb);
			} else {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "expression type not allowed as initializer");
				result = FALSE;
				return result;
			}
		}
		if (!vala_data_type_compatible (vala_expression_get_value_type (vala_local_variable_get_initializer (self)), vala_local_variable_get_variable_type (self))) {
			char* _tmp8_;
			char* _tmp7_;
			char* _tmp6_;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp8_ = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp6_ = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_local_variable_get_initializer (self))), _tmp7_ = vala_code_node_to_string ((ValaCodeNode*) vala_local_variable_get_variable_type (self))));
			_g_free0 (_tmp8_);
			_g_free0 (_tmp7_);
			_g_free0 (_tmp6_);
			result = FALSE;
			return result;
		}
		if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_local_variable_get_initializer (self)))) {
			gboolean _tmp9_;
			_tmp9_ = FALSE;
			if (!VALA_IS_POINTER_TYPE (vala_local_variable_get_variable_type (self))) {
				_tmp9_ = !vala_data_type_get_value_owned (vala_local_variable_get_variable_type (self));
			} else {
				_tmp9_ = FALSE;
			}
			if (_tmp9_) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
				result = FALSE;
				return result;
			}
		}
	}
	vala_scope_add (vala_symbol_get_scope (vala_semantic_analyzer_get_current_symbol (analyzer)), vala_symbol_get_name ((ValaSymbol*) self), (ValaSymbol*) self);
	block = _vala_code_node_ref0 ((_tmp10_ = vala_semantic_analyzer_get_current_symbol (analyzer), VALA_IS_BLOCK (_tmp10_) ? ((ValaBlock*) _tmp10_) : NULL));
	if (block != NULL) {
		vala_block_add_local_variable (block, self);
	}
	vala_symbol_set_active ((ValaSymbol*) self, TRUE);
	result = !vala_code_node_get_error ((ValaCodeNode*) self);
	_vala_code_node_unref0 (block);
	return result;
}


ValaExpression* vala_local_variable_get_initializer (ValaLocalVariable* self) {
	ValaExpression* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_initializer;
	return result;
}


void vala_local_variable_set_initializer (ValaLocalVariable* self, ValaExpression* value) {
	ValaExpression* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_initializer = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_initializer), _tmp0_);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_initializer, (ValaCodeNode*) self);
	}
}


ValaDataType* vala_local_variable_get_variable_type (ValaLocalVariable* self) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_variable_type;
	return result;
}


void vala_local_variable_set_variable_type (ValaLocalVariable* self, ValaDataType* value) {
	ValaDataType* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_variable_type = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_variable_type), _tmp0_);
	if (self->priv->_variable_type != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_variable_type, (ValaCodeNode*) self);
	}
}


gboolean vala_local_variable_get_is_result (ValaLocalVariable* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_is_result;
	return result;
}


void vala_local_variable_set_is_result (ValaLocalVariable* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_result = value;
}


gboolean vala_local_variable_get_floating (ValaLocalVariable* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_floating;
	return result;
}


void vala_local_variable_set_floating (ValaLocalVariable* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_floating = value;
}


gboolean vala_local_variable_get_captured (ValaLocalVariable* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_captured;
	return result;
}


void vala_local_variable_set_captured (ValaLocalVariable* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_captured = value;
}


gboolean vala_local_variable_get_no_init (ValaLocalVariable* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_no_init;
	return result;
}


void vala_local_variable_set_no_init (ValaLocalVariable* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_init = value;
}


static void vala_local_variable_class_init (ValaLocalVariableClass * klass) {
	vala_local_variable_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_local_variable_finalize;
	g_type_class_add_private (klass, sizeof (ValaLocalVariablePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_local_variable_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_local_variable_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_local_variable_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_local_variable_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_local_variable_real_check;
}


static void vala_local_variable_instance_init (ValaLocalVariable * self) {
	self->priv = VALA_LOCAL_VARIABLE_GET_PRIVATE (self);
}


static void vala_local_variable_finalize (ValaCodeNode* obj) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (obj);
	_vala_code_node_unref0 (self->priv->_initializer);
	_vala_code_node_unref0 (self->priv->_variable_type);
	VALA_CODE_NODE_CLASS (vala_local_variable_parent_class)->finalize (obj);
}


GType vala_local_variable_get_type (void) {
	static GType vala_local_variable_type_id = 0;
	if (vala_local_variable_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaLocalVariableClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_local_variable_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaLocalVariable), 0, (GInstanceInitFunc) vala_local_variable_instance_init, NULL };
		vala_local_variable_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaLocalVariable", &g_define_type_info, 0);
	}
	return vala_local_variable_type_id;
}




