/*
 * mp2_decoder.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * Copyright (C) 2003 Sven Goethel, <sven@jausoft.com>
 */

#include "types.h"

#include "mp2_decoder.h"

#include "bitstreamout.h"

#define USE_LOOK_UP_TABLE	1	// For getting speed
#define USE_LAST_FRAME		1	// In case of CRC error

# define debug_mp2(args...)
// # define debug_mp2(args...)	fprintf(stderr, args)

/*
struct LPCMHeader { int id:8;	 	   // id
		    int frame_count:8;     // number of frames
		    int access_ptr:16;     // first acces unit pointer, i.e. start of audio frame
		    bool emphasis:1;       // audio emphasis on-off
		    bool mute:1;	   // audio mute on-off
		    bool reserved:1;       // reserved
		    int frame_number:5;    // audio frame number
		    int quant_wlen:2;      // quantization word length
		    int sample_freq:2;     // audio sampling frequency (48khz=0, 96khz=1, 44,1khz=2, 32khz=3)
		    bool reserved2:1;      // reserved
		    int chan_count:3;      // number of audio channels - 1 (e.g. stereo = 1)
		    int dyn_range_ctrl:8;  // dynamic range control (0x80 if off)
		    };
*/


// ----------------------------------------------------------------


// ----------------------------------------------------------------


extern ctrl_t setup;

// This function requires two arguments:
//   first is the start of the data segment
//   second is the tail of the data segment
//   Note: the start will be shift by the amount
//   for a full MP2 frame found in the segment.

int MP2Decoder::instancenr=0;

MP2Decoder::MP2Decoder() : 
firstTime(true), madInit(false) 
{ 
  Mp2InputBuffer_cur = (uint_8 *)shm_malloc(MP2_BUFFER_SIZE);
  Mp2InputBuffer_nxt = (uint_8 *)shm_malloc(MP2_BUFFER_SIZE);
  lpcmFrame_ptr      = (LPCMFrame *) shm_malloc(sizeof(LPCMFrame));

  ExternalPCMData = NULL;
  ExternalPCMDataSize = 0 ;

#ifdef FILE_COPY
  fptr_raw=NULL; fptr_mp2=NULL;
  fnr=0;
#endif
  init(); 
  instancenr++;
}

MP2Decoder::~MP2Decoder() 
{ 
  finish(); 
  instancenr--;

  if (Mp2InputBuffer_cur)
	shm_free(Mp2InputBuffer_cur);
  Mp2InputBuffer_cur = NULL;

  if (Mp2InputBuffer_nxt)
	shm_free(Mp2InputBuffer_nxt);
  Mp2InputBuffer_nxt = NULL;

  if (lpcmFrame_ptr)
	shm_free(lpcmFrame_ptr);
  lpcmFrame_ptr = NULL;
}

void  MP2Decoder::finish()
{
	if(firstTime)
		return;
	debug_mp2("****** mp2_decode_pcm FINISH\n");
	mad_stream_finish(&m_stream);
	mad_frame_finish(&m_frame);
	#ifdef FILE_COPY
		if(fptr_raw)
			fclose(fptr_raw);
		fptr_raw=NULL;
		if(fptr_mp2)
			fclose(fptr_mp2);
		fptr_mp2=NULL;
	#endif
	madInit   = false;
}

void  MP2Decoder::init()
{
	debug_mp2("****** mp2_decode_pcm INIT\n");
	if ( madInit )
		finish();

	size=0;
	pcm_netto_size=0;
	remaining_samples = 0;
	remaining_mp2 = 0;
	digested_mp2 = 0;
	sample_rate = 0;
	channels = 0;
	abs_mp2_pos = NULL;
	delay = 0;

	mad_stream_init(&m_stream);
	mad_frame_init(&m_frame);
	mad_header_init(&m_frame.header);
	mad_synth_init(&m_synth);
	mad_timer_reset(&m_timer);
	FrameCount=0;

  	memset(lpcmFrame_ptr,0,sizeof(LPCMFrame));

	scale.Init();

	madInit   = true;
	debug_mp2("****** mp2_decode_pcm INIT done\n");
}

void  MP2Decoder::empty()
{
	debug_mp2("****** mp2_decode_pcm EMPTY\n");

	size=0;
	pcm_netto_size=0;
	remaining_samples = 0;
	remaining_mp2 = 0;
	digested_mp2 = 0;
	sample_rate = 0;
	channels = 0;
	delay = 0;

	FrameCount=0;

	debug_mp2("****** mp2_decode_pcm Empty done\n");
}

void MP2Decoder::setExternalOutBuffer ( unsigned char * ExternalBuffer,
				       size_t	  ExternalBufferSize )
{
    if(firstTime) {
	init();
	firstTime = false;
    }

    this->ExternalPCMData = ExternalBuffer;
    this->ExternalPCMDataSize=ExternalBufferSize;
}

int  MP2Decoder::pushData ( const uint_8 *head, int len )
{

	if (len > MP2_BUFFER_SIZE)	// Not more than buffer size
	    len = MP2_BUFFER_SIZE;
	int old = len;			// Remember this value

	int max = MP2_BUFFER_SIZE - remaining_mp2;
	if (len > max)			// Beware of the remaining bytes
	    len = max;

	if ((remaining_mp2 > 0) && (len >= MAD_BUFFER_GUARD)) 
	{
	    struct mad_stream stream;
	    //
	    // Find an mp2 sync frame.
	    //
	    mad_stream_buffer(&stream, head, len);
	    stream.sync = 0;

	    if ((mad_stream_sync(&stream) == 0) && ((stream.ptr.byte - head) == 0))
	    {
		debug_mp2 ("mp2_decode pushMP2Data found SYNC at start, remove remaining ..\n");
		remaining_mp2=0;
		len=old;		// Use previous len value
	    }
	}

	debug_mp2 ("mp2_decode merging buffer: 0x%4.4X+0x%4.4X=0x%4.4X\n",
		    remaining_mp2, len, remaining_mp2+len);
	memcpy(Mp2InputBuffer_nxt+remaining_mp2, head, len);
	remaining_mp2+=len;

	return len;			// Return the number of used bytes
}

bool  MP2Decoder::decode(int maxwritesize)
{
    struct mad_pcm *m_pcm=0;
    const uint_8 * out = NULL;
    const uint_8 * mp2_head = NULL;
    const uint_8 * mp2_tail = NULL;
    int len = 0;

    if(firstTime) {
	init();
	firstTime = false;
    }

#ifdef FILE_COPY
    if(fptr_raw==NULL)
    {
	char fnr_name[200];
	sprintf(fnr_name,"/tmp/mp2_i%d_%4.4d.raw", instancenr, fnr);
	debug_mp2("****** NEW FILE %s\n", fnr_name);
	fptr_raw = fopen(fnr_name, "wb");
	sprintf(fnr_name,"/tmp/mp2_i%d_%4.4d.mp2", instancenr, fnr++);
	debug_mp2("****** NEW FILE %s\n", fnr_name);
	fptr_mp2 = fopen(fnr_name, "wb");
    }
#endif

    if ( remaining_samples > 0 ) {
	    debug_mp2("****** mp2_decode_pcm -> REMAINING_SAMPLES = 0x%X ( new 0x%4.4X, rem 0x%4.4X)\n", remaining_samples, remaining_mp2);
    	    pushSamples2PCM(maxwritesize);
	    return true;
    }

    if ( remaining_mp2 < MAD_BUFFER_GUARD ) {
    	debug_mp2("****** mp2_decode_pcm start: got SHORT remaining 0x%4.4X\n", remaining_mp2);
	return false;
    }

    debug_mp2("****** mp2_decode_pcm start: got remaining 0x%4.4X\n", remaining_mp2);

    digested_mp2 = 0;
    size = 0;
    pcm_netto_size=0;
    sample_rate = 0;
    channels = 0;

  // PS1magic
  if ( ExternalPCMData == NULL ) {
	  lpcmFrame_ptr->PES[0]=0x00;
	  lpcmFrame_ptr->PES[1]=0x00;
	  lpcmFrame_ptr->PES[2]=0x01;
	  lpcmFrame_ptr->PES[3]=0xbd;
	  lpcmFrame_ptr->PES[6]=0x87;
	  lpcmFrame_ptr->LPCM[0]=0xa0; // substream ID
	  lpcmFrame_ptr->LPCM[1]=0xff;
	  lpcmFrame_ptr->LPCM[5]=0x01;
	  lpcmFrame_ptr->LPCM[6]=0x80;
  }

    // swap buffers: 
    //	next -> cur
    //  cur  -> next
    {
	    uint_8* tmp = Mp2InputBuffer_cur;
	    Mp2InputBuffer_cur = Mp2InputBuffer_nxt;
	    Mp2InputBuffer_nxt = tmp;
	    len = remaining_mp2;
	    remaining_mp2 = 0;
    }
    
    //
    // Find an mp2 sync frame.
    //
    mad_stream_buffer(&m_stream, Mp2InputBuffer_cur, len);

    m_stream.sync = 0;

    debug_mp2("mp2_decode_pcm STREAM SYNC ");
    if ( mad_stream_sync ( &m_stream ) != 0 )
    {
	    debug_mp2("FAILED %s\n", mad_stream_errorstr(&m_stream));
	    if ( m_stream.error == MAD_ERROR_BUFLEN ) {
		remaining_mp2=len;
		if( remaining_mp2 > 0 ) 
			memcpy(Mp2InputBuffer_nxt, Mp2InputBuffer_cur, remaining_mp2);
	    }
	    goto done;
    }
    out = m_stream.ptr.byte;
    mp2_head = out;
    debug_mp2("ok at pos 0x%4.4X (abs %p)\n", 
	mp2_head-Mp2InputBuffer_cur, abs_mp2_pos+(mp2_head-Mp2InputBuffer_cur));

    debug_mp2("mp2_decode_pcm DECODE ");
    if ( mad_frame_decode (&m_frame, &m_stream) != 0 )
    {
	debug_mp2("FAILED %s at 0x%4.4X\n", 
		mad_stream_errorstr(&m_stream),
		m_stream.next_frame - mp2_head );

	if ( m_stream.error == MAD_ERROR_BUFLEN ) 
	{
		if ( m_stream.next_frame != NULL ) {
			remaining_mp2=(m_stream.bufend-m_stream.next_frame);
			if( remaining_mp2 > 0 ) 
			    memcpy(Mp2InputBuffer_nxt, m_stream.next_frame, remaining_mp2);
		}
		if( m_stream.next_frame - mp2_head == 0 ) {
			debug_mp2("REST DATA AT START 0x%4.4X, at 0x%4.4X\n", 
				m_stream.bufend-m_stream.next_frame, 
				m_stream.next_frame - mp2_head);
			goto done;
		} else {
			debug_mp2("REST DATA WITHIN 0x%4.4X, at 0x%4.4X\n", 
				m_stream.bufend-m_stream.next_frame, 
				m_stream.next_frame - mp2_head);
		}
	} 
	else {
		if ( MAD_RECOVERABLE(m_stream.error) ) 
		{
			debug_mp2("ERRcase recoverable error at 0x%X\n", 
				m_stream.next_frame - mp2_head);
		} else {
			debug_mp2("ERRcase UNrecoverable error at 0x%X\n", 
				m_stream.next_frame - mp2_head);
			mad_stream_init(&m_stream);
			mad_frame_init(&m_frame);
			goto done;
		}
		if ( m_stream.next_frame != NULL ) {
			remaining_mp2=(m_stream.bufend-m_stream.next_frame);
			if( remaining_mp2 > 0 ) 
			    memcpy(Mp2InputBuffer_nxt, m_stream.next_frame, remaining_mp2);
		} else {
			debug_mp2("ERRcase WARNING NO SAVED DATA \n");
		}
		goto done;
	}
    }
    out = m_stream.next_frame ;

    mp2_tail = out;
    digested_mp2 = mp2_tail-mp2_head;

    debug_mp2("ok 0x%4.4X RANGE [0x%4.4X..0x%4.4X[ ( abs [0x%X..0x%X[ )\n", 
	digested_mp2, 
	0, digested_mp2, abs_mp2_pos, abs_mp2_pos+digested_mp2);

    abs_mp2_pos += digested_mp2;

#ifdef FILE_COPY
    if(fptr_mp2!=NULL)
	    fwrite(mp2_head, 1, (mp2_tail-mp2_head), fptr_mp2);
#endif

    remaining_mp2=(m_stream.bufend-m_stream.next_frame);
    if( remaining_mp2 > 0 ) 
	memcpy(Mp2InputBuffer_nxt, m_stream.next_frame, remaining_mp2);
    debug_mp2("DECODE REMAINING SAVE: 0x%4.4X\n", remaining_mp2);

    FrameCount++;
    
    if ( delay > 0 ) {
	uint_32 ms = mad_timer_count(m_frame.header.duration , MAD_UNITS_MILLISECONDS);

	if ( delay > ms ) {
		debug_mp2("MP2 Decoder PTS: HUGE delay - ms: %lu - %lu = %lu\n", delay, ms, delay-ms);
		delay  -= ms;
		goto cont;
	}
	debug_mp2("MP2 Decoder PTS: delay - ms: %lu - %lu = 0\n", delay, ms);
	delay  = 0;
	goto done;
    }

    mad_timer_add(&m_timer,m_frame.header.duration);

    mad_synth_frame(& m_synth, &m_frame);

    m_pcm = & m_synth.pcm ;

    nsamples[0]=nsamples[1]=m_pcm->length;
    data[0]=m_pcm->samples[0];
    data[1]=m_pcm->channels>1 ? m_pcm->samples[1]:0;

    channels=m_pcm->channels;

    // def. to 48000
    sample_rate=48000;  
    if ( ExternalPCMData == NULL ) 
	    lpcmFrame_ptr->LPCM[5]&=0xcf;

    switch(m_pcm->samplerate) {    // If one of the supported frequencies, do it without resampling.
      case 96000:		
	sample_rate=48000;   // -> downsampling to default
	break;

      //case 48000: // this is already the default ...
      //  sample_rate=48000;  
      //  break;

      case 11025:
      case 22050:
      case 44100:
	sample_rate=44100;   
	if ( ExternalPCMData == NULL ) 
		lpcmFrame_ptr->LPCM[5]|=2<<4;
	break;
      case 8000:
      case 16000:
      case 32000:
	sample_rate=32000;   
	if ( ExternalPCMData == NULL ) 
		lpcmFrame_ptr->LPCM[5]|=3<<4;
	break;
    }

    debug_mp2("DECODE summary .. #%lu: %lds, done 0x%4.4X, rem 0x%4.4X,\n\tsamples 0x%X, ch %d, sf %d -> %d\n", 
		FrameCount, m_timer.seconds, digested_mp2, remaining_mp2,
		m_pcm->length, m_pcm->channels,
		m_pcm->samplerate, sample_rate
		);

    if (m_pcm->samplerate != sample_rate) {
	    if(resample[0].SetInputRate(m_pcm->samplerate,sample_rate)) {
	      nsamples[0]=resample[0].ResampleBlock(nsamples[0],data[0]);
	      data[0]    =resample[0].Resampled();
	      debug_mp2("RESAMPLED data 0 0x%X\n", nsamples[0]);
	    }
	    if(data[1] && resample[1].SetInputRate(m_pcm->samplerate,sample_rate)) {
	      nsamples[1]=resample[1].ResampleBlock(nsamples[1],data[1]);
	      data[1]    =resample[1].Resampled();
	      debug_mp2("RESAMPLED data 1 0x%X\n", nsamples[1]);
	    }
    }

    pushSamples2PCM(maxwritesize);
    goto cont;

cont:

    return true;

done:

    return false;
}

void  MP2Decoder::pushSamples2PCM(int maxwritesize)
{
    if(firstTime) {
	init();
	firstTime = false;
    }

    if ( delay > 0 ) 
	return;

    debug_mp2("mp2_decode_pcm scaleBlock samples: 0x%X, max netto writesize 0x%X, ", 
	(int)nsamples[0], (int)maxwritesize);

    if(maxwritesize<0) {
	    debug_mp2("TOO SHORT exit\n");
	    goto done;
    }
    debug_mp2("OK\n");

    if ( ExternalPCMData == NULL ) {
	    if(maxwritesize==0 || (size_t)maxwritesize>sizeof(lpcmFrame_ptr->Data)) 
		maxwritesize=sizeof(lpcmFrame_ptr->Data);
    } else {
	    if(maxwritesize==0 || (size_t)maxwritesize>ExternalPCMDataSize)
		maxwritesize=ExternalPCMDataSize;
    }

    if ( ExternalPCMData == NULL ) {
	    size=scale.ScaleBlock(
		lpcmFrame_ptr->Data, maxwritesize,
		nsamples[0],data[0],data[1], (test_setup(MP2DITHER)) ? amDither : amRound );
	    remaining_samples = nsamples[0];
	    /**
		size=Convert(lpcmFrame_ptr->Data, maxwritesize);
		remaining_samples = 0;
	    */

	    #ifdef FILE_COPY
		if(fptr_raw!=NULL)
		    fwrite(lpcmFrame_ptr->Data, 1, size, fptr_raw );
	    #endif
    } else {
	    size=scale.ScaleBlock(
		ExternalPCMData, maxwritesize,
		nsamples[0],data[0],data[1], (test_setup(MP2DITHER)) ? amDither : amRound );
	    remaining_samples = nsamples[0];
	    /**
		size=Convert(ExternalPCMData, maxwritesize);
		remaining_samples = 0;
	    */

	    #ifdef FILE_COPY
		if(fptr_raw!=NULL)
		    fwrite(ExternalPCMData, 1, size, fptr_raw );
	    #endif
    }


    debug_mp2("mp2_decode_pcm payload 0x%X/0x%X\n", 
	size, maxwritesize);

    debug_mp2("mp2_decode_pcm ... n0 0x%X, n1 0x%X \n", 
	(int)nsamples[0], (int)nsamples[1]);

    if ( size <= 0 ) {
	debug_mp2("mp2_decode_pcm payload %d/%d failed\n", 
		size, maxwritesize);
	goto done;
    }

    pcm_netto_size = size;

    if ( ExternalPCMData == NULL ) {
	    size += sizeof(lpcmFrame_ptr->LPCM)+LEN_CORR;
	    lpcmFrame_ptr->PES[4]=size>>8;
	    lpcmFrame_ptr->PES[5]=size;
	    size -= LEN_CORR;

	    size += sizeof(lpcmFrame_ptr->PES);
    }

done:

    return;

}

int  MP2Decoder::Convert(unsigned char * Data, size_t DataSize)
{
	unsigned char * OutputPtr = Data;
	unsigned short  Sample;
	int i;

	if(firstTime) {
	    init();
	    firstTime = false;
	}

	for(i=0;i<m_synth.pcm.length;i++)
	{
		if( (unsigned int)((OutputPtr-Data)+4) >= DataSize )
			break;

		/* Left channel */
		Sample=MadFixedToUshort(data[0][i]);
		*(OutputPtr++)=Sample>>8;
		*(OutputPtr++)=Sample&0xff;

		/* Right channel. If the decoded stream is monophonic then
		 * the right output channel is the same as the left one.
		 */
		if(MAD_NCHANNELS(&m_frame.header)==2 && data[1]!=NULL)
			Sample=MadFixedToUshort(data[1][i]);
		*(OutputPtr++)=Sample>>8;
		*(OutputPtr++)=Sample&0xff;
	}

	return OutputPtr-Data;
}

/****************************************************************************
 * Converts a sample from mad's fixed point number format to an unsigned    *
 * short (16 bits).							    *
 ****************************************************************************/
unsigned short MP2Decoder::MadFixedToUshort(mad_fixed_t Fixed)
{
	/* A fixed point number is formed of the following bit pattern:
	 *
	 * SWWWFFFFFFFFFFFFFFFFFFFFFFFFFFFF
	 * MSB			  LSB
	 * S ==> Sign (0 is positive, 1 is negative)
	 * W ==> Whole part bits
	 * F ==> Fractional part bits
	 *
	 * This pattern contains MAD_F_FRACBITS fractional bits, one
	 * should alway use this macro when working on the bits of a fixed
	 * point number. It is not guaranteed to be constant over the
	 * different platforms supported by libmad.
	 *
	 * The unsigned short value is formed by the least significant
	 * whole part bit, followed by the 15 most significant fractional
	 * part bits. Warning: this is a quick and dirty way to compute
	 * the 16-bit number, madplay includes much better algorithms.
	 */
	Fixed=Fixed>>(MAD_F_FRACBITS-15);
	return((unsigned short)Fixed);
}


// ----------------------------------------------------------------

bool cResample::SetInputRate(unsigned int oldrate, unsigned int newrate)
{
  if(oldrate<8000 || oldrate>newrate*6) { // out of range
    esyslog("WARNING: samplerate %d out of range 8000-%d\n",oldrate,newrate*6);
    return 0;
    }
  ratio=mad_f_tofixed((double)oldrate/(double)newrate);
  step=0; last=0;
#ifdef DEBUG
  static mad_fixed_t oldratio=0;
  if(oldratio!=ratio) {
    printf("mad: new resample ratio %f (from %d kHz to %d kHz)\n",mad_f_todouble(ratio),oldrate,newrate);
    oldratio=ratio;
    }
#endif
  return ratio!=MAD_F_ONE;
}

unsigned int cResample::ResampleBlock(unsigned int nsamples, const mad_fixed_t *old)
{
  // This resampling algorithm is based on a linear interpolation, which is
  // not at all the best sounding but is relatively fast and efficient.
  //
  // A better algorithm would be one that implements a bandlimited
  // interpolation.

  mad_fixed_t *nsam=resampled;
  const mad_fixed_t *end=old+nsamples;
  const mad_fixed_t *begin=nsam;

  if(step < 0) {
    step = mad_f_fracpart(-step);

    while (step < MAD_F_ONE) {
      *nsam++ = step ? last+mad_f_mul(*old-last,step) : last;
      step += ratio;
      if(((step + 0x00000080L) & 0x0fffff00L) == 0)
	step = (step + 0x00000080L) & ~0x0fffffffL;
      }
    step -= MAD_F_ONE;
    }

  while (end - old > 1 + mad_f_intpart(step)) {
    old += mad_f_intpart(step);
    step = mad_f_fracpart(step);
    *nsam++ = step ? *old + mad_f_mul(old[1] - old[0], step) : *old;
    step += ratio;
    if (((step + 0x00000080L) & 0x0fffff00L) == 0)
      step = (step + 0x00000080L) & ~0x0fffffffL;
    }

  if (end - old == 1 + mad_f_intpart(step)) {
    last = end[-1];
    step = -step;
    }
  else step -= mad_f_fromint(end - old);

  return nsam-begin;
}


// ----------------------------------------------------------------

void cScale::Init(void)
{
#ifdef DEBUG
  clipped_samples=0; peak_clipping=peak_sample=0;
#endif
  memset(&leftD,0,sizeof(leftD));
  memset(&rightD,0,sizeof(rightD));
}

void cScale::Stats(void)
{
#ifdef DEBUG
  printf("mp3: scale stats clipped=%ld peak_clip=%f peak=%f\n",
	 clipped_samples,mad_f_todouble(peak_clipping),mad_f_todouble(peak_sample));
#endif
}

// gather signal statistics while clipping
mad_fixed_t cScale::Clip(mad_fixed_t sample, bool stats)
{
#ifndef DEBUG
  if (sample > MAX) sample = MAX;
  if (sample < MIN) sample = MIN;
#else
  if(!stats) {
    if (sample > MAX) sample = MAX;
    if (sample < MIN) sample = MIN;
    }
  else {
    if (sample >= peak_sample) {
      if (sample > MAX) {
	++clipped_samples;
	if (sample - MAX > peak_clipping)
	  peak_clipping = sample - MAX;
	sample = MAX;
	}
      peak_sample = sample;
      }
    else if (sample < -peak_sample) {
      if (sample < MIN) {
	++clipped_samples;
	if (MIN - sample > peak_clipping)
	  peak_clipping = MIN - sample;
	sample = MIN;
	}
      peak_sample = -sample;
      }
    }
#endif
  return sample;
}

// generic linear sample quantize routine
signed long cScale::LinearRound(mad_fixed_t sample)
{
  // round
  sample += (1L << (MAD_F_FRACBITS - OUT_BITS));
  // clip
  sample=Clip(sample);
  // quantize and scale
  return sample >> (MAD_F_FRACBITS + 1 - OUT_BITS);
}

// 32-bit pseudo-random number generator
unsigned long cScale::Prng(unsigned long state)
{
  return (state * 0x0019660dL + 0x3c6ef35fL) & 0xffffffffL;
}

// generic linear sample quantize and dither routine
signed long cScale::LinearDither(mad_fixed_t sample, struct dither *dither)
{
  unsigned int scalebits;
  mad_fixed_t output, mask, random;

  // noise shape
  sample += dither->error[0] - dither->error[1] + dither->error[2];
  dither->error[2] = dither->error[1];
  dither->error[1] = dither->error[0] / 2;
  // bias
  output = sample + (1L << (MAD_F_FRACBITS + 1 - OUT_BITS - 1));
  scalebits = MAD_F_FRACBITS + 1 - OUT_BITS;
  mask = (1L << scalebits) - 1;
  // dither
  random  = Prng(dither->random);
  output += (random & mask) - (dither->random & mask);
  dither->random = random;
  // clip
  output=Clip(output);
  sample=Clip(sample,false);
  // quantize
  output &= ~mask;
  // error feedback
  dither->error[0] = sample - output;
  // scale
  return output >> scalebits;
}

// write a block of signed 16-bit big-endian PCM samples
unsigned int cScale::ScaleBlock(unsigned char *data, unsigned int size, unsigned int &nsamples, const mad_fixed_t * &left, const mad_fixed_t * &right, eAudioMode mode)
{
  signed int sample;
  unsigned int len, res;

  len=size/OUT_FACT; res=size;
  if(len>nsamples) { len=nsamples; res=len*OUT_FACT; }
  nsamples-=len;

  if(right) {  // stereo
    switch (mode) {
      case amRound:
	while (len--) {
	  sample  = LinearRound(*left++);
	  *data++ = sample >> 8;
	  *data++ = sample >> 0;
	  sample  = LinearRound(*right++);
	  *data++ = sample >> 8;
	  *data++ = sample >> 0;
	  }
	break;
      case amDither:
	while (len--) {
	  sample  = LinearDither(*left++,&leftD);
	  *data++ = sample >> 8;
	  *data++ = sample >> 0;
	  sample  = LinearDither(*right++,&rightD);
	  *data++ = sample >> 8;
	  *data++ = sample >> 0;
	  }
	break;
      }
    }
  else {  // mono, duplicate left channel
    switch (mode) {
      case amRound:
	while (len--) {
	  sample  = LinearRound(*left++);
	  *data++ = sample >> 8;
	  *data++ = sample >> 0;
	  *data++ = sample >> 8;
	  *data++ = sample >> 0;
	  }
	break;
      case amDither:
	while (len--) {
	  sample  = LinearDither(*left++,&leftD);
	  *data++ = sample >> 8;
	  *data++ = sample >> 0;
	  *data++ = sample >> 8;
	  *data++ = sample >> 0;
	  }
	break;
      }
    }
 return res;
}


// ----------------------------------------------------------------
