/*
 * MP3/MPlayer plugin to VDR (C++)
 *
 * (C) 2001,2002 Stefan Huelswitt <huels@iname.com>
 *
 * This code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 */

#include <ctype.h>
#include <dirent.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <unistd.h>
#include <typeinfo>

#include <vdr/menuitems.h>
#include <vdr/status.h>
#include <vdr/recording.h>

#include "common.h"
#include "menu.h"
#include "data-mp3.h"
#include "menu-mp3.h"
#include "player-mp3.h"
#include "decoder.h"
#include "i18n.h"

// --- cMenuSetupMP3 --------------------------------------------------------

cMenuSetupMP3::cMenuSetupMP3(void)
{
  static const char allowed[] = { "abcdefghijklmnopqrstuvwxyz0123456789-_" };
  static const char *cddb[3];
  cddb[0]=tr("disabled");
  cddb[1]=tr("local only");
  cddb[2]=tr("local&remote");

  data=MP3Setup;
  SetSection(tr("MP3"));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Audio mode"),            &data.AudioMode, tr("Round"), tr("Dither")));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Use 48kHz mode only"),   &data.Only48kHz));
  Add(new cMenuEditIntItem( tr("Setup.MP3$Display mode"),          &data.DisplayMode, 1, 3));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Background mode"),       &data.BackgrMode, tr("Black"), tr("Live")));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Initial loop mode"),     &data.InitLoopMode));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Initial shuffle mode"),  &data.InitShuffleMode));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Id3 scan"),              &data.Id3PreScan, tr("On Play"), tr("Background")));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Editor display mode"),   &data.EditorMode, tr("Filenames"), tr("ID3 names")));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Mainmenu mode"),         &data.MenuMode, tr("Playlists"), tr("Browser")));
  Add(new cMenuEditIntItem( tr("Setup.MP3$Normalizer level"),      &data.TargetLevel, 0, MAX_TARGET_LEVEL));
  Add(new cMenuEditIntItem( tr("Setup.MP3$Limiter level"),         &data.LimiterLevel, MIN_LIMITER_LEVEL, 100));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Use HTTP proxy"),        &data.UseProxy));
  Add(new cMenuEditStrItem( tr("Setup.MP3$HTTP proxy host"),       data.ProxyHost,MAX_HOSTNAME,allowed));
  Add(new cMenuEditIntItem( tr("Setup.MP3$HTTP proxy port"),       &data.ProxyPort,1,65535));
  Add(new cMenuEditStraItem(tr("Setup.MP3$CDDB for CD-Audio"),     &data.UseCddb,3,cddb));
  Add(new cMenuEditStrItem( tr("Setup.MP3$CDDB server"),           data.CddbHost,MAX_HOSTNAME,allowed));
  Add(new cMenuEditIntItem( tr("Setup.MP3$CDDB port"),             &data.CddbPort,1,65535));
}

void cMenuSetupMP3::Store(void)
{
  MP3Setup=data;
  SetupStore("InitLoopMode",     MP3Setup.InitLoopMode   );
  SetupStore("InitShuffleMode",  MP3Setup.InitShuffleMode);
  SetupStore("AudioMode",        MP3Setup.AudioMode      );
  SetupStore("Id3PreScan",       MP3Setup.Id3PreScan     );
  SetupStore("EditorMode",       MP3Setup.EditorMode     );
  SetupStore("DisplayMode",      MP3Setup.DisplayMode    );
  SetupStore("BackgrMode",       MP3Setup.BackgrMode     );
  SetupStore("MenuMode",         MP3Setup.MenuMode       );
  SetupStore("TargetLevel",      MP3Setup.TargetLevel    );
  SetupStore("LimiterLevel",     MP3Setup.LimiterLevel   );
  SetupStore("Only48kHz",        MP3Setup.Only48kHz      );
  SetupStore("UseProxy",         MP3Setup.UseProxy       );
  SetupStore("ProxyHost",        MP3Setup.ProxyHost      );
  SetupStore("ProxyPort",        MP3Setup.ProxyPort      );
  SetupStore("UseCddb",          MP3Setup.UseCddb        );
  SetupStore("CddbHost",         MP3Setup.CddbHost       );
  SetupStore("CddbPort",         MP3Setup.CddbPort       );
}

// --- cMP3Control --------------------------------------------------------

class cMP3Control : public cMP3PlayerControl {
private:
  static cPlayList *plist;
  bool visible, shown, bigwin;
  time_t timeoutShow, greentime, oktime;
  int lastkeytime, number;
  bool selecting, selecthide;
//
  cMP3PlayInfo *lastMode;
  time_t fliptime, listtime;
  int hashlist[MAXOSDHEIGHT];
  int flip, flipint, top;
  int lastIndex, lastTotal, lastTop;
  int framesPerSecond;
//
  bool jumpactive, jumphide, jumpsecs;
  int jumpmm;
//
  void ShowTimed(int Seconds=0);
  void ShowProgress(bool open=false, bool bigWin=false);
  char Hash(const char *text);
  void DisplayInfo(const char *s=0);
  void JumpDisplay(void);
  void JumpProcess(eKeys Key);
  void Jump(void);
public:
  cMP3Control(void);
  virtual ~cMP3Control();
  virtual eOSState ProcessKey(eKeys Key);
  virtual void Show(void) { ShowTimed(); }
  virtual void Hide(void);
  bool Visible(void) { return visible; }
  static bool SetPlayList(cPlayList *PList);
  };

cPlayList *cMP3Control::plist=0;

cMP3Control::cMP3Control(void)
:cMP3PlayerControl(plist)
{
  visible=shown=bigwin=selecting=selecthide=jumpactive=jumphide=false;
  timeoutShow=greentime=oktime=0;
  lastkeytime=number=0;
  lastMode=0;
  framesPerSecond=SecondsToFrames(1);
  plist=0; // the player is responsible to delete playlist
}

cMP3Control::~cMP3Control()
{
  delete lastMode;
  Hide();
  Stop();
  cStatus::MsgReplaying(this, NULL);
}

bool cMP3Control::SetPlayList(cPlayList *PList)
{
  delete plist;
  plist=PList;

  if(plist) {
    cControl *control=cControl::Control();
    if(control && typeid(*control)==typeid(cMP3Control)) {    // is there a running MP3 player?
      static_cast<cMP3Control*>(control)->NewPlaylist(plist); // signal the running player to load the new playlist
      plist=0;
      return true;
      }
    cControl::Launch(new cMP3Control);
    }
  return false;
}

void cMP3Control::ShowTimed(int Seconds)
{
  if(!visible) {
    ShowProgress(true);
    if(Seconds>0) timeoutShow=time(0)+Seconds;
    }
}

void cMP3Control::Hide(void)
{
  if(visible) {
    Interface->Close();
    needsFastResponse=visible=bigwin=false;
    }
}

#define CTAB    11 // some tabbing values for the progress display
#define CTAB2   5

#define PL_ROWS (Setup.OSDheight-3)

void cMP3Control::ShowProgress(bool open, bool bigWin)
{
  int index, total;

  if(GetIndex(index,total) && total>=0) {
    index/=framesPerSecond; total/=framesPerSecond;
    if(!visible && open) {
      static int cols[] = { 5,0 };
      Interface->Open(Setup.OSDwidth, bigWin?0:-2);
      Interface->SetCols(cols);
      bigwin=bigWin;
      needsFastResponse=visible=true;
      Interface->Clear();
      fliptime=listtime=0; flipint=0; flip=-1; top=lastTop=-1; lastIndex=lastTotal=-1;
      delete lastMode; lastMode=0;
      }

    cMP3PlayInfo *mode=new cMP3PlayInfo;
    bool valid=GetPlayInfo(mode);

    if(valid) { // send progress to status monitor
      if(!lastMode || mode->Num!=lastMode->Num || strcmp(mode->Title,lastMode->Title) ||
                      mode->Loop!=lastMode->Loop || mode->Shuffle!=lastMode->Shuffle) {
        char *buf=0;
        asprintf(&buf,mode->Artist[0]?"[%c%c] (%d/%d) %s - %s":"[%c%c] (%d/%d) %s",
                  mode->Loop?'L':'.',mode->Shuffle?'S':'.',mode->Num,mode->MaxNum,mode->Title,mode->Artist);
        cStatus::MsgReplaying(this,buf);
        free(buf);
        }
      }

    if(visible) { // refresh the OSD progress display
      bool changed=(!lastMode || mode->Num!=lastMode->Num);
      bool flush=false;

      if(!selecting && changed) {
        char *buf=0;
        asprintf(&buf,"(%d/%d)",mode->Num,mode->MaxNum);
        Interface->Fill(0,-2,CTAB,1,clrBackground);
        Interface->Write(0,-2,buf);
        free(buf);
        flush=true;
        }

      if(!lastMode || mode->Loop!=lastMode->Loop) {
        if(mode->Loop) Interface->Write(-4,-1,"L",clrBlack,clrYellow);
        else Interface->Fill(-4,-1,2,1,clrBackground);
        flush=true;
        }
      if(!lastMode || mode->Shuffle!=lastMode->Shuffle) {
        if(mode->Shuffle) Interface->Write(-2,-1,"S",clrWhite,clrRed);
        else Interface->Fill(-2,-1,2,1,clrBackground);
        flush=true;
        }

      if(index!=lastIndex || total!=lastTotal) {
        if(total>0) {
          cProgressBar ProgressBar((Width()-CTAB-CTAB2)*cOsd::CellWidth(), cOsd::LineHeight(), index, total);
          Interface->SetBitmap(CTAB*cOsd::CellWidth(), (abs(Height())-1)*cOsd::LineHeight(), ProgressBar);
          }
        char *buf;
        asprintf(&buf,total?"%02d:%02d/%02d:%02d":"%02d:%02d",index/60,index%60,total/60,total%60);
        Interface->Write(0,-1,buf);
        free(buf);
        flush=true;
        }

      if(valid) {
        if(!jumpactive) {
          bool doflip=false;
          if(changed) {
            fliptime=time(0); flip=0;
	    doflip=true;
	    }
          else if(time(0)>fliptime+flipint) {
	    fliptime=time(0);
	    flip++; if(flip>=MP3Setup.DisplayMode) flip=0;
            doflip=true;
	    }
          if(doflip || changed) {
            char *buf=0;
            switch(flip) {
	      default:
	        flip=0;
	        // fall through
	      case 0:
	        asprintf(&buf,mode->Artist[0]?"%s - %s":"%s",mode->Title,mode->Artist);
	        flipint=6;
	        break;
	      case 1:
                if(mode->Album[0]) {
      	          asprintf(&buf,mode->Year>0?"from: %s (%d)":"from: %s",mode->Album,mode->Year);
	          flipint=4;
	          }
                else fliptime=0;
                break;
	      case 2:
                if(mode->MaxBitrate>0)
                  asprintf(&buf,"%.1f kHz, %d-%d kbps, %s",mode->SampleFreq/1000.0,mode->Bitrate/1000,mode->MaxBitrate/1000,mode->SMode);
                else
                  asprintf(&buf,"%.1f kHz, %d kbps, %s",mode->SampleFreq/1000.0,mode->Bitrate/1000,mode->SMode);
	        flipint=3;
	        break;
	      }
            if(buf) { DisplayInfo(buf); free(buf); flush=true; }
            }
          }

        if(bigwin && mode->PlayList) {
          bool all=(top!=lastTop || changed);
          if(all || time(0)>listtime+2) {
            int num=(top>0 && mode->Num==lastMode->Num) ? top : mode->Num - PL_ROWS/2;
            if(num+PL_ROWS>mode->MaxNum) num=mode->MaxNum-PL_ROWS+1;
            if(num<1) num=1;
            top=num;
            cSong *song=mode->PlayList->Get(num-1);
            for(int i=0 ; i<PL_ROWS && song ; i++) {
              char *buf=0;
              cSongInfo *si=song->Info(false);
              if(si && si->HasInfo())
                asprintf(&buf,si->Artist?"%d.\t%s - %s":"%d.\t%s",num,si->Title,si->Artist);
              else
                asprintf(&buf,"%d.\t%s",num,song->Name());
              eDvbColor fg=clrWhite, bg=clrBackground;
              int hash=MakeHash(buf);
              if(num==mode->Num) { fg=clrBlack; bg=clrCyan; hash=(hash^77) + 23; }
              if(all || hash!=hashlist[i]) {
                Interface->Fill(0,i,Setup.OSDwidth,1,clrBackground);
                Interface->WriteText(0,i,buf,fg,bg);
                flush=true;
                hashlist[i]=hash;
                }
              free(buf);
              num++; song=(cSong *)song->Next();
              }
            listtime=time(0); lastTop=top;
            }
          }
        }

      if(flush) Interface->Flush();
      }

    lastIndex=index; lastTotal=total;
    delete lastMode; lastMode=mode;
    }
}

void cMP3Control::DisplayInfo(const char *s)
{
  Interface->Fill(CTAB,-2,Width()-CTAB,1,clrBackground);
  if(s) Interface->Write(CTAB,-2,s);
}

void cMP3Control::JumpDisplay(void)
{
  char buf[64];
  const char *j=tr("Jump: "), u=jumpsecs?'s':'m';
  if(!jumpmm) sprintf(buf,"%s- %c",  j,u);
  else        sprintf(buf,"%s%d- %c",j,jumpmm,u);
  DisplayInfo(buf);
}

void cMP3Control::JumpProcess(eKeys Key)
{
 int n=Key-k0, d=jumpsecs?1:60;
  switch (Key) {
    case k0 ... k9:
      if(jumpmm*10+n <= lastTotal/d) jumpmm=jumpmm*10+n;
      JumpDisplay();
      break;
    case kBlue:
      jumpsecs=!jumpsecs;
      JumpDisplay();
      break;
    case kPlay:
    case kUp:
      jumpmm-=lastIndex/d;
      // fall through
    case kFastRew:
    case kFastFwd:
    case kLeft:
    case kRight:
      SkipSeconds(jumpmm*d * ((Key==kLeft || Key==kFastRew) ? -1:1));
      // fall through
    default:
      jumpactive=false;
      break;
    }

  if(!jumpactive && jumphide) Hide();
}

void cMP3Control::Jump(void)
{
  jumpmm=0; jumphide=jumpsecs=false;
  if(!visible) {
    ShowTimed(); if(!visible) return;
    jumphide=true;
    }
  JumpDisplay();
  jumpactive=true; fliptime=0; flip=-1;
}

eOSState cMP3Control::ProcessKey(eKeys Key)
{
  if(!Active()) return osEnd;

  if(visible && timeoutShow && time(0)>timeoutShow) { Hide(); timeoutShow=0; }
  ShowProgress();

  if(jumpactive && Key!=kNone) { JumpProcess(Key); return osContinue; }

  switch(Key) {
    case kUp:      Forward(); break;

    case kDown:    Backward(); break;

    case kLeft:    if(bigwin) {
                     if(top>0) { top-=PL_ROWS; if(top<1) top=1; }
                     break;
                     }
                   // fall through
    case kLeft|k_Repeat:
                   if(bigwin) break;
                   // fall through
    case kFastRew:
    case kFastRew|k_Repeat:
                   if(!IsStream()) SkipSeconds(-JUMPSIZE);
                   break;

    case kRight:   if(bigwin) {
                     if(top>0) top+=PL_ROWS;
                     break;
                     }
                   // fall through
    case kRight|k_Repeat:
                   if(bigwin) break;
                   // fall through
    case kFastFwd:
    case kFastFwd|k_Repeat:
                   if(!IsStream()) SkipSeconds(JUMPSIZE);
                   break;

    case kRed:     if(!IsStream()) Jump();
                   break;

    case kGreen:   if(lastMode) {
                     if(time(0)>greentime) {
                       if(lastMode->Loop || (!lastMode->Loop && !lastMode->Shuffle)) ToggleLoop();
                       if(lastMode->Shuffle) ToggleShuffle();
                       }
                     else {
                       if(!lastMode->Loop) ToggleLoop();
                       else if(!lastMode->Shuffle) ToggleShuffle();
                       else ToggleLoop();
                       }
                     greentime=time(0)+MULTI_TIMEOUT;
                     }
                   break;

    case kPause:
    case kYellow:  if(!IsStream()) Pause();
                   break;

    case kStop:
    case kBlue:    Hide();
                   Stop();
                   return osEnd;

    case k0 ... k9:number=number*10+Key-k0;
                   if(lastMode && number>0 && number<=lastMode->MaxNum) {
                     if(!visible) { ShowTimed(); selecthide=true; }
                     selecting=true; lastkeytime=time_ms();
                     char *buf=0;
                     asprintf(&buf,"(%d-/%d)",number,lastMode->MaxNum);
                     Interface->Fill(0,-2,CTAB,1,clrBackground);
                     Interface->Write(0,-2,buf);
                     Interface->Flush();
                     free(buf);
                     break;
                     }
                   number=0; lastkeytime=0;
                   // fall through
    case kNone:    if(selecting && time_ms()-lastkeytime>SELECT_TIMEOUT) {
                     if(number>0) Goto(number);
                     if(selecthide) timeoutShow=time(0)+SELECTHIDE_TIMEOUT;
                     if(lastMode) lastMode->Num=0;
                     number=0; selecting=selecthide=false;
                     }
                   break;

    case kOk:      if(time(0)>oktime) {
                     visible ? Hide() : ShowTimed();
                     }
                   else {
                     if(visible && !bigwin) { Hide(); ShowProgress(true,true); }
                     else { Hide(); ShowTimed(); }
                     }
                   oktime=time(0)+MULTI_TIMEOUT;
                   break;

    case kBack:    Hide(); return osEnd; //XXX osMP3;

    default:       return osUnknown;
    }
  return osContinue;
}

// --- cMenuID3Info ------------------------------------------------------------

class cMenuID3Info : public cOsdMenu {
private:
  cOsdItem *Item(const char *name, const char *text);
  cOsdItem *Item(const char *name, const char *format, const float num);
  void Build(cSongInfo *info, const char *name);
public:
  cMenuID3Info(cSong *song);
  cMenuID3Info(cSongInfo *si, const char *name);
  virtual eOSState ProcessKey(eKeys Key);
  };

cMenuID3Info::cMenuID3Info(cSong *song)
:cOsdMenu(tr("ID3 information"),12)
{
  Build(song->Info(),song->Name());
}

cMenuID3Info::cMenuID3Info(cSongInfo *si, const char *name)
:cOsdMenu(tr("ID3 information"),12)
{
  Build(si,name);
}

void cMenuID3Info::Build(cSongInfo *si, const char *name)
{
  if(si) {
    Item(tr("Filename"),name);
    if(si->HasInfo() && si->Total>0) {
      char *buf=0;
      asprintf(&buf,"%02d:%02d",si->Total/60,si->Total%60);
      Item(tr("Length"),buf);
      free(buf);
      Item(tr("Title"),si->Title);
      Item(tr("Artist"),si->Artist);
      Item(tr("Album"),si->Album);
      Item(tr("Year"),0,(float)si->Year);
      Item(tr("Samplerate"),"%.1f kHz",si->SampleFreq/1000.0);
      Item(tr("Bitrate"),"%.f kbit/s",si->Bitrate/1000.0);
      Item(tr("Channels"),0,(float)si->Channels);
      }
    Display();
    }
}

cOsdItem *cMenuID3Info::Item(const char *name, const char *format, const float num)
{
  cOsdItem *item;
  if(num>=0.0) {
    char *buf=0;
    asprintf(&buf,format?format:"%.f",num);
    item=Item(name,buf);
    free(buf);
    }
  else item=Item(name,"");
  return item;
}

cOsdItem *cMenuID3Info::Item(const char *name, const char *text)
{
  char *buf=0;
  asprintf(&buf,"%s:\t%s",name,text?text:"");
  cOsdItem *item = new cOsdItem(buf,osBack);
  item->SetColor(clrWhite, clrBackground);
  free(buf);
  Add(item); return item;
}

eOSState cMenuID3Info::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);

  if(state==osUnknown) {
     switch(Key) {
       case kRed:
       case kGreen:
       case kYellow:
       case kBlue:   return osContinue;
       case kMenu:   return osEnd;
       default: break;
       }
     }
  return state;
}

// --- cMenuInstantBrowse -------------------------------------------------------

class cMenuInstantBrowse : public cMenuBrowse {
private:
  const char *selecttext, *alltext;
  virtual void SetButtons(void);
  virtual eOSState ID3Info(void);
public:
  cMenuInstantBrowse(cFileSource *Source, const char *Selecttext, const char *Alltext);
  virtual eOSState ProcessKey(eKeys Key);
  };

cMenuInstantBrowse::cMenuInstantBrowse(cFileSource *Source, const char *Selecttext, const char *Alltext)
:cMenuBrowse(Source,true,true,tr("Directory browser"))
{
  selecttext=Selecttext; alltext=Alltext;
  SetButtons();
}

void cMenuInstantBrowse::SetButtons(void)
{
  SetHelp(selecttext, currentdir?tr("Parent"):0, currentdir?0:alltext, tr("ID3 info"));
  Display();
}

eOSState cMenuInstantBrowse::ID3Info(void)
{
  cDirItem *item=CurrentItem();
  if(item && item->Type==itFile) {
    char *name=item->Path();
    cSong *song=new cSong(name,source);
    cSongInfo *si;
    if(song && (si=song->Info())) {
      AddSubMenu(new cMenuID3Info(si,name));
      }
    delete song;
    free(name);
    }
  return osContinue;
}

eOSState cMenuInstantBrowse::ProcessKey(eKeys Key)
{
  eOSState state = cMenuBrowse::ProcessKey(Key);

  if (state == osUnknown) {
     switch (Key) {
       case kYellow: lastselect=new cDirItem(source,0,0,itBase);
                     return osBack;
       default: break;
       }
     }
  return state;
}

// --- cMenuPlayListItem -------------------------------------------------------

class cMenuPlayListItem : public cOsdItem {
  private:
  bool showID3;
  cSong *song;
public:
  cMenuPlayListItem(cSong *Song, bool showid3);
  cSong *Song(void) { return song; }
  virtual void Set(void);
  void Set(bool showid3);
  };

cMenuPlayListItem::cMenuPlayListItem(cSong *Song, bool showid3)
{
  song=Song;
  Set(showid3);
}

void cMenuPlayListItem::Set(bool showid3)
{
  showID3=showid3;
  Set();
}

void cMenuPlayListItem::Set(void)
{
  char *buffer=0;
  cSongInfo *si=song->Info(false);
  if(showID3 && !si) si=song->Info();
  if(showID3 && si && si->Title)
    asprintf(&buffer, "%d.\t%s%s%s",song->Index()+1,si->Title,si->Artist?" - ":"",si->Artist?si->Artist:"");
  else
    asprintf(&buffer, "%d.\t<%s>",song->Index()+1,song->Name());
  SetText(buffer,false);
}

// --- cMenuPlayList ------------------------------------------------------

class cMenuPlayList : public cOsdMenu {
private:
  cPlayList *playlist;
  bool browsing, showid3;
  void Buttons(void);
  void Refresh(bool all = false);
  void Add(void);
  void AddItem(const char *name);
  virtual void Move(int From, int To);
  eOSState Remove(void);
  eOSState ShowID3(void);
  eOSState ID3Info(void);
public:
  cMenuPlayList(cPlayList *Playlist);
  virtual eOSState ProcessKey(eKeys Key);
  };

cMenuPlayList::cMenuPlayList(cPlayList *Playlist)
:cOsdMenu(tr("Playlist editor"),4)
{
  browsing=showid3=false;
  playlist=Playlist;
  if(MP3Setup.EditorMode) showid3=true;

  cSong *mp3 = playlist->First();
  while(mp3) {
    cOsdMenu::Add(new cMenuPlayListItem(mp3,showid3));
    mp3 = playlist->cList<cSong>::Next(mp3);
    }
  Buttons(); Display();
}

void cMenuPlayList::Buttons(void)
{
  SetHelp(tr("Add"), showid3?tr("Filenames"):tr("ID3 names"), tr("Remove"), tr("Mark"));
}

void cMenuPlayList::Refresh(bool all)
{
  cMenuPlayListItem *cur=(cMenuPlayListItem *)((all || Count()<2) ? First() : Get(Current()));
  while(cur) {
    cur->Set(showid3);
    cur=(cMenuPlayListItem *)Next(cur);
    }
}

void cMenuPlayList::AddItem(const char *name)
{
  cSong *mp3 = new cSong(name,MP3Sources.GetSource());
  if(Count()>0) {
    cMenuPlayListItem *current=(cMenuPlayListItem *)Get(Current());
    playlist->Add(mp3,current->Song());
    cOsdMenu::Add(new cMenuPlayListItem(mp3,showid3),true,current);
    }
  else {
    playlist->Add(mp3);
    cOsdMenu::Add(new cMenuPlayListItem(mp3,showid3),true);
    }
}

void cMenuPlayList::Add(void)
{
  cDirItem *item=cMenuInstantBrowse::GetSelected();
  if(item) {
    Interface->Status(tr("Scanning directory..."));
    Interface->Flush();
    cInstantPlayList *newpl=new cInstantPlayList(item);
    if(newpl->Load()) {
      if(newpl->Count()) {
        if(newpl->Count()==1 || Interface->Confirm(tr("Add recursivly?"))) {
          cSong *mp3=newpl->First();
          while(mp3) {
            AddItem(mp3->Name());
            mp3=newpl->cList<cSong>::Next(mp3);
            }
          playlist->Save();
          Refresh(); Display();
          }
        }
      else Interface->Error(tr("Empty directory!"));
      }
    else Interface->Error(tr("Error scanning directory!"));
    delete newpl;
    }
}

void cMenuPlayList::Move(int From, int To)
{
  playlist->Move(From,To); playlist->Save();
  cOsdMenu::Move(From,To);
  Refresh(true); Display();
}

eOSState cMenuPlayList::ShowID3(void)
{
  showid3=!showid3;
  Buttons(); Refresh(true); Display();
  return osContinue;
}

eOSState cMenuPlayList::ID3Info(void)
{
  if(Count()>0) {
    cMenuPlayListItem *current = (cMenuPlayListItem *)Get(Current());
    AddSubMenu(new cMenuID3Info(current->Song()));
    }
  return osContinue;
}

eOSState cMenuPlayList::Remove(void)
{
  if(Count()>0) {
    cMenuPlayListItem *current = (cMenuPlayListItem *)Get(Current());
    if(Interface->Confirm(tr("Remove entry?"))) {
      playlist->Del(current->Song()); playlist->Save();
      cOsdMenu::Del(Current());
      Refresh(); Display();
      }
    }
  return osContinue;
}

eOSState cMenuPlayList::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);

  if(browsing && !HasSubMenu() && state==osContinue) { Add(); browsing=false; }

  if(state==osUnknown) {
     switch(Key) {
       case kOk:     return ID3Info();
       case kRed:    browsing=true;
                     return AddSubMenu(new cMenuInstantBrowse(MP3Sources.GetSource(),tr("Add"),tr("Add all")));
       case kGreen:  return ShowID3();
       case kYellow: return Remove();
       case kBlue:   Mark(); return osContinue;
       case kMenu:   return osEnd;
       default: break;
       }
     }
  return state;
}


// --- cPlaylistRename --------------------------------------------------------

class cPlaylistRename : public cOsdMenu {
private:
  static char *newname;
  const char *oldname;
  char data[64];
public:
  cPlaylistRename(const char *Oldname);
  virtual eOSState ProcessKey(eKeys Key);
  static const char *GetNewname(void) { return newname; }
  };

char *cPlaylistRename::newname = NULL;

cPlaylistRename::cPlaylistRename(const char *Oldname)
:cOsdMenu(tr("Rename playlist"), 15)
{
  free(newname); newname=0;

  oldname=Oldname;
  char *buf=NULL;
  asprintf(&buf,"%s\t%s",tr("Old name:"),oldname);
  cOsdItem *old = new cOsdItem(buf,osContinue);
  old->SetColor(clrWhite, clrBackground);
  Add(old);
  free(buf);

  data[0]=0;
  Add(new cMenuEditStrItem( tr("New name"), data, sizeof(data)-1, tr(FileNameChars)),true);
}

eOSState cPlaylistRename::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);

  if (state == osUnknown) {
     switch (Key) {
       case kOk:     if(data[0] && strcmp(data,oldname)) newname=strdup(data);
                     return osBack;
       case kRed:
       case kGreen:
       case kYellow:
       case kBlue:   return osContinue;
       default: break;
       }
     }
  return state;
}

// --- cMenuMP3Item -----------------------------------------------------

class cMenuMP3Item : public cOsdItem {
  private:
  cPlayList *playlist;
  virtual void Set(void);
public:
  cMenuMP3Item(cPlayList *PlayList);
  cPlayList *List(void) { return playlist; }
  };

cMenuMP3Item::cMenuMP3Item(cPlayList *PlayList)
{
  playlist=PlayList;
  Set();
}

void cMenuMP3Item::Set(void)
{
  char *buffer=0;
  asprintf(&buffer," %s",playlist->BaseName());
  SetText(buffer,false);
}

// --- cMenuMP3 --------------------------------------------------------

cMenuMP3::cMenuMP3(void)
:cOsdMenu(tr("MP3"))
{
  renaming=sourcing=instanting=false;
  lists=new cPlayLists;
  ScanLists(); SetButtons(1);
  if(MP3Setup.MenuMode) Instant(false);
}

cMenuMP3::~cMenuMP3(void)
{
  delete lists;
}

eOSState cMenuMP3::SetButtons(int num)
{
  switch(num) {
    case 1:
      SetHelp(tr("Edit"), tr("Source"), tr("Browse"), ">>");
      break;
    case 2:
      SetHelp("<<", tr("New"), tr("Delete"), tr("Rename"));
      break;
    }
  buttonnum=num; Display();
  return osContinue;
}

void cMenuMP3::ScanLists(void)
{
  Clear();
  Interface->Status(tr("Scanning playlists..."));
  Interface->Flush();
  if(lists->Load(MP3Sources.GetSource())) {
    cPlayList *plist=lists->First();
    while(plist) {
      Add(new cMenuMP3Item(plist));
      plist=lists->Next(plist);
      }
    }
  else Interface->Error(tr("Error scanning playlists!"));
}

eOSState cMenuMP3::Delete(void)
{
  if(Count()>0) {
    if(Interface->Confirm(tr("Delete playlist?")) &&
       Interface->Confirm(tr("Are you sure?")) ) {
      cPlayList *plist = ((cMenuMP3Item *)Get(Current()))->List();
      if(plist->Delete()) {
        lists->Del(plist);
        cOsdMenu::Del(Current());
        Display();
        }
      else Interface->Error(tr("Error deleting playlist!"));
      }
    }
  return osContinue;
}

eOSState cMenuMP3::New(void)
{
  cPlayList *plist = new cPlayList(0,MP3Sources.GetSource());
  char name[64];
  int i=0;

  do {
    if(i) sprintf(name,"%s%d",tr("unnamed"),i++);
    else { strcpy(name,tr("unnamed")); i++; }
    } while(plist->TestName(name));
  
  if(plist->Create(name)) {
    lists->Add(plist);
    Add(new cMenuMP3Item(plist), true);

    isyslog("MP3: playlist %s added", plist->Name());
    return AddSubMenu(new cMenuPlayList(plist));
    }
  Interface->Error(tr("Error creating playlist!"));
  delete plist;
  return osContinue;
}

eOSState cMenuMP3::Rename(bool second)
{
  if(HasSubMenu() || Count() == 0) return osContinue;

  cPlayList *plist = ((cMenuMP3Item *)Get(Current()))->List();
  if(!second) {
    renaming=true;
    return AddSubMenu(new cPlaylistRename(plist->BaseName()));
    }
  renaming=false;
  const char *newname=cPlaylistRename::GetNewname();
  if(newname) {
    if(plist->Rename(newname)) {
      RefreshCurrent();
      DisplayCurrent(true);
      }
    else Interface->Error(tr("Error renaming playlist!"));
    }
  return osContinue;
}

eOSState cMenuMP3::Edit(void)
{
  if(HasSubMenu() || Count() == 0) return osContinue;

  cPlayList *plist = ((cMenuMP3Item *)Get(Current()))->List();
  if(!plist->Load()) Interface->Error(tr("Error loading playlist!"));
  else if(!plist->IsWinAmp()) {
    isyslog("MP3: editing playlist %s", plist->Name());
    return AddSubMenu(new cMenuPlayList(plist));
    }
  else Interface->Error(tr("Can't edit a WinAmp playlist!"));
  return osContinue;
}

eOSState cMenuMP3::Play(void)
{
  if(HasSubMenu() || Count() == 0) return osContinue;

  Interface->Status(tr("Loading playlist..."));
  Interface->Flush();
  cPlayList *plist = ((cMenuMP3Item *)Get(Current()))->List();
  cPlayList *newpl = new cPlayList(plist->Name(),MP3Sources.GetSource());
  if(newpl->Load() && newpl->Count()) {
    isyslog("mp3: playback started with playlist %s", newpl->Name());
    cMP3Control::SetPlayList(newpl);
    return osEnd;
    }
  delete newpl;
  Interface->Error(tr("Error loading playlist!"));
  return osContinue;
}

eOSState cMenuMP3::Source(bool second)
{
  if(HasSubMenu()) return osContinue;

  if(!second) {
    sourcing=true;
    return AddSubMenu(new cMenuSource(&MP3Sources,tr("MP3 source")));
    }
  sourcing=false;
  cFileSource *src=cMenuSource::GetSelected();
  if(src) {
    MP3Sources.SetSource(src);
    ScanLists();
    Display();
    }
  return osContinue;
}

eOSState cMenuMP3::Instant(bool second)
{
  if(HasSubMenu()) return osContinue;

  if(!second) {
    instanting=true;
    return AddSubMenu(new cMenuInstantBrowse(MP3Sources.GetSource(),tr("Play"),tr("Play all")));
    }
  instanting=false;
  cDirItem *item=cMenuInstantBrowse::GetSelected();
  if(item) {
    Interface->Status(tr("Building playlist..."));
    Interface->Flush();
    cInstantPlayList *newpl = new cInstantPlayList(item);
    if(newpl->Load() && newpl->Count()) {
      isyslog("mp3: playback started with instant playlist %s", newpl->Name());
      cMP3Control::SetPlayList(newpl);
      return osEnd;
      }
    delete newpl;
    Interface->Error(tr("Error building playlist!"));
    }
  return osContinue;
}

eOSState cMenuMP3::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);

  if(!HasSubMenu() && state==osContinue) { // eval the return value from submenus
    if(renaming) return Rename(true);
    if(sourcing) return Source(true);
    if(instanting) return Instant(true);
    }

  if(state == osUnknown) {
    switch(Key) {
      case kOk:     return Play();
      case kRed:    return (buttonnum==1 ? Edit() : SetButtons(1)); 
      case kGreen:  return (buttonnum==1 ? Source(false) : New());
      case kYellow: return (buttonnum==1 ? Instant(false) : Delete());
      case kBlue:   return (buttonnum==1 ? SetButtons(2) : Rename(false));
      case kMenu:   return osEnd;
      default:      break;
      }
    }
  return state;
}
