/*  $Id: main-window.cpp,v 1.13 2009/03/10 03:03:52 sarrazip Exp $
    main-window.cpp - Input and conjugation window

    verbiste - French conjugation system
    Copyright (C) 2003-2009 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "main-window.h"

#include "gui/conjugation.h"
#include "util.h"

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>

#include <libintl.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <iostream>
#include <fstream>
#include <sstream>

#define _(x) gettext(x)

using namespace std;
using namespace verbiste;


/*****************************************************************************/

gboolean hideOnDelete = TRUE;


FrenchVerbDictionary *fvd = NULL;

static GtkWidget *resultWin = NULL;
static GtkWidget *verbEntry = NULL;
static GtkWidget *conjButton = NULL;
static GtkWidget *saveButton = NULL;
static GtkWidget *showPronounsCB = NULL;
static GtkWidget *resultNotebook = NULL;

static const gint SP = 2;  // default spacing for the GTK+ boxes


/*  External functions to be provided by the application.
    A GNOME app would have these functions call gnome_config_get_string(), etc.
*/
char *get_config_string(const char *path);
void set_config_string(const char *path, const char *value);
void sync_config();


/*****************************************************************************/


// Encapsulates a gchar pointer to memory that needs to be freed with g_free().
class Catena
{
public:
    Catena(gchar *p) : ptr(p) {}
    ~Catena() { g_free(ptr); }
    gchar *get() const { return ptr; }
    gint atoi() const { return ptr == NULL ? INT_MIN : static_cast<gint>(::atoi(ptr)); }
    bool startsWithDigit() const { return ptr != NULL && ::isdigit(ptr[0]); }
private:
    gchar *ptr;
};


/*****************************************************************************/


class ResultPage
{
public:

    GtkWidget *notebookPage;
    GtkWidget *table;

    ResultPage();

    // No destructor because this object does not own the two GtkWidgets.
};


ResultPage::ResultPage()
  : notebookPage(gtk_vbox_new(FALSE, SP)),
    table(gtk_table_new(4, 4, FALSE))
{
    GtkWidget *scrolledWin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(
			GTK_SCROLLED_WINDOW(scrolledWin),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_add_with_viewport(
			GTK_SCROLLED_WINDOW(scrolledWin), table);

    g_object_set_data(G_OBJECT(notebookPage), "ResultPage", this);
	// make the vbox widget point to the corresponding ResultPage object

    gtk_box_pack_start(GTK_BOX(notebookPage), scrolledWin, TRUE, TRUE, 0);
}


/*****************************************************************************/


// All code that wants to make the program quit must use this function
// instead of calling gtk_main_quit() directly.
//
static
void
quit()
{
    // Save the current size of the results window.
    gint resultWinWidth, resultWinHeight;
    gtk_window_get_size(GTK_WINDOW(resultWin), &resultWinWidth, &resultWinHeight);

    stringstream ss;
    ss << resultWinWidth;
    set_config_string("Preferences/ResultWindowWidth", ss.str().c_str());
    ss.str("");
    ss << resultWinHeight;
    set_config_string("Preferences/ResultWindowHeight", ss.str().c_str());

    // Save the state of the Show Pronouns checkbox.
    bool active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showPronounsCB));
    set_config_string("Preferences/ShowPronouns", active ? "1" : "0");

    sync_config();  // commit changes to file

    gtk_main_quit();
}


static
void
hideOrQuit(GtkWidget *, gpointer)
{
    if (hideOnDelete)
	gtk_widget_hide(resultWin);
    else
	quit();
}


static
gboolean
onKeyPressInResultWin(GtkWidget *, GdkEventKey *event, gpointer)
{
    g_return_val_if_fail(event != NULL, TRUE);

    if (event->keyval == GDK_w && (event->state & GDK_CONTROL_MASK) != 0)
    {
	hideOrQuit(NULL, NULL);
	return TRUE;
    }

    return FALSE;
}


static
gboolean
onKeyPressInAbout(GtkWidget *about, GdkEventKey *event, gpointer)
{
    g_return_val_if_fail(event != NULL, TRUE);

    switch (event->keyval)
    {
	case GDK_Escape:
	    gtk_dialog_response(GTK_DIALOG(about), GTK_RESPONSE_OK);
	    return TRUE;

	default:
	    return FALSE;
    }
}


void
showAbout()
{
    const gchar *authors[] =
    {
	"Pierre Sarrazin <http://sarrazip.com/>",
	NULL
    };

    string logoFilename = string(PIXMAPDIR) + "/" + PACKAGE ".png";

    GdkPixbuf *logo = gdk_pixbuf_new_from_file(logoFilename.c_str(), NULL);

    string copyright =
	string("Copyright (C) 2003-2009 Pierre Sarrazin http://sarrazip.com/\n")
	+ _("Distributed under the GNU General Public License");

    GtkWidget *about = gtk_about_dialog_new();
    gtk_about_dialog_set_name(GTK_ABOUT_DIALOG(about), PACKAGE_FULL_NAME);
    gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(about), VERSION);
    gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(about), copyright.c_str());
    gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(about),
					    _("A French conjugation system"));
    gtk_about_dialog_set_authors(GTK_ABOUT_DIALOG(about), authors);
    if (logo != NULL)
	gtk_about_dialog_set_logo(GTK_ABOUT_DIALOG(about), logo);

    {
	ifstream licenseFile((LIBDATADIR + string("/COPYING")).c_str());
	if (licenseFile.good())
	{
	    stringstream ss;
	    ss << licenseFile.rdbuf();
	    gtk_about_dialog_set_license(GTK_ABOUT_DIALOG(about),
						    ss.str().c_str());
	}
    }

    if (logo != NULL)
	gdk_pixbuf_unref(logo);

    g_signal_connect(G_OBJECT(about), "key-press-event",
			G_CALLBACK(onKeyPressInAbout), NULL);

    gtk_dialog_run(GTK_DIALOG(about));
    gtk_widget_hide(about);
    gtk_widget_destroy(about);
}


static
gboolean
onKeyPressInEntry(GtkWidget *entry, GdkEventKey *event, gpointer /*data*/)
{
    g_return_val_if_fail(event != NULL, TRUE);

    switch (event->keyval)
    {
	case GDK_Return:
	case GDK_KP_Enter:
	    {
		const gchar *text = gtk_entry_get_text(GTK_ENTRY(entry));
		processText(text);
	    }
	    return TRUE;

	default:
	    return FALSE;
    }
}


static
void
onChangeInEntry(GtkEditable *, gpointer)
{
    const gchar *text = gtk_entry_get_text(GTK_ENTRY(verbEntry));
    gtk_widget_set_sensitive(GTK_WIDGET(conjButton), text[0] != '\0');
}


static
void
onConjugateButton(GtkWidget *, gpointer)
{
    const gchar *text = gtk_entry_get_text(GTK_ENTRY(verbEntry));
    processText(text);
}


// Escape for HTML, including converting newline into <br/>.
static
string
esc(const string &s)
{
    string out;
    string::size_type len = s.length();
    for (string::size_type i = 0; i < len; i++)
    {
	switch (s[i])
	{
	    case '\n': out += "<br/>"; break;
	    case '<': out += "&lt;"; break;
	    case '>': out += "&gt;"; break;
	    case '&': out += "&amp;"; break;
	    default:  out += s[i];
	}
    }
    return out;
}


static
void
onSaveButton(GtkWidget *, gpointer)
{
    // Show a Save dialog that points to the last save dir, if available.
    GtkWidget *chooser = gtk_file_chooser_dialog_new(
				_("Save Conjugation to HTML File"),
				GTK_WINDOW(resultWin),
				GTK_FILE_CHOOSER_ACTION_SAVE,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				NULL);
    gtk_file_chooser_set_do_overwrite_confirmation(
				GTK_FILE_CHOOSER(chooser),
				true);

    Catena saveDir = get_config_string("Files/SaveDirectory");
    if (saveDir.get() != NULL)
	gtk_file_chooser_set_current_folder(
				    GTK_FILE_CHOOSER(chooser),
				    saveDir.get());

    gtk_file_chooser_set_current_name(
				GTK_FILE_CHOOSER(chooser),
				_("Untitled conjugation.html"));

    bool cancel = (gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_CANCEL);
    string filename;
    if (!cancel)
    {
	Catena f = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
	filename = f.get();

	// Remember the chosen directory:
	Catena dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(chooser));
	set_config_string("Files/SaveDirectory", dir.get());

	// We do not call sync_config() because it will be done by quit()
	// when the program quits.
    }
    gtk_widget_destroy(chooser);
    if (cancel)
	return;

    // Try to create the file:

    ofstream html(filename.c_str(), ios::out);
    if (!html)
    {
	showErrorDialog(_("Could not create file:") + ("\n" + filename));
	return;
    }

    html << "<?xml version='1.0'?>\n";
    html << "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\""
	    << " \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n";
    html << "<html>\n";
    html << "<head>\n";
    html << "<title>" << esc(gtk_entry_get_text(GTK_ENTRY(verbEntry)))
	    << "</title>\n";
    html << "<meta http-equiv='Content-Type'"
	    << " content='text/html; charset=UTF-8' />\n";
    html << "</head>\n";
    html << "<body>\n";


    assert(resultNotebook != NULL);
    GtkWidget *w;
    for (int i = 0; (w = gtk_notebook_get_nth_page(
			GTK_NOTEBOOK(resultNotebook), i)) != NULL;
		    i++)
    {
	ResultPage *rp = (ResultPage *) g_object_get_data(
						G_OBJECT(w), "ResultPage");
	g_return_if_fail(rp != NULL);
	g_return_if_fail(GTK_IS_TABLE(rp->table));

	GtkWidget *label = gtk_notebook_get_tab_label(
					    GTK_NOTEBOOK(resultNotebook),
					    rp->notebookPage);
	g_return_if_fail(GTK_IS_LABEL(label));

	const gchar *infinitive = gtk_label_get_text(GTK_LABEL(label));
	html << "<h1>" << esc(infinitive) << "</h1>\n";
	html << "<table border='1' cellspacing='0' cellpadding='4'>\n";
	string tr = "<tr valign='top'>\n";
	html << tr;

	GList *children = gtk_container_get_children(GTK_CONTAINER(rp->table));
	children = g_list_reverse(children);
	int childCounter = 0;
	for (GList *p = children; p != NULL; p = p->next, childCounter++)
	{
	    GtkWidget *vbox = (GtkWidget *) p->data;
	    g_return_if_fail(vbox != NULL);
	    g_return_if_fail(GTK_IS_VBOX(vbox));

	    GList *vboxChildren = gtk_container_get_children(
						GTK_CONTAINER(vbox));
	    g_return_if_fail(vboxChildren != NULL);
	    g_return_if_fail(vboxChildren->next != NULL);
	    g_return_if_fail(vboxChildren->next->next == NULL);

	    GtkWidget *nameLabel = (GtkWidget *) vboxChildren->data;
	    GtkWidget *personsLabel = (GtkWidget *) vboxChildren->next->data;

	    g_return_if_fail(nameLabel != NULL);
	    g_return_if_fail(GTK_IS_LABEL(nameLabel));
	    g_return_if_fail(personsLabel != NULL);
	    g_return_if_fail(GTK_IS_LABEL(personsLabel));

	    const gchar *name = gtk_label_get_text(GTK_LABEL(nameLabel));
	    const gchar *persons = gtk_label_get_text(GTK_LABEL(personsLabel));

	    g_return_if_fail(name != NULL);
	    g_return_if_fail(persons != NULL);

	    html << "<td>\n";
	    html << "<strong>" << esc(name) << "</strong>\n<br/>\n";
	    html << esc(persons) << "\n";
	    html << "</td>\n";

	    if (childCounter == 0)
	    {
		html << "<td>&nbsp;</td>\n";
		html << "<td>&nbsp;</td>\n";
		html << "<td>&nbsp;</td>\n";
		html << "</tr>\n";
		html << tr;
	    }
	    else if (childCounter == 4)
	    {
		html << "</tr>\n";
		html << tr;
	    }
	    else if (childCounter == 7 || childCounter == 10)
	    {
		html << "<td>&nbsp;</td>\n";
		html << "</tr>\n";
		if (childCounter == 7)
		    html << tr;
	    }

	    g_list_free(vboxChildren);
	}
	g_list_free(children);

	html << "</table>\n";
    }

    html << "</body>\n";
    html << "</html>\n";

    html.close();
    if (!html)
    {
	showErrorDialog(_("Could not close file:") + ("\n" + filename));
	return;
    }
}


static
GtkWidget *
newLabel(const string &markup, gboolean selectable)
{
    GtkWidget *label = gtk_label_new("");
    gtk_label_set_markup(GTK_LABEL(label), markup.c_str());
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_label_set_selectable(GTK_LABEL(label), selectable);
    return label;
}


static
ResultPage *
appendResultPage(const string &utf8LabelText)
{
    ResultPage *rp = new ResultPage();
    GtkWidget *label = newLabel("<b>" + utf8LabelText + "</b>" , FALSE);

    assert(resultNotebook != NULL);
    gtk_notebook_append_page(GTK_NOTEBOOK(resultNotebook),
				GTK_WIDGET(rp->notebookPage),
				GTK_WIDGET(label));
    return rp;
}


static
void
onAboutButton(GtkWidget *, gpointer)
{
    showAbout();
}


static
void
onShowPronounsToggled(GtkToggleButton *, gpointer)
{
    onConjugateButton(NULL, NULL);
}


static
bool
getResultWindowSize(gint &width, gint &height)
{
    Catena w = get_config_string("Preferences/ResultWindowWidth");
    Catena h = get_config_string("Preferences/ResultWindowHeight");
    if (!w.startsWithDigit() || !h.startsWithDigit())
	return false;

    enum { MIN = 0, MAX = 65535 };
    width = w.atoi();
    if (width < MIN || width > MAX)
	return false;
    height = h.atoi();
    if (height < MIN || height > MAX)
	return false;

    return true;
}


static
void
showResultWin()
{
    if (resultWin == NULL)
    {
	resultWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(resultWin), PACKAGE_FULL_NAME);

	// Get the size of the results window from the saved configuration, if available.
	gint resultWinWidth, resultWinHeight;
	if (!getResultWindowSize(resultWinWidth, resultWinHeight))  // if failed
	    resultWinWidth = 610, resultWinHeight = 530;  // use default values
	gtk_window_set_default_size(GTK_WINDOW(resultWin), resultWinWidth, resultWinHeight);

	gtk_container_set_border_width(GTK_CONTAINER(resultWin), 4);

	if (hideOnDelete)
	{
	    /*
		When user clicks on title bar's close button, the window must
		only be hidden, not be destroyed.
	    */
	    g_signal_connect(G_OBJECT(resultWin), "delete_event",
			G_CALLBACK(gtk_widget_hide_on_delete), NULL);
	}
	else
	    g_signal_connect(G_OBJECT(resultWin), "delete_event",
			G_CALLBACK(quit), NULL);

	/*
	    Capture the key presses in order to hide or close the window
	    on Ctrl-W.
	*/
	g_signal_connect(G_OBJECT(resultWin), "key-press-event",
			G_CALLBACK(onKeyPressInResultWin), NULL);


	/*
	    Create a text field where the user can enter requests:
	*/
	verbEntry = gtk_entry_new_with_max_length(255);
	g_signal_connect(G_OBJECT(verbEntry), "key-press-event",
			G_CALLBACK(onKeyPressInEntry), NULL);
	g_signal_connect(G_OBJECT(verbEntry), "changed",
			G_CALLBACK(onChangeInEntry), NULL);
	GtkWidget *prompt = gtk_label_new_with_mnemonic(_("_Verb:"));
	gtk_label_set_mnemonic_widget(GTK_LABEL(prompt), verbEntry);

	GtkWidget *promptBox = gtk_hbox_new(FALSE, SP);
	gtk_box_pack_start(GTK_BOX(promptBox), prompt, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(promptBox), verbEntry, TRUE, TRUE, 0);

	conjButton = gtk_button_new_with_mnemonic(_("_Conjugate"));
	gtk_widget_set_sensitive(GTK_WIDGET(conjButton), false);
	gtk_box_pack_start(GTK_BOX(promptBox), conjButton, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(conjButton), "clicked",
				    G_CALLBACK(onConjugateButton), NULL);

	saveButton = gtk_button_new_from_stock(GTK_STOCK_SAVE_AS);
	gtk_widget_set_sensitive(GTK_WIDGET(saveButton), false);
	gtk_box_pack_start(GTK_BOX(promptBox), saveButton, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(saveButton), "clicked",
				    G_CALLBACK(onSaveButton), NULL);


	/*
	    Create an options box.
	*/
	GtkWidget *optionsBox = gtk_hbox_new(FALSE, SP);
	showPronounsCB = gtk_check_button_new_with_mnemonic(
							_("Show _Pronouns"));
	gtk_box_pack_start(GTK_BOX(optionsBox), showPronounsCB,
							FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(showPronounsCB), "toggled",
				    G_CALLBACK(onShowPronounsToggled), NULL);
	Catena showPronouns = get_config_string("Preferences/ShowPronouns");
	if (showPronouns.get() != NULL)
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showPronounsCB),
					strcmp(showPronouns.get(), "1") == 0);

	/*
	    Create a notebook that receives the conjugations.
	*/
	resultNotebook = gtk_notebook_new();
	assert(resultNotebook != NULL);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(resultNotebook), TRUE);



	/*
	    Create a button box at the bottom.
	*/
	GtkWidget *bottomBox = gtk_hbox_new(FALSE, SP);

	GtkWidget *closeButton = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_box_pack_end(GTK_BOX(bottomBox), closeButton, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(closeButton), "clicked",
				    G_CALLBACK(hideOrQuit), NULL);

	GtkWidget *aboutButton = gtk_button_new_from_stock(GTK_STOCK_ABOUT);
	g_signal_connect(G_OBJECT(aboutButton), "clicked",
				    G_CALLBACK(onAboutButton), NULL);
	gtk_box_pack_end(GTK_BOX(bottomBox), aboutButton, FALSE, FALSE, 0);



	/*
	    Finish the window setup:
	*/
	GtkWidget *vbox = gtk_vbox_new(FALSE, SP);
	gtk_box_pack_start(GTK_BOX(vbox), promptBox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), optionsBox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), resultNotebook, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), bottomBox, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(resultWin), vbox);
	set_window_icon_to_default(resultWin);
	gtk_widget_show_all(GTK_WIDGET(resultWin));
    }

    gtk_widget_grab_focus(GTK_WIDGET(verbEntry));

    gtk_window_present(GTK_WINDOW(resultWin));
}


static
void
clearResultNotebook()
{
    if (resultNotebook != NULL)
    {
	GtkWidget *w;
	while ((w = gtk_notebook_get_nth_page(
			    GTK_NOTEBOOK(resultNotebook), 0)) != NULL)
	{
	    ResultPage *rp = (ResultPage *) g_object_get_data(
						    G_OBJECT(w), "ResultPage");
	    if (rp == NULL)
		g_warning("clearResultNotebook: null ResultPage pointer");
	    gtk_notebook_remove_page(GTK_NOTEBOOK(resultNotebook), 0);
	    delete rp;
	}
    }

    gtk_widget_set_sensitive(GTK_WIDGET(saveButton), false);
}


static
string
tolowerUTF8(const string &s)
{
    Catena down = g_utf8_strdown(s.data(), s.length());
    return down.get();
}


static
GtkWidget *
createTableCell(const VVS &latin1Tense,
		const string &utf8TenseName,
		const string &utf8UserText,
		FrenchVerbDictionary *fvd)
{
    GtkWidget *vbox = gtk_vbox_new(FALSE, SP);
    GtkWidget *nameLabel = newLabel(
			    "<b><u>" + utf8TenseName + "</u></b>", TRUE);

    string latin1Persons = createTableCellText(
				latin1Tense,
				fvd->utf8ToLatin1(tolowerUTF8(utf8UserText)),
				"<span foreground=\"red\">",
				"</span>");

    string utf8Persons = fvd->latin1ToUTF8(latin1Persons);
    GtkWidget *personsLabel = newLabel(utf8Persons, TRUE);

    gtk_box_pack_start(GTK_BOX(vbox), nameLabel, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), personsLabel, FALSE, FALSE, 0);

    return vbox;
}


void
processText(const string &utf8UserText)
/*
    'utf8UserText' must be a UTF-8 string to be deconjugated.
    It must not contain a newline character('\n').
*/
{
    g_return_if_fail(fvd != NULL);

    showResultWin();

    gtk_entry_set_text(GTK_ENTRY(verbEntry), utf8UserText.c_str());
    gtk_editable_select_region(GTK_EDITABLE(verbEntry), 0, -1);

    clearResultNotebook();

    string lowerCaseUserText = fvd->utf8ToLatin1(tolowerUTF8(utf8UserText));

    bool includePronouns = gtk_toggle_button_get_active(
					    GTK_TOGGLE_BUTTON(showPronounsCB));

    /*
	For each possible deconjugation, take the infinitive form and
	obtain its complete conjugation.
    */
    vector<InflectionDesc> v;
    fvd->deconjugate(lowerCaseUserText, v);

    string prevLatin1Infinitive;
    size_t numPages = 0;

    for (vector<InflectionDesc>::const_iterator it = v.begin();
					    it != v.end(); it++)
    {
	const InflectionDesc &d = *it;

	VVVS latin1Conjug;
	getConjugation(*fvd, d.infinitive, latin1Conjug, includePronouns);

	if (latin1Conjug.size() == 0           // if no tenses
	    || latin1Conjug[0].size() == 0     // if no infinitive tense
	    || latin1Conjug[0][0].size() == 0  // if no person in inf. tense
	    || latin1Conjug[0][0][0].empty())  // if infinitive string empty
	{
	    continue;
	}

	string latin1Infinitive = latin1Conjug[0][0][0];

	if (latin1Infinitive == prevLatin1Infinitive)
	    continue;

	ResultPage *rp = appendResultPage(fvd->latin1ToUTF8(latin1Infinitive));
	numPages++;

	int i = 0;
	for (VVVS::const_iterator t = latin1Conjug.begin();
				t != latin1Conjug.end(); t++, i++)
	{
	    const VVS &latin1Tense = *t;

	    if (i == 1)
		i = 4;
	    else if (i == 11)
		i = 12;
	    assert(i >= 0 && i < 16);

	    int row = i / 4;
	    int col = i % 4;

	    string utf8TenseName = getTenseNameForTableCell(row, col);
	    assert(!utf8TenseName.empty());

	    GtkWidget *cell = createTableCell(
				latin1Tense, utf8TenseName, utf8UserText, fvd);
	    gtk_table_attach(GTK_TABLE(rp->table), cell,
				col, col + 1, row, row + 1,
				GTK_FILL, GTK_FILL,
				8, 8);
	}

	gtk_widget_show_all(GTK_WIDGET(rp->notebookPage));
		/* must be done here to show the elements added in the for() */

	prevLatin1Infinitive = latin1Infinitive;
    }

    if (numPages == 0 && !utf8UserText.empty())
    {
	ResultPage *rp = appendResultPage(
					"<i>" + string(_("error")) + "</i>");
	GtkWidget *cell = newLabel(_("Unknown verb."), FALSE);
	gtk_table_attach(GTK_TABLE(rp->table), cell,
				0, 1, 0, 1,
				GTK_FILL, GTK_FILL,
				8, 8);
	gtk_widget_show_all(GTK_WIDGET(rp->notebookPage));
    }


    gtk_widget_set_sensitive(GTK_WIDGET(saveButton), numPages > 0);


    if (resultNotebook != NULL)
    {
	/*  PATCH: without this hack, a 2-page notebook has its first tab
	    active but showing the contents of the second page.  This can
	    be tested by entering "parais", which corresponds to the two
	    infinitives "paratre" and "parer".
	*/
	if (gtk_notebook_get_n_pages(GTK_NOTEBOOK(resultNotebook)) > 1)
	    gtk_notebook_set_current_page(GTK_NOTEBOOK(resultNotebook), 1);



	gtk_notebook_set_current_page(GTK_NOTEBOOK(resultNotebook), 0);
    }
}


void
showErrorDialog(const string &msg)
{
    GtkWidget *dlg = gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_CLOSE,
					"%s", msg.c_str());
    gtk_dialog_run(GTK_DIALOG(dlg));
    gtk_widget_destroy(dlg);
}


#if 0
void
initVerbDict() throw(logic_error)
{
    /*
	Create the French verb dictionary, which can conjugate and
	deconjugate French verbs.
    */
    const char *libdatadir = getenv("LIBDATADIR");
    if (libdatadir == NULL)
	libdatadir = LIBDATADIR;
    string conjFN  = libdatadir + string("/") + "conjugation-fr.xml";
    string verbsFN = libdatadir + string("/") + "verbs-fr.xml";
    fvd = new FrenchVerbDictionary(conjFN, verbsFN);  // may throw
}
#endif
