/*  $Id: Command.cpp,v 1.14 2009/10/31 02:02:05 sarrazip Exp $
    Command.cpp - Abstract command-line driver

    verbiste - French conjugation system
    Copyright (C) 2003-2009 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "Command.h"

#include <iostream>

using namespace std;
using namespace verbiste;


namespace verbiste {


Command::Command(bool _utf8Mode) throw (logic_error)
  : fvd(new FrenchVerbDictionary(false)),
		// command-line tools do not tolerate missing accents
    utf8Mode(_utf8Mode)
{
}


Command::~Command()
{
    delete fvd;
}


int
Command::run(int argc, char *argv[]) throw()
{
    int optind = 0;
    bool useArgs = (optind < argc);

    try
    {
	if (fvd == NULL)
	    throw logic_error("fvd is NULL");

	for (;;)
	{
	    string inputWord;

	    if (useArgs)
	    {
		if (optind == argc)
		    break;

		inputWord = argv[optind];
		optind++;
	    }
	    else
	    {
		if (!getline(cin, inputWord))
		    break;
	    }


	    string latin1InputWord;
	    if (utf8Mode)
	    {
		try
		{
		    latin1InputWord = fvd->utf8ToLatin1(inputWord);
		}
		catch (int)
		{
		    // Conversion failed. Leave latin1InputWord empty.
		}
	    }
	    else
	    {
		// Source is assumed to be in ISO-8859-1:
		latin1InputWord = inputWord;
	    }

	    processInputWord(fvd->tolowerLatin1(latin1InputWord));
	}
    }
    catch (const exception &e)
    {
	cerr << "Exception: " << e.what() << endl;
	return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}


const FrenchVerbDictionary &
Command::getFrenchVerbDictionary() const throw (logic_error)
{
    if (fvd == NULL)
	throw logic_error("fvd is NULL");
    return *fvd;
}


FrenchVerbDictionary &
Command::getFrenchVerbDictionary() throw (logic_error)
{
    if (fvd == NULL)
	throw logic_error("fvd is NULL");
    return *fvd;
}


int Command::listAllInfinitives(std::ostream &out) const
						throw (std::logic_error)
{
    (void) getFrenchVerbDictionary();  // to check that fvd is not null

    for (VerbTable::const_iterator it = fvd->beginKnownVerbs();
				  it != fvd->endKnownVerbs(); ++it)
    {
	string inf = it->first;
	if (utf8Mode)
	    inf = fvd->latin1ToUTF8(inf);
	out << inf << '\n';	
    }
    return EXIT_SUCCESS;
}


}  // namespace verbiste
