/* @(#) Two images as input: must match in size and type. Build an output
 * @(#) image choosing pixels from the left if a conditional image is
 * @(#) non-zero and from the right otherwise.
 * @(#)
 * @(#) The conditional image can have n bands or 1 band. If n bands, then we
 * @(#) choose from the two source images an element at a time. If 1 band,
 * @(#) then choose from the source images a pixel at a time.
 * @(#)
 * @(#)		int
 * @(#)		im_ifthenelse( c, a, b, out )
 * @(#)		IMAGE *c, *a, *b;
 * @(#)		IMAGE *out;
 * @(#)
 * @(#) Returns either 0 (success) or -1 (fail).
 *
 * Modified:
 * 9/2/95 JC
 *	- partialed and ANSIfied
 * 11/9/95 JC
 *	- return(0) missing! oops
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#include <stdio.h>

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

static void
process_buffer( PEL **p, PEL *q, int width, int es, int bands )
{
	int i, x, z;
	int n = width * bands * es;

	PEL *c = p[0];
	PEL *a = p[1];
	PEL *b = p[2];

	for( i = 0, x = 0; x < n; i++, x += es ) {
		if( c[i] )
			/* Copy pixel from 'then' image.
			 */
			for( z = x; z < x + es; z++ )
				q[z] = a[z];
		else
			/* Copy pixel from 'else' image.
			 */
			for( z = x; z < x + es; z++ )
				q[z] = b[z];
	}
}

/* if-then-else. Use IM_BANDFMT_UCHAR image to choose between two other images.
 * Either: all same number of bands, or choice image is one band, others are
 * n band.
 */
int
im_ifthenelse( IMAGE *c, IMAGE *a, IMAGE *b, IMAGE *out )
{
	IMAGE *invec[4];

	/* Check args.
	 */
	if( a->Coding != IM_CODING_NONE && a->Coding != IM_CODING_LABQ ) {
		im_errormsg( "im_ifthenelse: then image must be uncoded or "
			"labpack" );
		return( -1 );
	}
	if( b->Coding != IM_CODING_NONE && b->Coding != IM_CODING_LABQ ) {
		im_errormsg( "im_ifthenelse: else image must be uncoded or "
			"labpack" );
		return( -1 );
	}
	if( c->Coding != IM_CODING_NONE ) {
		im_errormsg( "im_ifthenelse: condition image must be uncoded" );
		return( -1 );
	}
	if( a->BandFmt != b->BandFmt ||
		a->Bands != b->Bands ) {
		im_errormsg( "im_ifthenelse: size and type of then and "
				"else must match" );
		return( -1 );
	}
	if( c->BandFmt != IM_BANDFMT_UCHAR ) {
		im_errormsg( "im_ifthenelse: conditional image must be uchar" );
		return( -1 );
	}

	/* Make output image.
	 */
	if( im_cp_desc( out, a ) ) 
		return( -1 );

	/* Process!
	 */
	invec[0] = c; invec[1] = a; invec[2] = b; invec[3] = NULL;
	if( c->Bands == 1 ) {
		/* Process, copying IM_IMAGE_SIZEOF_PEL() sized-units with a one-band
		 * conditional.
		 */
		if( im_wrapmany( invec, out,
			(im_wrapmany_fn) process_buffer, 
				(void *) IM_IMAGE_SIZEOF_PEL( a ), (void *) 1 ) )
			return( -1 );
	}
	else if( c->Bands == a->Bands ) {
		/* Process, copying IM_IMAGE_SIZEOF_ELEMENT() sized-units with an n-band
		 * conditional.
		 */
		if( im_wrapmany( invec, out,
			(im_wrapmany_fn) process_buffer, 
				(void *) IM_IMAGE_SIZEOF_ELEMENT( a ), (void *) a->Bands ) )
			return( -1 );
	}
	else {
		im_errormsg( "im_ifthenelse: conditional image must be " 
			"one band or same as then and else images" );
		return( -1 );
	}

	return( 0 );
}
