/* @(#) Find the absolute value of an image. Copy for UNSIGNED types, negate
 * @(#) for int types, fabs() for float types, and calculate modulus for
 * @(#) complex types.
 * @(#)
 * @(#) int 
 * @(#) im_abs( in, out )
 * @(#) IMAGE *in, *out;
 * @(#)
 * @(#) Returns 0 on success and -1 on error
 * @(#)
 *
 * Copyright: 1990, N. Dessipris, based on im_powtra()
 * Author: Nicos Dessipris
 * Written on: 02/05/1990
 * Modified on: 
 * 5/5/93 J.Cupitt
 *	- adapted from im_lintra to work with partial images
 *	- complex and signed support added
 * 30/6/93 JC
 *	- adapted for partial v2
 *	- ANSI conversion
 *	- spe29873r6k3h()**!@lling errors removed
 * 9/2/95 JC
 *	- adapted for im_wrap...
 * 20/6/02 JC
 *	- tiny speed up
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <vips/vips.h>
#include <vips/util.h>
#include <vips/region.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Integer abs operation: just test and negate.
 */
#define intabs(TYPE) \
	{ \
		TYPE *p = (TYPE *) in; \
		TYPE *q = (TYPE *) out; \
		\
		for( x = 0; x < sz; x++ ) { \
			TYPE v = p[x]; \
			\
			if( v < 0 ) \
				q[x] = 0 - v; \
			else \
				q[x] = v; \
		} \
	}

/* Float abs operation: call fabs().
 */
#define floatabs(TYPE)\
	{\
		TYPE *p = (TYPE *) in;\
		TYPE *q = (TYPE *) out;\
		\
		for( x = 0; x < sz; x++ )\
			q[x] = fabs( p[x] );\
	}

/* Complex abs operation: calculate modulus.
 */
#define complexabs(TYPE)\
	{\
		TYPE *p = (TYPE *) in;\
		TYPE *q = (TYPE *) out;\
		\
		for( x = 0; x < sz; x++ ) {\
			double rp = p[0];\
			double ip = p[1];\
			\
			p += 2;\
			q[x] = sqrt( rp * rp + ip * ip );\
		}\
	}

/* Abs a buffer of PELs.
 */
static void
abs_gen( PEL *in, PEL *out, int width, IMAGE *im )
{	
	int x;
	int sz = width * im->Bands;

	/* Abs all input types.
         */
        switch( im->BandFmt ) {
        case IM_BANDFMT_CHAR: 		intabs(signed char); break; 
        case IM_BANDFMT_SHORT: 		intabs(signed short); break; 
        case IM_BANDFMT_INT: 		intabs(signed int);  break; 
        case IM_BANDFMT_FLOAT: 		floatabs(float); break; 
        case IM_BANDFMT_DOUBLE:		floatabs(double); break; 
        case IM_BANDFMT_COMPLEX:	complexabs(float); break;
        case IM_BANDFMT_DPCOMPLEX:	complexabs(double); break;

        default:
		error_exit( "im_abs: internal error" );
        }
}

/* Abs of image.
 */
int 
im_abs( IMAGE *in, IMAGE *out )
{	
	/* Check args.
	 */
	if( in->Coding != IM_CODING_NONE ) {
		im_errormsg( "im_abs: in must be uncoded" );
		return( -1 );
	}

	/* Is this one of the unsigned types? Degenerate to im_copy() if it
	 * is.
	 */
	if( im_isuint( in ) )
		return( im_copy( in, out ) );

	/* Prepare output header. Output type == input type, except for
	 * complex.
	 */
	if( im_cp_desc( out, in ) )
		return( -1 );
	switch( in->BandFmt ) {
                case IM_BANDFMT_CHAR:
                case IM_BANDFMT_SHORT:
                case IM_BANDFMT_INT:
		case IM_BANDFMT_FLOAT:
		case IM_BANDFMT_DOUBLE:
			/* No action.
			 */
			break;

		case IM_BANDFMT_COMPLEX:
			out->Bbits = IM_BBITS_FLOAT;
			out->BandFmt = IM_BANDFMT_FLOAT;
			break;

		case IM_BANDFMT_DPCOMPLEX:
			out->Bbits = IM_BBITS_DOUBLE;
			out->BandFmt = IM_BANDFMT_DOUBLE;
			break;

		default:
			im_errormsg( "im_abs: unknown input type" );
                        return( -1 );
	}

	/* Generate!
	 */
	if( im_wrapone( in, out, 
		(im_wrapone_fn) abs_gen, in, NULL ) )
		return( -1 );

	return( 0 );
}
