/* @(#) Function to extract a portion of a Vasari format picture.
 * @(#) Function im_extract() assumes that the imin image
 * @(#) is either memory mapped or in the buffer pimin->data.
 * @(#)
 * @(#) int im_extract(pimin, pimout, pbox)
 * @(#) IMAGE *pimin, *pimout;
 * @(#) IMAGE_BOX *pbox;
 * @(#)
 * @(#) All functions return 0 on success and -1 on error
 * @(#)
 *
 * Copyright: 1990, J. Cupitt
 *
 * Author: J. Cupitt
 * Written on: 12/02/1990
 * Modified on: 4/6/92, J.Cupitt
 *	- speed up! why wasn't this done before? Why am I stupid?
 *	- layout, messages fixed
 * now extracts IM_CODING_LABQ to IM_CODING_LABQ file: K.Martinez 1/7/93
 * 2/7/93 JC
 *	- adapted for partial v2
 *	- ANSIfied
 * 7/7/93 JC
 *	- behaviour for IM_CODING_LABQ fixed
 *	- better messages
 * 7/10/94 JC
 *	- new IM_NEW()
 * 22/2/95 JC
 *	- new use of im_region_region()
 * 6/7/98 JC
 *	- im_extract_area() and im_extract_band() added
 * 11/7/01 JC
 *	- im_extract_band() now numbers from zero
 * 7/11/01 JC
 *	- oh what pain, im_extract now numbers from zero as well
 * 6/9/02 JC
 *	- zero xoff/yoff for extracted area
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Extract a band.
 */
static int
extract_band( REGION *or, REGION *ir, IMAGE *in, IMAGE_BOX *box )
{
	Rect iarea;
	Rect *r = &or->valid;
	int le = r->left;
	int ri = IM_RECT_RIGHT(r);
	int to = r->top;
	int bo = IM_RECT_BOTTOM(r);
	int es = IM_IMAGE_SIZEOF_ELEMENT( ir->im );	/* sizeof band element */
	int sk = IM_IMAGE_SIZEOF_PEL( ir->im )-es;	/* skip to next pel */
	char *p, *q, *s;
	int x, y, z;

	/* Ask for input we need.
	 */
	iarea = or->valid;
	iarea.left += box->xstart;
	iarea.top += box->ystart;
	if( im_prepare( ir, &iarea ) )
		return( -1 );

	for( y = to; y < bo; y++ ) {
		p = IM_REGION_ADDR( ir, le + box->xstart, y + box->ystart );
		q = IM_REGION_ADDR( or, le, y );
		s = p + box->chsel * es;

		for( x = le; x < ri; x++ ) {
			/* Copy the band element we want.
			 */
			for( z = 0; z < es; z++ )
				*q++ = *s++;

			/* Skip the bands we don't want.
			 */
			s += sk;
		}
	}

	return( 0 );
}

/* Extract an area. Can just use pointers.
 */
static int
extract_area( REGION *or, REGION *ir, IMAGE *in, IMAGE_BOX *box )
{
	Rect iarea;

	/* Ask for input we need. Translate from demand in or's space to
	 * demand in ir's space.
	 */
	iarea = or->valid;
	iarea.left += box->xstart;
	iarea.top += box->ystart;
	if( im_prepare( ir, &iarea ) )
		return( -1 );

	/* Attach or to ir.
	 */
	if( im_region_region( or, ir, &or->valid, iarea.left, iarea.top ) )
		return( -1 );
	
	return( 0 );
}

int
im_extract( IMAGE *in, IMAGE *out, IMAGE_BOX *box )
{	
	IMAGE_BOX *cbox;			/* Our copy of IMAGE_BOX */

	/* Check args we were passed.  
	 */
	if( box->xstart + box->xsize > in->Xsize ||
		box->ystart + box->ysize > in->Ysize ||
		box->xstart < 0 || box->ystart < 0 ||
		box->xsize <= 0 || box->ysize <= 0 ) {
		im_errormsg( "im_extract: bad extract area" );
		return( -1 );
	}
	if( box->chsel < -1 || box->chsel >= in->Bands ) {
		im_errormsg( "im_extract: band selection out of range" );
		return( -1 );
	}
	if( in->Coding != IM_CODING_NONE && in->Coding != IM_CODING_LABQ ) {
		im_errormsg( "im_extract: unknown image coding type" );
		return( -1 );
	}
	if( box->chsel != -1 && in->Coding == IM_CODING_LABQ ) {
		im_errormsg( "im_extract: can't extract band from IM_CODING_LABQ" );
		return( -1 );
	}
	if( im_piocheck( in, out ) )
		return( -1 );

	/* Set up the output header.  
	 */
	if( im_cp_desc( out, in ) ) 
                return( -1 );
	out->Xsize = box->xsize;
	out->Ysize = box->ysize;
	if( out->Xsize != in->Xsize || out->Ysize != in->Ysize ) {
		out->Xoffset = 0;
		out->Yoffset = 0;
	}

	/* Take a copy of IMAGE_BOX.
	 */
	if( !(cbox = IM_NEW( out, IMAGE_BOX )) )
		return( -1 );
	*cbox = *box;

	/* Set demand hints.
	 */
	if( im_demand_hint( out, IM_THINSTRIP, in, NULL ) )
		return( -1 );

	/* Two main cases: extract all bands, and extract one band.
	 */
	if( box->chsel >= 0 && in->Bands > 1 ) {
		out->Bands = 1;

		/* Need a special extracter and a copy operation for this.
		 */
		if( im_generate( out, 
			im_start_one, extract_band, im_stop_one,
			in, cbox ) )
			return( -1 );
	}
	else 
		/* Extract all bands. Can do it all with pointers.
		 */
		if( im_generate( out, 
			im_start_one, extract_area, im_stop_one,
			in, cbox ) )
			return( -1 );

	return( 0 );
}

/* Extract a band from an image.
 */
int
im_extract_band( IMAGE *in, IMAGE *out, int band )
{
	IMAGE_BOX box;

	if( band < 0 || band > in->Bands ) {
		im_errormsg( "im_extract_band: band out of range [0,%d]",
			in->Bands );
		return( -1 );
	}

	box.xstart = 0;
	box.ystart = 0;
	box.xsize = in->Xsize;
	box.ysize = in->Ysize;
	box.chsel = band;

	return( im_extract( in, out, &box ) );
}

/* Extract an area from an image.
 */
int
im_extract_area( IMAGE *in, IMAGE *out, int x, int y, int w, int h )
{
	IMAGE_BOX box;

	box.xstart = x;
	box.ystart = y;
	box.xsize = w;
	box.ysize = h;
	box.chsel = -1;

	return( im_extract( in, out, &box ) );
}
