/* Read a file using libMagick
 * 
 * 7/1/03 JC
 *	- from im_tiff2vips
 * 3/2/03 JC
 *	- some InitializeMagick() fail with NULL arg
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* Turn on debugging output.
#define DEBUG
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#ifndef HAVE_MAGICK

#include <vips/vips.h>

int
im_magick2vips( const char *filename, IMAGE *im )
{
	im_errormsg( "im_magick2vips: libMagick support disabled" );
	return( -1 );
}

int
im_magick2vips_header( const char *filename, IMAGE *im )
{
	im_errormsg( "im_magick2vips_header: libMagick support disabled" );
	return( -1 );
}

#else /*HAVE_MAGICK*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <sys/types.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>
#include <vips/thread.h>

#include <magick/api.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* What we track during a read call.
 */
typedef struct _Read {
	const char *filename;
	IMAGE *im;

	Image *image;
	ImageInfo *image_info;
	ExceptionInfo exception;

	/* Mutex to serialise calls to libMagick during threaded read.
	 */
	im_lock_t lock;
} Read;

static int
read_destroy( Read *read )
{
#ifdef DEBUG
	printf( "im_magick2vips: read_destroy: %s\n", read->filename );
#endif /*DEBUG*/

	if( read->image ) {
		DestroyImage( read->image );
		read->image = NULL;
	}
	if( read->image_info ) {
		DestroyImageInfo( read->image_info );
		read->image_info = NULL;
	}
	if( read->filename ) {
		im_free( (void *) read->filename );
		read->filename = NULL;
	}
	DestroyExceptionInfo( &read->exception );
	im_lock_destroy( &read->lock );
	im_free( read );

	return( 0 );
}

static Read *
read_new( const char *filename, IMAGE *im )
{
	Read *read;
	static int inited = 0;

	if( !inited ) {
		InitializeMagick( "" );
		inited = 1;
	}

	if( !(read = IM_NEW( NULL, Read )) )
		return( NULL );
	read->filename = im_strdup( NULL, filename );
	read->im = im;
	read->image = NULL;
	read->image_info = CloneImageInfo( NULL );
	GetExceptionInfo( &read->exception );
	im_lock_init( &read->lock );

        if( im_add_close_callback( im,
                (im_callback_fn) read_destroy, read, NULL ) ) {
                read_destroy( read );
                return( NULL );
        }

	if( !read->filename || !read->image_info ) 
		return( NULL );

	im_strncpy( read->image_info->filename, filename, MaxTextExtent );

#ifdef DEBUG
	printf( "im_magick2vips: read_new: %s\n", read->filename );
#endif /*DEBUG*/

	return( read );
}

static int
parse_header( Image *image, IMAGE *im )
{
	char hist[MaxTextExtent];

	im->Xsize = image->columns;
	im->Ysize = image->rows;

	switch( image->colorspace ) {
	case GRAYColorspace:
		im->Bands = 1;
		im->Type = IM_TYPE_B_W;
		break;

	case RGBColorspace:
		im->Bands = 3;
		im->Type = IM_TYPE_RGB;
		break;

	case sRGBColorspace:
		im->Bands = 3;
		im->Type = IM_TYPE_RGB;
		break;

	case CMYKColorspace:
		im->Bands = 4;
		im->Type = IM_TYPE_CMYK;
		break;

	default:
		im_errormsg( "im_magick2vips: unsupported colorspace %d",
			(int) image->colorspace );
		return( -1 );
	}

	/* Alpha as well?
	 */
	if( image->matte ) {
		assert( image->colorspace != CMYKColorspace );
		im->Bands += 1;
	}

	switch( image->depth ) {
	case 8:
		im->BandFmt = IM_BANDFMT_UCHAR;
		im->Bbits = IM_BBITS_BYTE;
		break;

	case 16:
		im->BandFmt = IM_BANDFMT_USHORT;
		im->Bbits = IM_BBITS_SHORT;
		break;

	default:
		im_errormsg( "im_magick2vips: unsupported bit depth %d",
			(int) image->depth );
		return( -1 );
	}

	switch( image->units ) {
	case PixelsPerInchResolution:
		im->Xres = image->x_resolution / 25.4;
		im->Yres = image->y_resolution / 25.4;
		break;

	case PixelsPerCentimeterResolution:
		im->Xres = image->x_resolution / 10.0;
		im->Yres = image->y_resolution / 10.0;
		break;

	default:
		im->Xres = 1.0;
		im->Yres = 1.0;
		break;
	}

	/* Other fields.
	 */
	im->Coding = IM_CODING_NONE;
	im->Xoffset = 0;
	im->Yoffset = 0;
	im_snprintf( hist, MaxTextExtent, "%s image imported by libMagick\n",
		image->magick );
	im->Hist = im_strdup( NULL, hist );

	return( 0 );
}

static void
unpack_pixels( IMAGE *im, 
	unsigned char *q8, PixelPacket *pixels, int n )
{
	unsigned short *q16 = (unsigned short *) q8;
	int x;

	switch( im->Bands ) {
	case 1:
		/* Gray.
		 */
		switch( im->BandFmt ) {
		case IM_BANDFMT_UCHAR:
			for( x = 0; x < n; x++ ) 
				q8[x] = pixels[x].green >> 8;
			break;

		case IM_BANDFMT_USHORT:
			for( x = 0; x < n; x++ )
				q16[x] = pixels[x].green;
			break;

		default:
			assert( 0 );
		}
		break;

	case 2:
		/* Gray plus alpha.
		 */
		switch( im->BandFmt ) {
		case IM_BANDFMT_UCHAR:
			for( x = 0; x < n; x++ ) {
				q8[0] = pixels[x].green >> 8;
				q8[1] = pixels[x].opacity >> 8;

				q8 += 2;
			}
			break;

		case IM_BANDFMT_USHORT:
			for( x = 0; x < n; x++ ) {
				q16[0] = pixels[x].green;
				q16[1] = pixels[x].opacity;

				q16 += 2;
			}
			break;

		default:
			assert( 0 );
		}
		break;

	case 3:
		/* RGB.
		 */
		switch( im->BandFmt ) {
		case IM_BANDFMT_UCHAR:
			for( x = 0; x < n; x++ ) {
				q8[0] = pixels[x].red >> 8;
				q8[1] = pixels[x].green >> 8;
				q8[2] = pixels[x].blue >> 8;

				q8 += 3;
			}
			break;

		case IM_BANDFMT_USHORT:
			for( x = 0; x < n; x++ ) {
				q16[0] = pixels[x].red;
				q16[1] = pixels[x].green;
				q16[2] = pixels[x].blue;

				q16 += 3;
			}
			break;

		default:
			assert( 0 );
		}
		break;

	case 4:
		/* RGBA or CMYK.
		 */
		switch( im->BandFmt ) {
		case IM_BANDFMT_UCHAR:
			for( x = 0; x < n; x++ ) {
				q8[0] = pixels[x].red >> 8;
				q8[1] = pixels[x].green >> 8;
				q8[2] = pixels[x].blue >> 8;
				q8[3] = pixels[x].opacity >> 8;

				q8 += 4;
			}
			break;

		case IM_BANDFMT_USHORT:
			for( x = 0; x < n; x++ ) {
				q16[0] = pixels[x].red;
				q16[1] = pixels[x].green;
				q16[2] = pixels[x].blue;
				q16[3] = pixels[x].opacity;

				q16 += 4;
			}
			break;

		default:
			assert( 0 );
		}
		break;

	default:
		assert( 0 );
	}
}

static PixelPacket *
get_pixels( Image *image, int left, int top, int width, int height )
{
	PixelPacket *pixels;
	
	if( !(pixels = GetImagePixels( image, left, top, width, height )) )
		return( NULL );

	/* Unpack palette.
	 */
	if( image->storage_class == PseudoClass ) {
		IndexPacket *indexes = GetIndexes( image );
		int i;

		for( i = 0; i < width * height; i++ ) {
			IndexPacket x = indexes[i];

			if( x < image->colors ) {
				pixels[i].red = image->colormap[x].red;
				pixels[i].green = image->colormap[x].green;
				pixels[i].blue = image->colormap[x].blue;
			}
		}
	}

	return( pixels );
}

static int
magick_fill_region( REGION *out, void *dummy, Read *read )
{
	Rect *r = &out->valid;
	int y;

	for( y = 0; y < r->height; y++ ) {
		PixelPacket *pixels;

		im_lock( &read->lock );
		pixels = get_pixels( read->image, 
			r->left, r->top + y, r->width, 1 );
		im_unlock( &read->lock );

		if( !pixels ) {
			im_errormsg( "im_magick2vips: unable to read pixels" );
			return( -1 );
		}

		unpack_pixels( read->im, 
			IM_REGION_ADDR( out, r->left, r->top + y ), 
			pixels, r->width );
	}

	return( 0 );
}

int
im_magick2vips( const char *filename, IMAGE *im )
{
	Read *read;

	if( !(read = read_new( filename, im )) )
		return( -1 );

	read->image = ReadImage( read->image_info, &read->exception );
	if( !read->image ) {
		im_errormsg( "im_magick2vips: unable to read file \"%s\"\n"
			"libMagick error: %s %s",
			filename, 
			read->exception.reason, read->exception.description );
		return( -1 );
	}

	if( parse_header( read->image, im ) ||
		im_poutcheck( im ) || 
		im_generate( im, NULL, magick_fill_region, NULL, read, NULL ) )
		return( -1 );

	return( 0 );
}

int
im_magick2vips_header( const char *filename, IMAGE *im )
{
	Read *read;

	if( !(read = read_new( filename, im )) )
		return( -1 );

	read->image = PingImage( read->image_info, &read->exception );
	if( !read->image ) {
		im_errormsg( "im_magick2vips: unable to ping file \"%s\"\n"
			"libMagick error: %s %s",
			filename, 
			read->exception.reason, read->exception.description );
		return( -1 );
	}

	if( parse_header( read->image, im ) ) 
		return( -1 );

	return( 0 );
}

#endif /*HAVE_MAGICK*/
