/* TIFF PARTS:
 * Copyright (c) 1988, 1990 by Sam Leffler.
 * All rights reserved.
 *
 * This file is provided for unrestricted use provided that this
 * legend is included on all tape media and as a part of the
 * software program in whole or part.  Users may copy, modify or
 * distribute this file at will.

 MODIFICATION FOR VIPS Copyright 1991, K.Martinez 
 * software may be distributed FREE, with these copyright notices
 * no responsibility/warantee is implied or given
 *
 *
 * Modified and added im_LabQ2LabC() function. It can write IM_TYPE_LABQ image
 * in vips format  to LAB in tiff format.
 *  Copyright 1994 Ahmed Abbood.
 *
 * 19/9/95 JC
 *	- calls TIFFClose() more reliably
 *	- tidied up
 * 12/4/97 JC
 *	- thrown away and rewritten for TIFF 6 lib
 * 22/4/97 JC
 *	- writes a pyramid!
 *	- to separate TIFF files tho'
 * 23/4/97 JC
 *	- does 2nd gather pass to put pyramid into a single TIFF file
 *	- ... and shrinks IM_CODING_LABQ too
 * 26/10/98 JC
 *	- binary open for stupid systems
 * 7/6/99 JC
 *	- 16bit TIFF write too
 * 9/7/99 JC
 *	- ZIP tiff added
 * 11/5/00 JC
 *	- removed TIFFmalloc/TIFFfree
 * 5/8/00 JC
 *	- mode string now part of filename
 * 23/4/01 JC
 *	- HAVE_TIFF turns on TIFFness
 * 19/3/02 ruven
 *	- pyramid stops at tile size, not 64x64
 * 29/4/02 JC
 * 	- write any number of bands (but still with photometric RGB, so not
 * 	  very useful)
 * 10/9/02 JC
 *	- oops, handle TIFF errors better
 *	- now writes CMYK correctly
 * 13/2/03 JC
 *	- tries not to write mad resolutions
 * 7/5/03 JC
 *	- only write CMYK if Type == CMYK
 *	- writes EXTRASAMPLES ALPHA for bands == 2 or 4 (if we're writing RGB)
 * 17/11/03 JC
 *	- write float too
 * 28/11/03 JC
 *	- read via a "p" so we work from mmap window images
 *	- uses threadgroups for speedup
 * 9/3/04 JC
 *	- 1 bit write mode added
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* Turn on IM_REGION_ADDR() range checks, don't delete intermediates.
#define DEBUG
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#ifndef HAVE_TIFF

#include <vips/vips.h>

int
im_vips2tiff( IMAGE *im, const char *filename )
{
	im_errormsg( "im_vips2tiff: TIFF support disabled" );

	return( -1 );
}

#else /*HAVE_TIFF*/

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /*HAVE_UNISTD_H*/
#include <string.h>
#include <assert.h>

#include <vips/vips.h>
#include <vips/util.h>
#include <vips/region.h>
#include <vips/thread.h>
#include <vips/threadgroup.h>

#include <tiffio.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Max no of tiles we buffer in a layer. Enough to buffer a line of 64x64
 * tiles on a 20k by 20k image.
 */
#define IM_MAX_LAYER_BUFFER (200)

/* Bits we OR together for quadrants in a tile.
 */
typedef enum pyramid_bits {
	PYR_TL = 1,			/* Top-left etc. */
	PYR_TR = 2,
	PYR_BL = 4,
	PYR_BR = 8,
	PYR_ALL = 15,
	PYR_NONE = 0
} PyramidBits;

/* A tile in our pyramid.
 */
typedef struct pyramid_tile {
	REGION *tile;
	PyramidBits bits;
} PyramidTile;

/* A layer in the pyramid.
 */
typedef struct pyramid_layer {
	/* Parameters.
	 */
	struct tiff_write *tw;		/* Main TIFF write struct */
	int width, height;		/* Layer size */
	int sub;			/* Subsample factor for this layer */

	char *lname;			/* Name of this TIFF file */
	TIFF *tif;			/* TIFF file we write this layer to */
	PEL *tbuf;			/* TIFF output buffer */
	PyramidTile tiles[IM_MAX_LAYER_BUFFER];

	struct pyramid_layer *below;	/* Tiles go to here */
	struct pyramid_layer *above;	/* Tiles come from here */
} PyramidLayer;

/* A TIFF image in the process of being written.
 */
typedef struct tiff_write {
	IMAGE *input_im;		/* Original input image */
	IMAGE *im;			/* Image we read from ... */
	char *name;			/* Final name we write to */
	char *mode;			/* Mode string */

	/* Read from im with these.
	 */
	REGION *reg;
	im_threadgroup_t *tg;

	char *bname;			/* Name for base layer */
	TIFF *tif;			/* Image we write to */

	PyramidLayer *layer;		/* Top of pyramid, if in use */
	PEL *tbuf;			/* TIFF output buffer */

	int compression;		/* Compression type */
	int jpqual;			/* JPEG q-factor */
	int tile;			/* Tile or not */
	int tilew, tileh;		/* Tile size */
	int pyramid;			/* Write pyramid */
	int onebit;			/* Write as 1-bit TIFF */
} TiffWrite;

/* Handle TIFF errors here. 
 */
static void 
vhandle( char *module, char *fmt, va_list ap )
{
	im_errormsg( "TIFF error in \"%s\": ", module );
	im_verrormsg( fmt, ap );
}

/* Open TIFF for output.
 */
static TIFF *
tiff_openout( char *name )
{
	TIFF *tif;

#ifdef BINARY_OPEN
	if( !(tif = TIFFOpen( name, "wb" )) ) {
#else /*BINARY_OPEN*/
	if( !(tif = TIFFOpen( name, "w" )) ) {
#endif /*BINARY_OPEN*/
		im_errormsg( "im_vips2tiff: unable to open \"%s\" for output",
			name );
		return( NULL );
	}

	return( tif );
}

/* Open TIFF for input.
 */
static TIFF *
tiff_openin( char *name )
{
	TIFF *tif;

#ifdef BINARY_OPEN
	if( !(tif = TIFFOpen( name, "rb" )) ) {
#else /*BINARY_OPEN*/
	if( !(tif = TIFFOpen( name, "r" )) ) {
#endif /*BINARY_OPEN*/
		im_errormsg( "im_vips2tiff: unable to open \"%s\" for input",
			name );
		return( NULL );
	}

	return( tif );
}

/* Convert VIPS LabQ to TIFF LAB. Just take the first three bands.
 */
static void
LabQ2LabC( PEL *q, PEL *p, int n )
{
        int x;

        for( x = 0; x < n; x++ ) {
                /* Get most significant 8 bits of lab.
                 */
                q[0] = p[0];
                q[1] = p[1];
                q[2] = p[2];

                p += 4;
                q += 3;
        }
}

/* Pack 8 bit VIPS to 1 bit TIFF.
 */
static void
eightbit2onebit( PEL *q, PEL *p, int n )
{
        int x;
	PEL bits;

        for( bits = 0, x = 0; x < n; x++, bits <<= 1 ) {
		if( p[x] )
			bits |= 1;

		if( (x & 0x7) == 0x7 ) {
			*q++ = bits;
			bits = 0;
		}
        }

	/* Any left-over bits? Need to be left-aligned.
	 */
	if( (x & 0x7) != 0 ) 
		*q++ = (bits >> 1) << (8 - (x & 0x7));
}

/* Round N down to P boundary. 
 */
#define ROUND_DOWN(N,P) ((N) - ((N) % P)) 

/* Round N up to P boundary. 
 */
#define ROUND_UP(N,P) (ROUND_DOWN( (N) + (P) - 1, (P) ))

/* Pack a VIPS region into a TIFF tile buffer.
 */
static void
pack2tiff( TiffWrite *tw, REGION *in, PEL *q )
{
	Rect *r = &in->valid;
	int tls;
	int y;

	/* Sizeof a line of bytes in the TIFF tile.
	 */
	if( in->im->Coding == IM_CODING_LABQ )
		tls = r->width * 3;
	else if( tw->onebit )
		tls = ROUND_UP( r->width, 8 ) / 8;
	else
		tls = IM_IMAGE_SIZEOF_PEL( in->im ) * r->width;

	for( y = r->top; y < IM_RECT_BOTTOM( r ); y++ ) {
		PEL *p = (PEL *) IM_REGION_ADDR( in, r->left, y );

		if( in->im->Coding == IM_CODING_LABQ )
			LabQ2LabC( q, p, r->width );
		else if( tw->onebit ) 
			eightbit2onebit( q, p, r->width );
		else
			memcpy( q, p, tls );

		q += tls;
	}
}

/* Write a TIFF header. width and height are the size of the IMAGE we are
 * writing (may have been shrunk!).
 */
static void
write_tiff_header( TiffWrite *tw, TIFF *tif, int width, int height )
{
	/* Don't write mad resolutions (eg. zero), it confuses some programs.
	 */
	double xres = IM_CLIP( 1, tw->im->Xres * 10.0, 10000 );
	double yres = IM_CLIP( 1, tw->im->Yres * 10.0, 10000 );

	uint16 v[1];

	/* Output base header fields.
	 */
	TIFFSetField( tif, TIFFTAG_IMAGEWIDTH, width );
	TIFFSetField( tif, TIFFTAG_IMAGELENGTH, height );
	TIFFSetField( tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG );
	TIFFSetField( tif, TIFFTAG_COMPRESSION, tw->compression );
	TIFFSetField( tif, TIFFTAG_XRESOLUTION, xres );
	TIFFSetField( tif, TIFFTAG_YRESOLUTION, yres );
	TIFFSetField( tif, TIFFTAG_RESOLUTIONUNIT, RESUNIT_CENTIMETER );

	if( tw->compression == COMPRESSION_JPEG ) {
		TIFFSetField( tif, TIFFTAG_JPEGQUALITY, tw->jpqual );
		TIFFSetField( tif, TIFFTAG_JPEGCOLORMODE, JPEGCOLORMODE_RGB );
	}

	/* And colour fields.
	 */
	if( tw->im->Coding == IM_CODING_LABQ ) {
		TIFFSetField( tif, TIFFTAG_SAMPLESPERPIXEL, 3 );
		TIFFSetField( tif, TIFFTAG_BITSPERSAMPLE, 8 );
		TIFFSetField( tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_CIELAB );
	}
	else if( tw->onebit ) {
		TIFFSetField( tif, TIFFTAG_SAMPLESPERPIXEL, 1 );
		TIFFSetField( tif, TIFFTAG_BITSPERSAMPLE, 1 );
		TIFFSetField( tif, 
			TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK );
	}
	else {
		TIFFSetField( tif, TIFFTAG_SAMPLESPERPIXEL, tw->im->Bands );
		TIFFSetField( tif, TIFFTAG_BITSPERSAMPLE, tw->im->Bbits );

		switch( tw->im->Bands ) {
		case 1:
			TIFFSetField( tif, 
				TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK );
			break;

		case 2:
			TIFFSetField( tif, 
				TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK );
			v[0] = EXTRASAMPLE_ASSOCALPHA;
	                TIFFSetField( tif, TIFFTAG_EXTRASAMPLES, 1, v );
			break;

		case 3:
			TIFFSetField( tif, 
				TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB );
			break;

		case 4:
			if( tw->im->Type == IM_TYPE_CMYK ) {
				TIFFSetField( tif, 
					TIFFTAG_PHOTOMETRIC, 
					PHOTOMETRIC_SEPARATED );
				TIFFSetField( tif, 
					TIFFTAG_INKSET, INKSET_CMYK );
			}
			else {
				TIFFSetField( tif, 
					TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB );
				v[0] = EXTRASAMPLE_ASSOCALPHA;
				TIFFSetField( tif, TIFFTAG_EXTRASAMPLES, 1, v );
			}
			break;

		default:
			assert( 0 );
		}
	}

	/* Layout.
	 */
	if( tw->tile ) {
		TIFFSetField( tif, TIFFTAG_TILEWIDTH, tw->tilew );
		TIFFSetField( tif, TIFFTAG_TILELENGTH, tw->tileh );
	}
	else
		TIFFSetField( tif, TIFFTAG_ROWSPERSTRIP, 16 );

	/* Sample format ... for float, we write IEEE.
	 */
	if( tw->im->BandFmt == IM_BANDFMT_FLOAT )
		TIFFSetField( tif, TIFFTAG_SAMPLEFORMAT, SAMPLEFORMAT_IEEEFP );
}

/* Free a pyramid layer.
 */
static void
free_layer( PyramidLayer *layer )
{
	int i;

	for( i = 0; i < IM_MAX_LAYER_BUFFER; i++ )
		if( layer->tiles[i].tile ) {
			im_region_free( layer->tiles[i].tile );
			layer->tiles[i].tile = NULL;
		}

	/* And close the TIFF file we are writing to.
	 */
	if( layer->tbuf ) {
		im_free( layer->tbuf );
		layer->tbuf = NULL;
	}
	if( layer->tif ) {
		TIFFClose( layer->tif );
		layer->tif = NULL;
	}
}

/* Free an entire pyramid.
 */
static void
free_pyramid( PyramidLayer *layer )
{
	if( layer->below ) 
		free_pyramid( layer->below );

	free_layer( layer );
}

/* Make a name for a new TIFF layer. Base it on sub factor.
 */
static char *
new_tiff_name( TiffWrite *tw, char *name, int sub )
{
	char buf[FILENAME_MAX];
	char buf2[FILENAME_MAX];

	/* Remove existing .tif/.tiff suffix, if any.
	 */
	strcpy( buf, name );
	if( im__ispostfix( buf, ".tif" ) )
		buf[strlen( buf ) - 4] = '\0';
	if( im__ispostfix( buf, ".tiff" ) )
		buf[strlen( buf ) - 5] = '\0';

	im_snprintf( buf2, FILENAME_MAX, "%s.%d.tif", buf, sub );

	return( im_strdup( tw->im, buf2 ) );
}

/* Build a pyramid. w & h are size of layer above this layer. Write new layer
 * struct into *zap, return 0/-1 for success/fail.
 */
static int
build_pyramid( TiffWrite *tw, PyramidLayer *above, 
	PyramidLayer **zap, int w, int h )
{
	PyramidLayer *layer = IM_NEW( tw->im, PyramidLayer );
	int i;

	if( !layer )
		return( -1 );
	layer->tw = tw;
	layer->width = w / 2;
	layer->height = h / 2;

	if( !above )
		/* Top of pyramid.
		 */
		layer->sub = 2;	
	else
		layer->sub = above->sub * 2;

	layer->lname = NULL;
	layer->tif = NULL;
	layer->tbuf = NULL;

	for( i = 0; i < IM_MAX_LAYER_BUFFER; i++ ) {
		layer->tiles[i].tile = NULL;
		layer->tiles[i].bits = PYR_NONE;
	}

	layer->below = NULL;
	layer->above = above;

	/* Save layer, to make sure it gets freed properly.
	 */
	*zap = layer;

	if( layer->width > tw->tilew || layer->height > tw->tileh ) 
		if( build_pyramid( tw, layer, 
			&layer->below, layer->width, layer->height ) )
			return( -1 );

	if( !(layer->lname = new_tiff_name( tw, tw->name, layer->sub )) )
		return( -1 );

	/* Make output image.
	 */
	if( !(layer->tif = tiff_openout( layer->lname )) ) 
		return( -1 );

	/* Write the TIFF header for this layer.
	 */
	write_tiff_header( tw, layer->tif, layer->width, layer->height );

	if( !(layer->tbuf = im_malloc( NULL, TIFFTileSize( layer->tif ) )) ) 
		return( -1 );

	return( 0 );
}

/* Pick a new tile to write to in this layer. Either reuse a tile we have
 * previously filled, or make a new one.
 */
static int
find_new_tile( PyramidLayer *layer )
{
	int i;

	/* Exisiting buffer we have finished with? 
	 */
	for( i = 0; i < IM_MAX_LAYER_BUFFER; i++ )
		if( layer->tiles[i].bits == PYR_ALL ) 
			return( i );

	/* Have to make a new one.
	 */
	for( i = 0; i < IM_MAX_LAYER_BUFFER; i++ )
		if( !layer->tiles[i].tile ) {
			if( !(layer->tiles[i].tile = 
				im_region_create( layer->tw->im )) )
				return( -1 );
			return( i );
		}

	/* Out of space!
	 */
	im_errormsg( "im_vips2tiff: layer buffer exhausted -- "
		"try making TIFF output tiles smaller" );
	return( -1 );
}

/* Find a tile in the layer buffer - if it's not there, make a new one.
 */
static int
find_tile( PyramidLayer *layer, Rect *pos )
{
	int i;
	Rect quad;
	Rect image;
	Rect inter;

	/* Do we have a REGION for this position?
	 */
	for( i = 0; i < IM_MAX_LAYER_BUFFER; i++ ) {
		REGION *reg = layer->tiles[i].tile;

		if( reg && reg->valid.left == pos->left && 
			reg->valid.top == pos->top )
			return( i );
	}

	/* Make a new one.
	 */
	if( (i = find_new_tile( layer )) < 0 )
		return( -1 );
	if( im_region_local( layer->tiles[i].tile, pos ) )
		return( -1 );
	layer->tiles[i].bits = PYR_NONE;

	/* Do any quadrants of this tile fall entirely outside the image? 
	 * If they do, set their bits now.
	 */
	quad.width = layer->tw->tilew / 2;
	quad.height = layer->tw->tileh / 2;
	image.left = 0;
	image.top = 0;
	image.width = layer->width;
	image.height = layer->height;

	quad.left = pos->left;
	quad.top = pos->top;
	im_rect_intersectrect( &quad, &image, &inter );
	if( im_rect_isempty( &inter ) )
		layer->tiles[i].bits |= PYR_TL;

	quad.left = pos->left + quad.width;
	quad.top = pos->top;
	im_rect_intersectrect( &quad, &image, &inter );
	if( im_rect_isempty( &inter ) )
		layer->tiles[i].bits |= PYR_TR;

	quad.left = pos->left;
	quad.top = pos->top + quad.height;
	im_rect_intersectrect( &quad, &image, &inter );
	if( im_rect_isempty( &inter ) )
		layer->tiles[i].bits |= PYR_BL;

	quad.left = pos->left + quad.width;
	quad.top = pos->top + quad.height;
	im_rect_intersectrect( &quad, &image, &inter );
	if( im_rect_isempty( &inter ) )
		layer->tiles[i].bits |= PYR_BR;

	return( i );
}

/* Shrink a region by a factor of two, writing the result to a specified 
 * offset in another region. IM_CODING_LABQ only.
 */
static void
shrink_region_labpack( REGION *from, REGION *to, int xoff, int yoff )
{
	int ls = IM_REGION_LSKIP( from );
	Rect *f = &from->valid;
	Rect *t = &to->valid;

	int x, y;
	Rect out;

	/* Calculate output size and position.
	 */
	out.left = t->left + xoff;
	out.top = t->top + yoff;
	out.width = f->width / 2;
	out.height = f->height / 2;

	/* Shrink ... ignore the extension byte for speed.
	 */
	for( y = 0; y < out.height; y++ ) {
		PEL *p = (PEL *) 
			IM_REGION_ADDR( from, f->left, f->top + y * 2 );
		PEL *q = (PEL *) 
			IM_REGION_ADDR( to, out.left, out.top + y );

		for( x = 0; x < out.width; x++ ) {
			signed char *sp = (signed char *) p;
			unsigned char *up = (unsigned char *) p;

			int l = up[0] + up[4] + 
				up[ls] + up[ls + 4];
			int a = sp[1] + sp[5] + 
				sp[ls + 1] + sp[ls + 5];
			int b = sp[2] + sp[6] + 
				sp[ls + 2] + sp[ls + 6];

			q[0] = l >> 2;
			q[1] = a >> 2;
			q[2] = b >> 2;
			q[3] = 0;

			q += 4;
			p += 8;
		}
	}
}

/* Shrink a region by a factor of two, writing the result to a specified 
 * offset in another region. n-band uchar only.
 */
static void
shrink_region_uchar( REGION *from, REGION *to, int xoff, int yoff )
{
	int ls = IM_REGION_LSKIP( from );
	int ps = IM_IMAGE_SIZEOF_PEL( from->im );
	int ls1 = ls + ps;
	Rect *f = &from->valid;
	Rect *t = &to->valid;

	int x, y, z;
	Rect out;

	/* Calculate output size and position.
	 */
	out.left = t->left + xoff;
	out.top = t->top + yoff;
	out.width = f->width / 2;
	out.height = f->height / 2;

	for( y = 0; y < out.height; y++ ) {
		PEL *p = (PEL *) 
			IM_REGION_ADDR( from, f->left, f->top + y * 2 );
		PEL *q = (PEL *) 
			IM_REGION_ADDR( to, out.left, out.top + y );

		for( x = 0; x < out.width; x++ ) {
			for( z = 0; z < from->im->Bands; z++ ) {
				int tot = p[z] + p[z + ps] + 
					p[z + ls] + p[z + ls1];
				
				q[z] = tot >> 2;
			}

			/* Move on two pels in input.
			 */
			p += ps << 1;
			q += ps;
		}
	}
}

/* Write a tile from a layer.
 */
static int
save_tile( TiffWrite *tw, TIFF *tif, PEL *tbuf, REGION *reg )
{
	/* Have to repack pixels.
	 */
	pack2tiff( tw, reg, tbuf );

#ifdef DEBUG
	printf( "Writing %dx%d pixels at position %dx%d to image %s\n",
		tw->tilew, tw->tileh, reg->valid.left, reg->valid.top,
		TIFFFileName( tif ) );
#endif /*DEBUG*/

	/* Write to TIFF! easy.
	 */
	if( TIFFWriteTile( tif, tbuf, 
		reg->valid.left, reg->valid.top, 0, 0 ) < 0 ) {
		im_errormsg( "TIFF write tile failed" );
		return( -1 );
	}

	return( 0 );
}

/* A new tile has arrived! Shrink into this layer, if we fill a region, write
 * it and recurse.
 */
static int
new_tile( PyramidLayer *layer, REGION *tile )
{
	TiffWrite *tw = layer->tw;
	Rect *r = &tile->valid;
	int xoff, yoff;

	int t, ri, bo;
	Rect out, new;
	PyramidBits bit;

	/* Calculate pos and size of new pixels we make inside this layer.
	 */
	new.left = r->left / 2;
	new.top = r->top / 2;
	new.width = r->width / 2;;
	new.height = r->height / 2;;

	/* Has size fallen to zero? Can happen if this is a one-pixel-wide
	 * strip.
	 */
	if( im_rect_isempty( &new ) )
		return( 0 );

	/* Offset into this tile ... ie. which quadrant we are writing.
	 */
	xoff = new.left % layer->tw->tilew;
	yoff = new.top % layer->tw->tileh;

	/* Calculate pos for tile we shrink into in this layer.
	 */
	out.left = new.left - xoff;
	out.top = new.top - yoff;

	/* Clip against edge of image.
	 */
	ri = IM_MIN( layer->width, out.left + layer->tw->tilew );
	bo = IM_MIN( layer->height, out.top + layer->tw->tileh );
	out.width = ri - out.left;
	out.height = bo - out.top;

	if( (t = find_tile( layer, &out )) < 0 )
		return( -1 );

	/* Shrink into place.
	 */
	if( tw->im->Coding == IM_CODING_NONE )
		shrink_region_uchar( tile, layer->tiles[t].tile, xoff, yoff );
	else
		shrink_region_labpack( tile, layer->tiles[t].tile, xoff, yoff );

	/* Set that bit.
	 */
	if( xoff )
		if( yoff )
			bit = PYR_BR;
		else
			bit = PYR_TR;
	else
		if( yoff )
			bit = PYR_BL;
		else
			bit = PYR_TL;
	if( layer->tiles[t].bits & bit ) {
		im_errormsg( "im_tiff2vips: internal error #9876345" );
		return( -1 );
	}
	layer->tiles[t].bits |= bit;

	if( layer->tiles[t].bits == PYR_ALL ) {
		/* Save this complete tile.
		 */
		if( save_tile( tw, layer->tif, 
			layer->tbuf, layer->tiles[t].tile ) )
			return( -1 );

		/* And recurse down the pyramid!
		 */
		if( layer->below &&
			new_tile( layer->below, layer->tiles[t].tile ) )
			return( -1 );
	}

	return( 0 );
}

/* Write as tiles.
 */
static int
write_tif_tile( TiffWrite *tw )
{
	IMAGE *im = tw->im;
	Rect area;
	int x, y;

	if( !(tw->tbuf = im_malloc( NULL, TIFFTileSize( tw->tif ) )) ) 
		return( -1 );

	/* Write pyramid too? Only bother if bigger than tile size.
	 */
	if( tw->pyramid && 
		(im->Xsize > tw->tilew || im->Ysize > tw->tileh) &&
		build_pyramid( tw, NULL, &tw->layer, im->Xsize, im->Ysize ) )
			return( -1 );

	for( y = 0; y < im->Ysize; y += tw->tileh ) 
		for( x = 0; x < im->Xsize; x += tw->tilew ) {
			/* Set up rect we ask for.
			 */
			area.left = x;
			area.top = y;
			area.width = tw->tilew;
			area.height = tw->tileh;

			if( im_prepare_thread( tw->tg, tw->reg, &area ) )
				return( -1 );

			/* Write to TIFF.
			 */
			if( save_tile( tw, tw->tif, tw->tbuf, tw->reg ) )
				return( -1 );

			/* Is there a pyramid? Write to that too.
			 */
			if( tw->layer && new_tile( tw->layer, tw->reg ) )
				return( -1 );
		}

	return( 0 );
}

/* Write as scan-lines.
 */
static int
write_tif_strip( TiffWrite *tw )
{
	IMAGE *im = tw->im;
	Rect area;
	int y;

	if( !(tw->tbuf = im_malloc( NULL, TIFFScanlineSize( tw->tif ) )) ) 
		return( -1 );

	/* Set up rect we ask for.
	 */
	area.left = 0;
	area.top = 0;
	area.width = im->Xsize;
	area.height = 1;

	for( y = 0; y < im->Ysize; y++ ) {
		PEL *p;

		area.top = y;
		if( im_prepare_thread( tw->tg, tw->reg, &area ) )
			return( -1 );
		p = (PEL *) IM_REGION_ADDR( tw->reg, 0, y );

		/* Any repacking?
		 */
		if( im->Coding == IM_CODING_LABQ ) {
			LabQ2LabC( tw->tbuf, p, im->Xsize );
			p = tw->tbuf;
		}
		if( tw->onebit ) {
			eightbit2onebit( tw->tbuf, p, im->Xsize );
			p = tw->tbuf;
		}

		/* Write to TIFF! easy.
		 */
		if( TIFFWriteScanline( tw->tif, p, y, 0 ) < 0 ) {
			im_errormsg( "TIFF write failed" );
			return( -1 );
		}
	}

	return( 0 );
}

/* Delete any temp files we wrote.
 */
static void
delete_files( TiffWrite *tw )
{
	PyramidLayer *layer = tw->layer;

	if( tw->bname ) {
		unlink( tw->bname );
		tw->bname = NULL;
	}

	for( layer = tw->layer; layer; layer = layer->below ) 
		if( layer->lname ) {
			unlink( layer->lname );
			layer->lname = NULL;
		}
}

/* Free a TiffWrite.
 */
static void
free_tiff_write( TiffWrite *tw )
{
#ifndef DEBUG
	delete_files( tw );
#endif /*DEBUG*/

	if( tw->tg ) {
		im_threadgroup_free( tw->tg );
		tw->tg = NULL;
	}
	if( tw->reg ) {
		im_region_free( tw->reg );
		tw->reg = NULL;
	}
	if( tw->im ) {
		im_close( tw->im );
		tw->im = NULL;
	}
	if( tw->tif ) {
		TIFFClose( tw->tif );
		tw->tif = NULL;
	}
	if( tw->tbuf ) {
		im_free( tw->tbuf );
		tw->tbuf = NULL;
	}
	if( tw->layer ) 
		free_pyramid( tw->layer );
}

/* Make and init a TiffWrite.
 */
static TiffWrite *
make_tiff_write( IMAGE *im, const char *filename )
{
	TiffWrite *tw;
	char *p, *q, *r;
	char name[FILENAME_MAX];
	char mode[FILENAME_MAX];
	char buf[FILENAME_MAX];

	if( !(tw = IM_NEW( im, TiffWrite )) )
		return( NULL );
	tw->input_im = im;
	tw->im = NULL;
	im__filename_split( filename, name, mode );
	tw->name = im_strdup( im, name );
	tw->mode = im_strdup( im, mode );
	tw->tg = NULL;
	tw->reg = NULL;
	tw->bname = NULL;
	tw->tif = NULL;
	tw->layer = NULL;
	tw->tbuf = NULL;
	tw->compression = COMPRESSION_NONE;
	tw->jpqual = 75;
	tw->tile = 0;
	tw->tilew = 128;
	tw->tileh = 128;
	tw->pyramid = 0;
	tw->onebit = 0;

	/* Parse mode string.
	 */
	strcpy( buf, mode ); 
	p = &buf[0];
	if( (q = im__getnextoption( &p )) ) {
		if( im__isprefix( "none", q ) ) 
			tw->compression = COMPRESSION_NONE;
		else if( im__isprefix( "packbits", q ) ) 
			tw->compression = COMPRESSION_PACKBITS;
		else if( im__isprefix( "lzw", q ) ) 
			tw->compression = COMPRESSION_LZW;
		else if( im__isprefix( "deflate", q ) ) 
			tw->compression = COMPRESSION_DEFLATE;
		else if( im__isprefix( "jpeg", q ) ) {
			tw->compression = COMPRESSION_JPEG;

			if( (r = im__getsuboption( q )) ) 
				if( sscanf( r, "%d", &tw->jpqual ) != 1 ) {
					im_errormsg( "im_vips2tiff: bad JPEG "
						"quality parameter" );
					return( NULL );
				}
		}
		else {
			im_errormsg( "im_vips2tiff: unknown compression mode "
				"\"%s\"\nshould be one of \"none\", "
				"\"packbits\", \"lzw\", \"deflate\""
				"or \"jpeg\"", q );
			return( NULL );
		}
	}
	if( (q = im__getnextoption( &p )) ) {
		if( im__isprefix( "tile", q ) ) {
			tw->tile = 1;

			if( (r = im__getsuboption( q )) ) {
				if( sscanf( r, "%dx%d", 
					&tw->tilew, &tw->tileh ) != 2 ) {
					im_errormsg( "im_vips2tiff: bad tile "
						"sizes" );
					return( NULL );
				}

				if( tw->tilew < 10 || tw->tileh < 10 || 
					tw->tilew > 1000 || tw->tileh > 1000 ) {
					im_errormsg( "im_vips2tiff: bad tile "
						"size %dx%d", 
						tw->tilew, tw->tileh );
					return( NULL );
				}

				if( (tw->tilew & 0xf) != 0 || 
					(tw->tileh & 0xf) != 0 ){
					im_errormsg( "im_vips2tiff: tile "
						"size not a multiple of 16" );
					return( NULL );
				}
			}
		}
		else if( im__isprefix( "strip", q ) ) 
			tw->tile = 0;
		else {
			im_errormsg( "im_vips2tiff: unknown layout mode "
				"\"%s\"\nshould be one of \"tile\" or "
				"\"strip\"", q );
			return( NULL );
		}
	}
	if( (q = im__getnextoption( &p )) ) {
		if( im__isprefix( "pyramid", q ) ) 
			tw->pyramid = 1;
		else if( im__isprefix( "flat", q ) ) 
			tw->pyramid = 0;
		else {
			im_errormsg( "im_vips2tiff: unknown multi-res mode "
				"\"%s\"\nshould be one of \"flat\" or "
				"\"pyramid\"", q );
			return( NULL );
		}
	}
	if( (q = im__getnextoption( &p )) ) {
		if( im__isprefix( "onebit", q ) ) 
			tw->onebit = 1;
		else if( im__isprefix( "manybit", q ) ) 
			tw->onebit = 0;
		else {
			im_errormsg( "im_vips2tiff: unknown format "
				"\"%s\"\nshould be one of \"onebit\" or "
				"\"manybit\"", q );
			return( NULL );
		}
	}
	if( (q = im__getnextoption( &p )) ) {
		im_errormsg( "im_vips2tiff: unknown extra options \"%s\"", q );
		return( NULL );
	}
	if( !tw->tile && tw->pyramid ) {
		im_warning( "im_vips2tiff: can't have strip pyramid -- "
			"enabling tiling" );
		tw->tile = 1;
	}

	/* We can only pyramid LABQ and uchar images. Add another tile
	 * shrinker is this becomes important.
	 */
	if( tw->pyramid ) {
		if( im->Coding == IM_CODING_NONE && 
			im->BandFmt != IM_BANDFMT_UCHAR ) {
			im_errormsg( "im_vips2tiff: can only pyramid LABQ and "
				"uchar images" );
			return( NULL );
		}
	}

	/* Only 1-bit-ize 8 bit mono images.
	 */
	if( tw->onebit ) {
		if( im->Coding != IM_CODING_NONE || 
			im->BandFmt != IM_BANDFMT_UCHAR ||
			im->Bands != 1 ) 
			tw->onebit = 0;
	}

	if( tw->onebit && tw->compression == COMPRESSION_JPEG ) {
		im_warning( "im_vips2tiff: can't have 1-bit JPEG -- "
			"disabling JPEG" );
		tw->compression = COMPRESSION_NONE;
	}

	if( !(tw->im = im_open( "im_iterate_buffer", "p" )) || 
		im_copy( im, tw->im ) ||
		!(tw->reg = im_region_create( tw->im )) ||
		!(tw->tg = im_threadgroup_create( tw->im )) ) {
		free_tiff_write( tw );
		return( NULL );
	}

	return( tw );
}

/* Copy fields.
 */
#define CopyField(tag, v) \
    if (TIFFGetField(in, tag, &v)) TIFFSetField(out, tag, v)

/* Copy a TIFF file ... we know we wrote it, so just copy the tags we know 
 * we might have set.
 */
static int
tiff_copy( TIFF *out, TIFF *in )
{
	uint32 i32;
	uint16 i16;
	int i;
	float f;
	tdata_t buf;
	ttile_t tile;
	ttile_t n;

	/* All the fields we might have set.
	 */
	CopyField( TIFFTAG_IMAGEWIDTH, i32 );
	CopyField( TIFFTAG_IMAGELENGTH, i32 );
	CopyField( TIFFTAG_PLANARCONFIG, i16 );
	CopyField( TIFFTAG_COMPRESSION, i16 );
	CopyField( TIFFTAG_XRESOLUTION, f );
	CopyField( TIFFTAG_YRESOLUTION, f );
	CopyField( TIFFTAG_RESOLUTIONUNIT, i16 );
	CopyField( TIFFTAG_JPEGQUALITY, i );
	CopyField( TIFFTAG_SAMPLESPERPIXEL, i16 );
	CopyField( TIFFTAG_BITSPERSAMPLE, i16 );
	CopyField( TIFFTAG_PHOTOMETRIC, i16 );
	CopyField( TIFFTAG_TILEWIDTH, i32 );
	CopyField( TIFFTAG_TILELENGTH, i32 );
	CopyField( TIFFTAG_ROWSPERSTRIP, i32 );

	buf = im_malloc( NULL, TIFFTileSize( in ) );
	n = TIFFNumberOfTiles( in );
	for( tile = 0; tile < n; tile++ ) {
		tsize_t len;

		len = TIFFReadEncodedTile( in, tile, buf, (tsize_t) -1 );
		if( len < 0 ||
			TIFFWriteEncodedTile( out, tile, buf, len ) < 0 ) {
			im_free( buf );
			return( -1 );
		}
	}
	im_free( buf );

	return( 0 );
}

/* Append a file to a TIFF file.
 */
static int
tiff_append( TIFF *out, char *name )
{
	TIFF *in;

	if( !(in = tiff_openin( name )) ) 
		return( -1 );

	if( tiff_copy( out, in ) ) {
		TIFFClose( in );
		return( -1 );
	}
	TIFFClose( in );

	if( !TIFFWriteDirectory( out ) ) 
		return( -1 );

	return( 0 );
}

/* Gather all of the files we wrote into single output file.
 */
static int
gather_pyramid( TiffWrite *tw )
{
	PyramidLayer *layer;
	TIFF *out;

#ifdef DEBUG
	printf( "Starting pyramid gather ...\n" );
#endif /*DEBUG*/

	if( !(out = tiff_openout( tw->name )) ) 
		return( -1 );

	if( tiff_append( out, tw->bname ) ) {
		TIFFClose( out );
		return( -1 );
	}

	for( layer = tw->layer; layer; layer = layer->below ) 
		if( tiff_append( out, layer->lname ) ) {
			TIFFClose( out );
			return( -1 );
		}

	TIFFClose( out );

#ifdef DEBUG
	printf( "Pyramid built\n" );
#endif /*DEBUG*/

	return( 0 );
}

int
im_vips2tiff( IMAGE *im, const char *filename )
{
	TiffWrite *tw;
	int res;

#ifdef DEBUG
	printf( "im_tiff2vips: libtiff version is \"%s\"\n", TIFFGetVersion() );
#endif /*DEBUG*/

	/* Override the default TIFF error handler.
	 */
	TIFFSetErrorHandler( (TIFFErrorHandler) vhandle );

	/* Check input image.
	 */
	if( im_pincheck( im ) )
		return( -1 );
	if( im->Coding != IM_CODING_LABQ && im->Coding != IM_CODING_NONE ) {
		im_errormsg( "im_vips2tiff: unknown coding type" );
		return( -1 );
	}
	if( im->BandFmt != IM_BANDFMT_UCHAR && 
		im->BandFmt != IM_BANDFMT_USHORT &&
		im->BandFmt != IM_BANDFMT_FLOAT ) {
		im_errormsg( "im_vips2tiff: unsigned 8- and 16-bit int, "
			"and 32-bit float only" );
		return( -1 );
	}
	if( im->Coding == IM_CODING_NONE ) {
		if( im->Bands < 1 || im->Bands > 4 ) {
			im_errormsg( "im_vips2tiff: 1 to 4 bands only" );
			return( -1 );
		}
	}

	/* Make output image. If this is a pyramid, write the base image to
	 * fred.1.tif rather than fred.tif.
	 */
	if( !(tw = make_tiff_write( im, filename )) )
		return( -1 );
	if( tw->pyramid ) {
		if( !(tw->bname = new_tiff_name( tw, tw->name, 1 )) ||
			!(tw->tif = tiff_openout( tw->bname )) ) {
			free_tiff_write( tw );
			return( -1 );
		}
	}
	else {
		/* No pyramid ... write straight to name.
		 */
		if( !(tw->tif = tiff_openout( tw->name )) ) {
			free_tiff_write( tw );
			return( -1 );
		}
	}

	/* Write the TIFF header for the full-res file.
	 */
	write_tiff_header( tw, tw->tif, im->Xsize, im->Ysize );

	if( tw->tile ) 
		res = write_tif_tile( tw );
	else
		res = write_tif_strip( tw );
	if( res ) {
		free_tiff_write( tw );
		return( -1 );
	}

	/* Free pyramid resources ... this will TIFFClose() the intermediates,
	 * ready for us to read from them again.
	 */
	if( tw->layer )
		free_pyramid( tw->layer );
	if( tw->tif ) {
		TIFFClose( tw->tif );
		tw->tif = NULL;
	}

	/* Gather layers together into final pyramid file.
	 */
	if( tw->pyramid && gather_pyramid( tw ) ) {
		free_tiff_write( tw );
		return( -1 );
	}

	free_tiff_write( tw );

	return( 0 );
}

#endif /*HAVE_TIFF*/
