/* @(#) Opposite of im_extract: embed an image within a larger image. flag
 * @(#) controls what appears in the new pels:
 * @(#)
 * @(#)	0 - black pels (only flag supported in this version)
 * @(#)	  - mirror pels from image
 * @(#)	  - tile pels from image
 * @(#)	  - extend pels from image to edge
 * @(#)	  - etc.
 * @(#)
 * @(#) int im_embed( in, out, flag, x, y, w, h )
 * @(#) IMAGE *in, *out;
 * @(#) int flag;
 * @(#) int x, y, w, h;
 * @(#)
 * @(#) All functions return 0 on success and -1 on error
 * @(#)
 *
 * Author: J. Cupitt
 * Written on: 21/2/95
 * Modified on: 
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Save arguments here.
 */
typedef struct {
	int x, y, w, h;
	int flag;
	IMAGE *in;

	/* Geometry calculations. 
	 */
	Rect rout;		/* Whole output area */
	Rect rsub;		/* Rect occupied by image */
	Rect rtop;		/* Rect of new pixels above image */
	Rect rleft;		/* etc. */
	Rect rright;
	Rect rbottom;
} EmbedInfo;

/* Paint an area of a region.
 */
static void
paint_border( REGION *or, Rect *r )
{
	Rect ovl;

	/* Does any of r lie within or->valid? Paint it if it does!
	 */
	im_rect_intersectrect( r, &or->valid, &ovl );
	if( !im_rect_isempty( &ovl ) ) {
		PEL *q = (PEL *) IM_REGION_ADDR( or, ovl.left, ovl.top );
		int wd = ovl.width * IM_IMAGE_SIZEOF_PEL( or->im );
		int ls = IM_REGION_LSKIP( or );
		int y;

		for( y = 0; y < ovl.height; y++, q += ls )
			memset( (char *) q, 0, wd );
	}
}

/* Embed an area. 
 */
static int
embed_gen( REGION *or, REGION *ir, IMAGE *in, EmbedInfo *inf )
{
	Rect *r = &or->valid;
	Rect ovl;

	/* Does any of the input image appear in the area we have been asked
	 * to make?
	 */
	im_rect_intersectrect( r, &inf->rsub, &ovl );
	if( !im_rect_isempty( &ovl ) ) {
		/* Find the part of in we need.
		 */
		ovl.left -= inf->x;
		ovl.top -= inf->y;

		/* Paint this area of pixels into or.
		 */
		if( im_prepare_to( ir, or, &ovl, 
			ovl.left + inf->x, ovl.top + inf->y ) )
			return( -1 );
	}

	/* Paint all four border rects into or.
	 */
	paint_border( or, &inf->rtop );
	paint_border( or, &inf->rleft );
	paint_border( or, &inf->rright );
	paint_border( or, &inf->rbottom );

	return( 0 );
}

int
im_embed( IMAGE *in, IMAGE *out, int flag, int x, int y, int w, int h )
{	
	EmbedInfo *inf = IM_NEW( out, EmbedInfo );
	Rect want;

	/* Check args we were passed.  
	 */
	if( im_piocheck( in, out ) )
		return( -1 );
	if( in->Coding != IM_CODING_NONE && in->Coding != IM_CODING_LABQ ) {
		im_errormsg( "im_extract: unknown image coding type" );
		return( -1 );
	}

	/* Set up the output header.  
	 */
	if( im_cp_desc( out, in ) ) 
                return( -1 );
	out->Xsize = w;
	out->Ysize = h;

	/* Take a copy of args.
	 */
	inf->x = x;
	inf->y = y;
	inf->w = w;
	inf->h = h;
	inf->flag = flag;
	inf->in = in;

	/* Pre-calculate some geometry stuff.
	 */
	inf->rout.left = 0;	/* Whole output area */
	inf->rout.top = 0;
	inf->rout.width = out->Xsize;
	inf->rout.height = out->Ysize;

	want.left = x;		/* Rect occupied by image */
	want.top = y;
	want.width = in->Xsize;
	want.height = in->Ysize;
	im_rect_intersectrect( &want, &inf->rout, &inf->rsub );

	inf->rtop.left = 0;	/* Rect of new pixels above image */
	inf->rtop.top = 0;
	inf->rtop.width = out->Xsize;
	inf->rtop.height = inf->rsub.top;

	inf->rleft.left = 0;	/* Rect of new pixels to left */
	inf->rleft.top = inf->rsub.top;
	inf->rleft.width = inf->rsub.left;
	inf->rleft.height = in->Ysize;

	inf->rright.left = IM_RECT_RIGHT(&inf->rsub);/* Rect of new pixels to right */
	inf->rright.top = inf->rsub.top;
	inf->rright.width = out->Xsize - IM_RECT_RIGHT(&inf->rsub);
	inf->rright.height = in->Ysize;

	inf->rbottom.left = 0;	/* Rect of new pixels below image */
	inf->rbottom.top = IM_RECT_BOTTOM(&inf->rsub);
	inf->rbottom.width = out->Xsize;
	inf->rbottom.height = out->Ysize - IM_RECT_BOTTOM(&inf->rsub);

	/* Set demand hints.
	 */
	if( im_demand_hint( out, IM_THINSTRIP, in, NULL ) )
		return( -1 );

	/* Generate!
	 */
	if( im_generate( out, 
		im_start_one, embed_gen, im_stop_one,
		in, inf ) )
		return( -1 );

	return( 0 );
}
