/* @(#) Functions which calculates the correlation coefficient between two 
 * @(#) images. 
 * @(#) 
 * @(#) int im_spcor( IMAGE *in, IMAGE *ref, IMAGE *out )
 * @(#) 
 * @(#) We calculate:
 * @(#) 
 * @(#) 	 sumij (ref(i,j)-mean(ref))(inkl(i,j)-mean(inkl))
 * @(#) c(k,l) = ------------------------------------------------
 * @(#) 	 sqrt(sumij (ref(i,j)-mean(ref))^2) *
 * @(#) 		       sqrt(sumij (inkl(i,j)-mean(inkl))^2)
 * @(#) 
 * @(#) where inkl is the area of in centred at position (k,l).
 * @(#) 
 * @(#) Writes float to out. in and ref must be 1 band uchar, or 1 band
 * @(#) ushort.
 * @(#)
 * @(#) Returns 0 on sucess  and -1 on error.
 *
 * Copyright: 1990, N. Dessipris.
 *
 * Author: Nicos Dessipris
 * Written on: 02/05/1990
 * Modified on : 
 * 20/2/95 JC
 *	- updated
 *	- ANSIfied, a little
 * 21/2/95 JC
 *	- rewritten
 *	- partialed 
 *	- speed-ups
 *	- new correlation coefficient (see above), from Niblack "An
 *	  Introduction to Digital Image Processing,", Prentice/Hall, pp 138.
 * 4/9/97 JC
 *	- now does short/ushort as well
 * 13/2/03 JC
 *	- oops, could segv for short images
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <math.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Hold global stuff here.
 */
typedef struct {
	IMAGE *ref;		/* Image we are searching for */
	double rmean;		/* Mean of search window */
	double c1;		/* sqrt(sumij (ref(i,j)-mean(ref))^2) */
} SpcorInfo;

#define LOOP(IN) \
{ \
	IN *a = (IN *) p; \
	IN *b = (IN *) ref->data; \
	int in_lsk = lsk / sizeof( IN ); \
	IN *a1, *b1; \
 	\
	/* For each pel in or, loop over ref. First, \
	 * calculate mean of area in ir corresponding to ref. \
	 */ \
	for( a1 = a, sum1 = 0, j = 0; j < ref->Ysize; j++, a1 += in_lsk )  \
		for( i = 0; i < ref->Xsize; i++ ) \
			sum1 += a1[i]; \
	imean = (double) sum1 / (ref->Xsize * ref->Ysize); \
 	\
	/* Loop over ir again, this time calculating  \
	 * sum-of-squares-of-differences for this window on \
	 * ir, and also sum-of-products-of-differences. \
	 */ \
	for( a1 = a, b1 = b, sum2 = 0.0, sum3 = 0.0, j = 0; \
		j < ref->Ysize; j++, a1 += in_lsk, b1 += ref->Xsize ) { \
		for( i = 0; i < ref->Xsize; i++ ) { \
			/* Reference pel, and input pel. \
			 */ \
			IN rp = b1[i]; \
			IN ip = a1[i]; \
			\
			/* Accumulate sum-of-squares-of- \
			 * differences for input image. \
			 */ \
			double t = ip - imean; \
			sum2 += t * t; \
			\
			/* Accumulate product-of-differences. \
			 */ \
			sum3 += (rp - inf->rmean) * (ip - imean); \
		} \
	} \
}

/* spcor generate function.
 */
static int
spcor_gen( REGION *or, REGION *ir, IMAGE *in, SpcorInfo *inf )
{
	IMAGE *ref = inf->ref;
	Rect irect;
	Rect *r = &or->valid;
	int le = r->left;
	int to = r->top;
	int bo = IM_RECT_BOTTOM(r);
	int ri = IM_RECT_RIGHT(r);

	int x, y, i, j;
	int lsk;

	double imean;
	double sum1;
	double sum2, sum3;
	double c2, cc;

	/* What part of ir do we need?
	 */
	irect.left = or->valid.left;
	irect.top = or->valid.top;
	irect.width = or->valid.width + ref->Xsize - 1;
	irect.height = or->valid.height + ref->Ysize - 1;

	if( im_prepare( ir, &irect ) )
		return( -1 );
	lsk = IM_REGION_LSKIP( ir );

	/* Loop over or.
	 */
	for( y = to; y < bo; y++ ) {
		float *q = (float *) IM_REGION_ADDR( or, le, y );

		for( x = le; x < ri; x++ ) {
			PEL *p = (PEL *) IM_REGION_ADDR( ir, x, y );

			/* Find sums for this position.
			 */
			switch( ref->BandFmt ) {
			case IM_BANDFMT_UCHAR:	LOOP(unsigned char); break;
			case IM_BANDFMT_USHORT: LOOP(unsigned short); break;
			case IM_BANDFMT_SHORT:	LOOP(signed short); break;
			default:
				error_exit( "im_spcor: internal error #7934" );

				/* Keep gcc -Wall happy.
				 */
				return( -1 );
			}

			/* Now: calculate correlation coefficient!
			 */
			c2 = sqrt( sum2 );
			cc = sum3 / (inf->c1 * c2);

			*q++ = cc;
		}
	}

	return( 0 );
}

/* Pre-calculate stuff for our reference image.
 */
static SpcorInfo *
make_inf( IMAGE *out, IMAGE *ref )
{
	SpcorInfo *inf = IM_NEW( out, SpcorInfo );
	int sz = ref->Xsize * ref->Ysize;
	PEL *p = (PEL *) ref->data;
	double s;
	int i;

	if( !inf )
		return( NULL );

	/* Pre-calculate stuff on our reference image.
	 */
	inf->ref = ref;
	if( im_avg( inf->ref, &inf->rmean ) )
		return( NULL );
	
	/* Find sqrt-of-sum-of-squares-of-differences.
	 */
	for( s = 0.0, i = 0; i < sz; i++ ) {
		double t = (int) p[i] - inf->rmean;
		s += t * t;
	}
	inf->c1 = sqrt( s );

	return( inf );
}

int 
im_spcor_raw( IMAGE *in, IMAGE *ref, IMAGE *out )
{
	SpcorInfo *inf;

	/* PIO between in and out; WIO from ref.
	 */
	if( im_piocheck( in, out ) || im_incheck( ref ) )
		return( -1 );

	/* Check sizes.
	 */
	if( in->Xsize < ref->Xsize || in->Ysize < ref->Ysize ) {
		im_errormsg( "im_spcor_raw: ref not smaller than in" );
		return( -1 );
	}

	/* Check types.
	 */
	if( in->Coding != IM_CODING_NONE || in->Bands != 1 ||
		ref->Coding != IM_CODING_NONE || ref->Bands != 1 ||
		in->BandFmt != ref->BandFmt ) {
		im_errormsg( "im_spcor_raw: input not uncoded 1 band" );
		return( -1 );
	}
	if( in->BandFmt != IM_BANDFMT_UCHAR && 
		in->BandFmt != IM_BANDFMT_SHORT &&
		in->BandFmt != IM_BANDFMT_USHORT ) {
		im_errormsg( "im_spcor_raw: input not char/short/ushort" );
		return( -1 );
	}

	/* Prepare the output image. 
	 */
	if( im_cp_desc( out, in ) )
		return( -1 );
	out->Bbits = IM_BBITS_FLOAT;
	out->BandFmt = IM_BANDFMT_FLOAT;
	out->Xsize = in->Xsize - ref->Xsize + 1;
	out->Ysize = in->Ysize - ref->Ysize + 1;

	/* Pre-calculate some stuff.
	 */
	if( !(inf = make_inf( out, ref )) )
		return( -1 );

	/* Set demand hints. FATSTRIP is good for us, as THINSTRIP will cause
	 * too many recalculations on overlaps.
	 */
	if( im_demand_hint( out, IM_FATSTRIP, in, NULL ) )
		return( -1 );

	/* Write the correlation.
	 */
	if( im_generate( out,
		im_start_one, spcor_gen, im_stop_one, in, inf ) )
		return( -1 );

	return( 0 );
}

/* The above, with a border to make out the same size as in.
 */
int 
im_spcor( IMAGE *in, IMAGE *ref, IMAGE *out )
{
	IMAGE *t1 = im_open_local( out, "im_spcor intermediate", "p" );

	if( !t1 )
		return( -1 );
	if( im_spcor_raw( in, ref, t1 ) )
		return( -1 );
	if( im_embed( t1, out, 0, 
		ref->Xsize/2, ref->Ysize/2, in->Xsize, in->Ysize ) )
		return( -1 );
	
	return( 0 );
}
