/* @(#) Function to read a binary file with no header to vasari file
 * @(#) Usage:
 * @(#)
 * @(#) IMAGE *
 * @(#) im_binfile(in, xs, ys, bands)
 * @(#) char *in;
 * @(#) int xs, ys, bands;
 * @(#)
 * @(#) The function returns NULL on error.
 * @(#) Works for uchar input only.
 * Author: N. Dessipris
 * Written on: 31/7/91
 * Modified on:
 * 15/6/93 JC
 *	- includes fixed
 *	- externs fixed
 * 7/6/94 JC
 *	- im_outcheck() added
 * 13/12/94 JC
 *	- now uses im_mapfile(), rather than read() and copy
 * 	- hence returns a new IMAGE descriptor
 * 25/5/95 JC
 *	- oops! mess up with fd on errors
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

IMAGE *
im_binfile( const char *name, int xs, int ys, int bands )
{
	IMAGE *im;
	struct stat st;
	char buf[ 1024 ];
	int psize;

	/* Check parameters.
	 */
	if( xs <= 0 || ys <= 0 || bands <=0 ) {
		im_errormsg( "im_binfile: bad parameters" );
		return( NULL );
	}

	/* Make new output image for us.
	 */
	if( !(im = im_init( name )) )
		return( NULL );
	if( (im->fd = im__open_image_file( name )) == -1 ) {
		im_close( im );
		return( NULL );
	}

	/* Predict file size.
	 */
	psize = xs * ys * bands;

	/* Read the file length and check against what we think 
	 * the size should be.
	 */
	if( fstat( im->fd, &st ) == -1 ) {
		im_errormsg( "im_binfile: unable to get file stats" );
		im_close( im );
		return( NULL );
	}

	/* Very common, so special message.
	 */
	if( psize > st.st_size ) {
		im_errormsg( "im_binfile: unable to open %s: "
			"file has been truncated", im->filename );
		im_close( im );
		return( NULL );
	}

	/* Just wierd. Only print a warning for this, since we should
	 * still be able to process it without coredumps.
	 */
	if( psize < st.st_size )
		im_warning( "im_binfile: %s is longer than expected",
			im->filename );

	/* Try to map the new file.
	 */
	if( !(im->baseaddr = im_mapfile( im->fd )) ) {
		im_close( im );
		return( NULL );
	}

	/* Set header fields.
	 */
	im->Xsize = xs;
	im->Ysize = ys;
	im->Bands = bands;

	/* Set others to standard values.
	 */
	im->Bbits = IM_BBITS_BYTE;
	im->BandFmt = IM_BANDFMT_UCHAR;
	im->Coding = IM_CODING_NONE;

	if( bands == 1 )
		im->Type = IM_TYPE_B_W;
	else if( bands == 3 )
		im->Type = IM_TYPE_RGB;
	else 
		im->Type = IM_TYPE_MULTIBAND;

	im->Xres = 1.0;
	im->Yres = 1.0;

	im->Length = 0;
	im->Compression = 0;
	im->Level = 0;

	im->Xoffset = 0;
	im->Yoffset = 0;

	/* Init others too.
	 */
	im_snprintf( buf, 1024, 
		"Image imported with im_binfile( %s, %d, %d, %d )\n",
		name, xs, ys, bands );
	im->Hist = im_strdup( NULL, buf );
	im->data = im->baseaddr;
	im->dtype = IM_MMAPIN;
	im->dhint = IM_THINSTRIP;

	return( im );
}
