/* @(#) Frees all resources associated with IMAGE, and frees the memory
 * @(#) occupied by IMAGE.
 * @(#)
 * @(#) int 
 * @(#) im_close( image )
 * @(#) IMAGE *image;
 * @(#)
 * @(#) As above, but just free resources attached to IMAGE. After call, IMAGE
 * @(#) is as if im_init() had just been called. Useful for closing and
 * @(#) re-opening as another image type. See im_piocheck() etc.
 * @(#)
 * @(#) int 
 * @(#) im__close( image )
 * @(#) IMAGE *image;
 * @(#)
 * @(#) Returns 0 on success and 1 on error.
 *
 * Copyright: Nicos Dessipris
 * Written on: 12/04/1990
 * Modified on :  
 * 24/7/92 JC
 *	- im_update_descfile code tidied up
 *     	- free on NULL string when junking Hist fixed
 *     	- now calls im_unmapfile
 *     	- better behaviour if image has been opened and closed with 
 *	  no im_setupout call
 *      - better behaviour for half-made IMAGE descriptors
 * 15/4/93 JC
 *      - additions for freeing partial images
 * 29/4/93 JC
 *      - close callback list added
 * 10/5/93 JC
 *      - im__close() added
 * 9/11/93 JC
 *	- im_update_descfile -> write_descfile
 *	- if Hist is NULL, no longer makes up and writes .desc file
 * 16/8/94 JC
 *	- evalend callbacks added
 *	- ANSIfied
 * 24/10/95 JC
 *	- now tracks open images ... see also im_init() and debug.c
 * 11/7/00 JC
 *	- SETBUF_FOREIGN added
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /*HAVE_UNISTD_H*/
#ifdef HAVE_IO_H
#include <io.h>
#endif /*HAVE_IO_H*/

#include <vips/vips.h>
#include <vips/list.h>
#include <vips/region.h>
#include <vips/debug.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Maximum file name length.
 */
#define NAMELEN 1024

/* If the image name ends in ".v", then make a file called filename."desc"
 * and write im->Hist to it.
 */
static int 
write_descfile( IMAGE *im )
{	
	FILE *fp;
	int len;
	char buf[NAMELEN];

	if( !im->filename || !im->Hist )
                return( 0 );

	/* No history if the output filename does not end in .v.
	 */
	len = strlen( im->filename );
	if( len < 3 || strcmp( im->filename + len - 2, ".v" ) != 0 )
		return( 0 );

	if( len > NAMELEN - 10 ) {
		im_errormsg( "im_close: file name too long" );
		return( -1 );
	}

	strcpy( buf, im->filename );
	strcpy( buf + len - 2, ".desc" );

	if( !(fp = fopen( buf, "w" ))  ) {
		im_errormsg( "im_close: unable to write to %s", buf );
		return( -1 );
	}
	if( fputs( im->Hist, fp ) == EOF ) {
		im_errormsg( "im_close: unable to write history for %s",
			im->filename );
		fclose( fp );
		return( -1 );
	}
	fclose( fp );

	return( 0 );
}

/* Free any resources owned by this descriptor. The descriptor is left as if a
 * call to im_init had just happened - ie. the filename is set, but no other
 * resources are attached. Information is lost if this is a im_setbuf()
 * image! On an error, return non-zero and leave the image in an indeterminate
 * state. Too hard to recover gracefully.
 */
int 
im__close( IMAGE *im )
{	
	/* No action for NULL image.
	 */
	if( !im )
		return( 0 );

#ifdef DEBUG_IO
	printf( "im__close: starting for %s ..\n", im->filename );
#endif /*DEBUG_IO*/

	/* Free any regions defined on this image. This will, in turn, call
	 * all stop functions still running, freeing all regions we have on 
	 * other images, etc.
	 */
#ifdef DEBUG_IO
	printf( "im__close: freeing %d regions ..\n", 
		im_list_len( (List *) im->regions ) );
#endif /*DEBUG_IO*/
	im_list_fix( (List **) &im->regions, 
		(im_list_map_fn) im_region_free, NULL, NULL );

	/* Make sure all evalend functions have been called, perform all close
	 * callbacks, and free eval callbacks.
	 */
	if( im__trigger_callbacks( (List *) im->evalendfns ) )
		return( -1 );
	im_list_free( (List **) &im->evalendfns, 
		(im_list_free_fn) im_free, NULL, NULL );

	if( im__trigger_callbacks( (List *) im->closefns ) )
		return( -1 );
	im_list_free( (List **) &im->closefns, 
		(im_list_free_fn) im_free, NULL, NULL );

	im_list_free( (List **) &im->evalfns, 
		(im_list_free_fn) im_free, NULL, NULL );

	/* Junk generate functions. 
	 */
	im->start = NULL;
	im->generate = NULL;
	im->stop = NULL;

	/* What resources are associated with this IMAGE descriptor?
	 */
	if( im->baseaddr ) {
		/* MMAP file.
		 */
#ifdef DEBUG_IO
		printf( "im__close: unmapping file ..\n" );
#endif /*DEBUG_IO*/

		if( im_unmapfile( im->fd, im->baseaddr ) )
			return( -1 );
		im->baseaddr = NULL; 
		im->data = NULL;
	}

	/* Is there a file descriptor?
	 */
	if( im->fd != -1 ) {
#ifdef DEBUG_IO
		printf( "im__close: closing output file ..\n" );
#endif /*DEBUG_IO*/

		if( im->dtype == IM_OPENOUT && write_descfile( im ) ) {
			im_errormsg( "im_close: unable to write history "
				"for %s", im->filename );
			return( -1 );
		}
		if( close( im->fd ) == -1 ) {
			im_errormsg( "im_close: unable to close fd (2) "
				"for %s", im->filename );
			return( -1 );
		}
		im->fd = -1;
	}

	/* Any image data?
	 */
	if( im->data ) {
		/* Buffer image. Only free stuff we know we allocated.
		 */
		if( im->dtype == IM_SETBUF ) {
#ifdef DEBUG_IO
			printf( "im__close: freeing buffer ..\n" );
#endif /*DEBUG_IO*/
			im_free( im->data );
		}

		im->data = NULL;
	}

	/* Free common resources.
	 */
	if( im->Hist ) {
		im_free( im->Hist );
		im->Hist = NULL;
	}

	/* Reset the lock.
	 */
	im_lock_destroy( &im->sslock );
	im_lock_init( &im->sslock );

	/* Reset other state.
	 */
	im->dtype = IM_NONE;
	im->dhint = IM_SMALLTILE;
	im->kill = 0;
	im->close_pending = 0;

#ifdef DEBUG_IO
	printf( "im__close: success for %s\n", im->filename );
#endif /*DEBUG_IO*/

	return( 0 );
}

/* Free resources and close descriptor.
 */
int 
im_close( IMAGE *im )
{	
	/* No action for NULL image.
	 */
	if( !im )
		return( 0 );

	/* Are there any regions left on this image? If there are, just set
	 * close_pending and return. The image will be then be closed when
	 * the last region is freed (see im_region_free()). This prevents 
	 * some dangling region pointers.
	 */
#ifdef DEBUG_IO
	if( im->close_pending )
		/* Strange! Just print a warning.
		 */
		printf( "im_close: im_close called twice on \"%s\"\n",
			im->filename );
#endif /*DEBUG_IO*/
	if( im->regions ) {
#ifdef DEBUG_IO
		printf( "im_close: pending close for \"%s\"\n", im->filename );
#endif /*DEBUG_IO*/
		im->close_pending = 1;
		return( 0 );
	}

	/* Is this descriptor currently being closed somewhere else? Immediate
	 * return if it is. This prevents infinite descent if a close callback
	 * includes an im_close for this image. This can happen sometimes!
	 */
	if( im->closing )
		return( 0 );
	im->closing = 1;

	/* Free IMAGE resources.
	 */
	if( im__close( im ) ) {
		im->closing = 0;
		return( -1 );
	}

	/* Shut lock down completely.
	 */
	im_lock_destroy( &im->sslock );

	/* Free filename and IMAGE struct data.
	 */
	if( im->filename ) {
		im_free( im->filename );
		im->filename = NULL;
	}
	im_free( im );

	/* Remove from list of open images.
	 */
	if( im_list_remove( &im__open_images, im ) )
		return( -1 );

	return( 0 );
}
