/* @(#) Function which returns a char pointer at the beginning of the file 
 * @(#) corresponding to fd.
 * @(#) 
 * @(#) char *
 * @(#) im_mapfile( fd )
 * @(#) int fd;
 * @(#)
 * @(#) As above, but map read-write.
 * @(#) 
 * @(#) char *
 * @(#) im_mapfilerw( fd )
 * @(#) int fd;
 * @(#)
 * @(#) Return NULL on error
 * 
 * Copyright: Nicos Dessipris
 * Wriiten on: 13/02/1990
 * Updated on:
 * 10/5/93 J.Cupitt
 *	- im_mapfilerw() added
 * 13/12/94 JC
 *	- ANSIfied
 * 5/7/99 JC
 *	- better error if unable to map rw
 * 31/3/02 JC
 *	- better mmap() fails error
 * 19/9/02 JC
 * 	- added im__mmap()/im__munmap() with windows versions
 * 5/1/04 Lev Serebryakov
 * 	- patched for freebsd compatibility
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* 
#define DEBUG
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <assert.h>

#include <sys/types.h>
#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif /*HAVE_SYS_MMAN_H*/
#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif /*HAVE_SYS_FILE_H*/
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /*HAVE_UNISTD_H*/
#ifdef HAVE_WINDOWS_H 
#ifndef S_ISREG
#define S_ISREG(m) (!!(m & _S_IFREG))
#endif
#endif /*HAVE_WINDOWS_H*/

#include <vips/vips.h>

#ifdef HAVE_WINDOWS_H
#include <windows.h>
#endif /*HAVE_WINDOWS_H*/

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

void *
im__mmap( int fd, int writeable, size_t length, off_t offset )
{
	void *baseaddr;

#ifdef DEBUG
	printf( "im__mmap: length = %d, offset = %lld\n", length, offset );
#endif /*DEBUG*/

#ifdef HAVE_WINDOWS_H
{
	HANDLE hFile = (HANDLE) _get_osfhandle( fd );
        HANDLE hMMFile;
	DWORD flProtect;
	DWORD dwDesiredAccess;
	DWORD dwFileOffsetHigh;
	DWORD dwFileOffsetLow;

	if( writeable ) {
		flProtect = PAGE_READWRITE;
		dwDesiredAccess = FILE_MAP_WRITE;
	}
	else {
		flProtect = PAGE_READONLY;
		dwDesiredAccess = FILE_MAP_READ;
	}

        if( !(hMMFile = CreateFileMapping( hFile,
		NULL, flProtect, 0, 0, NULL )) ) {
                im_errormsg_system( GetLastError(),
			"im__mmap: unable to CreateFileMapping" );
                return( NULL );
        }

	/*
		
		FIXME ... should do something to get the high and low dwords
		right, but off_t seems to always be 32 bits on win32, so don't
		bother

	int dws = sizeof( DWORD );
	int shift = 8 * dws;
	off_t mask = ((off_t) -1) >> shift;

	dwFileOffsetHigh = (offset >> shift) & mask;
	dwFileOffsetLow = offset & mask;
	 */

	dwFileOffsetHigh = 0;
	dwFileOffsetLow = offset;

        if( !(baseaddr = (char *)MapViewOfFile( hMMFile, dwDesiredAccess, 
		dwFileOffsetHigh, dwFileOffsetLow, length )) ) {
                im_errormsg_system( GetLastError(),
			"im__mmap: unable to MapViewOfFile" );
		CloseHandle( hMMFile );
                return( NULL );
        }

	/* Can close mapping now ... view stays until UnmapViewOfFile().

		FIXME ... is this a performance problem?

	 */
	CloseHandle( hMMFile );
}
#else /*HAVE_WINDOWS_H*/
{
	int prot;

	if( writeable ) 
		prot = PROT_WRITE;
	else 
		prot = PROT_READ;

	baseaddr = mmap( 0, length, prot, MAP_SHARED, fd, offset );
	if( baseaddr == MAP_FAILED ) { 
		im_errormsg_system( errno, "im__mmap: unable to mmap" );
		im_warning( "im__mmap: map failed, "
			"running very low on system resources, "
			"expect a crash soon" );
		return( NULL ); 
	}
}
#endif /*HAVE_WINDOWS_H*/

	return( baseaddr );
}

int
im__munmap( void *start, size_t length )
{
#ifdef HAVE_WINDOWS_H
	if( !UnmapViewOfFile( start ) ) {
		im_errormsg_system( GetLastError(),
			"im__munmap: unable to UnmapViewOfFile" );
		return( -1 );
	}
#else /*HAVE_WINDOWS_H*/
	if( munmap( start, length ) < 0 ) {
		im_errormsg_system( errno,
			"im__munmap: unable to munmap file" );
		return( -1 );
	}
#endif /*HAVE_WINDOWS_H*/

	return( 0 );
}

PEL *
im_mapfile( int fd )
{	
	void *baseaddr;
	struct stat st;
	mode_t m;

	/* Check the size of the file; if it is less than 64 bytes, then flag
	 * an error.
	 */
	if( fstat( fd, &st ) == -1 ) {
		im_errormsg( "im_mapfile: unable to get file status" );
		return( NULL );
	}
	m = (mode_t) st.st_mode;
	if( st.st_size < 64 ) {
		im_errormsg( "im_mapfile: file is less than 64 bytes" );
		return( NULL ); 
	}
	if( !S_ISREG(m) ) {
		im_errormsg( "im_mapfile: not a regular file" ); 
		return( NULL ); 
	}

	if( !(baseaddr = im__mmap( fd, 0, st.st_size, 0 )) )
		return( NULL );

	return( baseaddr );
}

/* As above, but map read/write.
 */
PEL *
im_mapfilerw( int fd )
{	
	void *baseaddr;
	struct stat st;
	mode_t m;

	/* Check the size of the file if it is less than 64 bytes return
	 * make also sure that it is a regular file
	 */
	if( fstat( fd, &st ) == -1 ) {
		im_errormsg( "im_mapfilerw: unable to get file status" );
		return( NULL );
	}
	m = (mode_t) st.st_mode;
	if( st.st_size < 64 || !S_ISREG(m) ) {
		im_errormsg( "im_mapfile: unable to read data" ); 
		return( NULL ); 
	}

	if( !(baseaddr = im__mmap( fd, 1, st.st_size, 0 )) )
		return( NULL );

	return( baseaddr );
}

/* From im_rwcheck() ... image needs to be a completely mapped read-only file, 
 * we try to remap it read-write. 
 */
int
im_remapfilerw( IMAGE *image )
{
	void *baseaddr;

#ifdef HAVE_WINDOWS_H
{
	HANDLE hFile = (HANDLE) _get_osfhandle( image->fd );
        HANDLE hMMFile;

        if( !(hMMFile = CreateFileMapping( hFile,
		NULL, PAGE_READWRITE, 0, 0, NULL )) ) {
                im_errormsg_system( GetLastError(),
			"im_remapfilerw: unable to CreateFileMapping" );
                return( -1 );
        }

	if( !UnmapViewOfFile( image->baseaddr ) ) {
		im_errormsg_system( GetLastError(),
			"im_remapfilerw: unable to UnmapViewOfFile" );
		return( -1 );
	}
        if( !(baseaddr = (char *)MapViewOfFileEx( hMMFile, FILE_MAP_WRITE, 
		0, 0, 0, image->baseaddr )) ) {
                im_errormsg_system( GetLastError(),
			"im_remapfilerw: unable to MapViewOfFile" );
		CloseHandle( hMMFile );
                return( -1 );
        }

	/* Can close mapping now ... view stays until UnmapViewOfFile().

		FIXME ... is this a performance problem?

	 */
	CloseHandle( hMMFile );
}
#else /*!HAVE_WINDOWS_H*/
{
	struct stat st;
	mode_t m;

	assert( image->dtype == IM_MMAPIN );

	if( fstat( image->fd, &st ) == -1 ) {
		im_errormsg( "im_mapfilerw: unable to get file status" );
		return( -1 );
	}
	m = (mode_t)st.st_mode;
	if( st.st_size < 64 || !S_ISREG(m) ) {
		im_errormsg( "im_mapfile: unable to read data" ); 
		return( -1 ); 
	}

	baseaddr = mmap( image->baseaddr, st.st_size, 
		PROT_READ | PROT_WRITE, MAP_SHARED | MAP_FIXED, 
		image->fd, 0 );
	if( baseaddr == (void *)-1 ) { 
		im_errormsg( "im_remapfilerw: unable to mmap: \"%s\" - %s",
				image->filename, strerror( errno ) );
		return( -1 ); 
	}
}
#endif /*HAVE_WINDOWS_H*/

	image->dtype = IM_MMAPINRW;

	if( baseaddr != image->baseaddr ) {
		im_errormsg( "im_remapfilerw: unable to mmap \"%s\" to same "
			"address", image->filename );
		image->baseaddr = baseaddr;
		return( -1 ); 
	}

	return( 0 );
}

